import React from 'react';
import ReactDOM from 'react-dom';
import noop from 'lodash/noop';
import FocusLock, {AutoFocusInside} from 'react-focus-lock';
import {block} from 'bem-cn';

import CloseIcon from 'components/basic/icons/CloseIcon';

import HideBodyVerticalScroll from './internal/HideBodyVerticalScroll';

import './Modal.scss';

const b = block('Modal');

export interface IModalProps {
    className?: string;

    /**
     * Контент модалки.
     */
    children?: React.ReactNode;

    /**
     * Компонент, который будет использоваться в качестве обертки для FocusLock.
     */
    container?: React.ReactType;

    /**
     * Пробрасывает класс для компонента `container`.
     */
    containerClassName?: string;

    /**
     * Параметры, которые будут передпны в компонент `container`.
     */
    containerProps?: {
        [key: string]: any;
    };

    /**
     * По умолчанию модалка ставит фокус на первый фокусируемый элемент.
     * `disableAutoFocus` отключает это поведение
     */
    disableAutoFocus?: boolean;

    /**
     * Отключает анимацию в модалке
     */
    disableAnimations?: boolean;

    /**
     * Запрещает скролл страницы под модальным окном.
     * Проблема встречается на iOS.
     */
    preventBodyScroll?: boolean;

    /**
     * Раскрывает модалку на весь экран
     */
    fullScreen?: boolean;

    /**
     * Включает мобильную вёрстку
     */
    isMobile?: boolean;

    /**
     * Позиция модалки по вертикали
     * @default 'middle'
     */
    verticalAlign?: 'top' | 'middle' | 'bottom';

    /**
     * Управляет видимостью модалки
     */
    isVisible: boolean;

    /**
     * Управляет отображением крестика в модалке
     *
     * @default true
     */
    hasCloseButton?: boolean;

    /**
     * Управляет закрытие модалки при клике на фон и на Escape
     *
     * @default true
     */
    autoclosable?: boolean;

    /**
     * Возвращать фокус в позицию до открытия модалки
     *
     * @default true
     */
    returnFocus?: boolean;

    /**
     * Вызывается при клике на крестик, фон и нажатие Escape.
     */
    onClose?(): void;
}

export interface IModalState {
    isInitialized: boolean;
    bodyHideScroll: boolean;
    shake: boolean | null;
    contentContainer: HTMLElement | undefined;
}

export const MODAL_ANIMATION_DURATION_MS = 200;

const MODAL_SHAKE_DURATION = 820;

class Modal extends React.Component<IModalProps, IModalState> {
    /**
     * Компонент предназначен для установки начального положения
     * фокуса в модалке
     */
    static AutoFocus = AutoFocusInside;

    protected static defaultProps = {
        verticalAlign: 'middle',
        isVisible: false,
        hasCloseButton: true,
        preventBodyScroll: true,
        returnFocus: true,
        autoclosable: true,
        onClose: noop,
    };

    private isOuterClickStarted = false;

    private readonly rootRef = React.createRef<HTMLDivElement>();

    constructor(props: IModalProps) {
        super(props);

        this.state = {
            isInitialized: props.isVisible,
            bodyHideScroll: Boolean(props.isVisible && props.preventBodyScroll),
            shake: null,
            contentContainer: undefined,
        };
    }

    componentDidMount(): void {
        document.addEventListener('keydown', this.onKeyDown);

        this.setState({});
    }

    componentWillUnmount(): void {
        document.removeEventListener('keydown', this.onKeyDown);
    }

    componentDidUpdate(prevProps: IModalProps): void {
        this.initializeOnFirstShow();
        this.updateScrollViewOnShow(prevProps);
        this.updateOnPreventBodyScrollChange(prevProps);
        this.updateScrollViewOnHide(prevProps);
    }

    private initializeOnFirstShow() {
        const {isVisible} = this.props;

        if (isVisible && !this.state.isInitialized) {
            this.setState({isInitialized: true});
        }
    }

    private updateOnPreventBodyScrollChange(prevProps: IModalProps) {
        const {preventBodyScroll} = this.props;

        if (prevProps.preventBodyScroll !== preventBodyScroll) {
            this.updateScrollView();
        }
    }

    /**
     * Обновление блокировки скрола при показе модального окна
     *
     * @param prevProps
     */
    private updateScrollViewOnShow(prevProps: IModalProps) {
        const {isVisible} = this.props;

        if (isVisible && !prevProps.isVisible) {
            this.setState({shake: null});
            this.updateScrollView();
        }
    }

    /**
     * Обновление блокировки скрола при скрывани модального окна
     *
     * @param prevProps
     */
    private updateScrollViewOnHide(prevProps: IModalProps) {
        const {isVisible, disableAnimations} = this.props;

        if (prevProps.isVisible && !isVisible) {
            if (disableAnimations) {
                this.updateScrollView();
            } else {
                setTimeout(
                    () => this.updateScrollView(),
                    MODAL_ANIMATION_DURATION_MS,
                );
            }
        }
    }

    private updateScrollView(): void {
        const {isVisible, preventBodyScroll} = this.props;

        this.setState({
            bodyHideScroll: Boolean(isVisible && preventBodyScroll),
        });
    }

    private readonly handleCloseButtonClick = () => {
        this.close();
    };

    private readonly handleMouseDown = (evt: React.SyntheticEvent) => {
        const {isVisible} = this.props;

        if (
            isVisible &&
            !this.state.contentContainer?.contains(evt.target as Node)
        ) {
            this.isOuterClickStarted = true;
        }

        evt.stopPropagation();
    };

    private readonly handleMouseUp = (evt: React.SyntheticEvent) => {
        const {isVisible} = this.props;

        if (
            isVisible &&
            this.isOuterClickStarted &&
            !this.state.contentContainer?.contains(evt.target as Node)
        ) {
            this.autoClose();
        }

        this.isOuterClickStarted = false;

        evt.stopPropagation();
    };

    private autoClose() {
        const {autoclosable} = this.props;
        const {shake} = this.state;

        if (autoclosable) {
            this.close();
        } else if (!shake) {
            this.shake();
        }
    }

    private shake() {
        const {disableAnimations} = this.props;

        if (!disableAnimations) {
            this.setState({shake: true});

            setTimeout(() => {
                this.setState({shake: false});
            }, MODAL_SHAKE_DURATION);
        }
    }

    private readonly onKeyDown = (evt: KeyboardEvent) => {
        const {isVisible} = this.props;

        if (isVisible && evt.key === 'Escape') {
            evt.preventDefault();
            evt.stopPropagation();
            evt.stopImmediatePropagation();
            this.autoClose();
        }
    };

    private close() {
        const {onClose} = this.props;

        if (onClose) {
            onClose();
        }
    }

    private setContainerRef = (elem: HTMLElement) => {
        this.setState({contentContainer: elem});
    };

    protected getModalElement() {
        const {
            children,
            container,
            containerClassName = '',
            containerProps,
            className = '',
            disableAnimations,
            disableAutoFocus,
            fullScreen,
            hasCloseButton,
            isMobile,
            isVisible,
            returnFocus,
            verticalAlign,
        } = this.props;

        const {bodyHideScroll, shake} = this.state;

        return (
            <div
                className={b({
                    visible: isVisible,
                    noAnimation: disableAnimations,
                    fullscreen: fullScreen,
                    mobile: isMobile,
                }).mix(`Modal_${verticalAlign}`, className)}
                style={{zIndex: 1000}}
                ref={this.rootRef}
                onMouseDown={this.handleMouseDown}
                onMouseUp={this.handleMouseUp}
            >
                <div
                    className={b('dialog', {
                        startShake: isVisible && shake,
                        stopShake: shake !== null && isVisible && !shake,
                    })}
                >
                    <FocusLock
                        className={b('content').mix(containerClassName)}
                        as={container || (fullScreen ? undefined : 'div')}
                        lockProps={
                            containerProps ||
                            (fullScreen ? undefined : {shadow: 'popup'})
                        }
                        autoFocus={!disableAutoFocus}
                        returnFocus={returnFocus}
                        disabled={!isVisible || isMobile}
                        ref={this.setContainerRef}
                    >
                        {children}
                        {hasCloseButton && (
                            <button
                                className={b('close-button')}
                                onClick={this.handleCloseButtonClick}
                            >
                                <CloseIcon />
                            </button>
                        )}
                    </FocusLock>

                    {bodyHideScroll && <HideBodyVerticalScroll />}
                </div>
            </div>
        );
    }

    render() {
        const {isInitialized} = this.state;

        if (!isInitialized || typeof document === 'undefined') {
            return null;
        }

        return ReactDOM.createPortal(this.getModalElement(), document.body);
    }
}

export default Modal;
