import React, {useEffect, useRef} from 'react';

import getScrollWidth from 'lib/dom/getScrollWidth';
import scrollTo from 'lib/dom/scrollTo';

function attachStyle(
    element: HTMLElement,
    style: {css: string; className: string},
) {
    element.classList.add(style.className);

    const removeStyleNode = attachStylesheet(style.css);

    return () => {
        removeStyleNode();
        element.classList.remove(style.className);
    };
}

function generateDocumentStyle(
    documentMargin: number,
    previousScrollY: number,
) {
    const className = 'document-HideBodyVerticalScroll';
    const css = `\
.${className} {
  overflow: hidden !important;
  margin-right: ${documentMargin}px !important;
  height: 100%;
}

.${className} #app {
  position: fixed;
  top: -${previousScrollY}px;
  width: calc(100% - ${documentMargin}px);
}
`;

    return {className, css};
}

function attachStylesheet(sheet: string) {
    const style = document.createElement('style');

    style.setAttribute('type', 'text/css');

    // @ts-ignore IE specific api
    if (style.styleSheet) {
        // @ts-ignore IE specific api
        style.styleSheet.cssText = sheet;
    } else {
        style.appendChild(document.createTextNode(sheet));
    }

    const head = document.getElementsByTagName('head')[0];

    head.appendChild(style);

    return () => {
        if (head.contains(style)) {
            head.removeChild(style);
        }
    };
}

let modalsCounter = 0;
let stopHidingScroll: (() => void) | null = null;

const HideBodyVerticalScroll: React.FC = () => {
    const initialWindowYScroll = useRef<number>();

    useEffect(() => {
        initialWindowYScroll.current = window.pageYOffset;

        const {
            documentElement,
            documentElement: {clientHeight, scrollHeight},
        } = document;

        if (modalsCounter === 0 && clientHeight < scrollHeight) {
            const documentComputedStyle = getComputedStyle(documentElement);
            const scrollWidth = getScrollWidth();
            const documentMarginRight = parseFloat(
                documentComputedStyle.marginRight || '',
            );
            const documentStyle = generateDocumentStyle(
                documentMarginRight + scrollWidth,
                initialWindowYScroll.current,
            );

            stopHidingScroll = attachStyle(documentElement, documentStyle);
        }

        modalsCounter++;

        return () => {
            modalsCounter--;

            if (modalsCounter === 0) {
                if (stopHidingScroll) {
                    stopHidingScroll();
                }

                scrollTo({top: initialWindowYScroll.current, left: 0});
            }
        };
    }, []);

    return null;
};

export default HideBodyVerticalScroll;
