import React, {FC, useCallback} from 'react';
import {useHistory} from 'react-router-dom';
import {block} from 'bem-cn';

import './index.scss';

const MAX_PAGE_AMOUNT = 100000;

interface IPaginationProps {
    className?: string;
    itemsCount: number;
    itemsPerPage: number;
    currentPage: number;
    buttonsLimit: number;
    pageLinkBuilder: (page: number, count: number) => string;
}

function getVisibleButtonsNumbers(
    currentPage: number,
    buttonsLimit: number,
    overallPagesCount: number,
): number[] {
    const visibleButtonsNumbers = [currentPage];

    let pageAwayFromCurrent = 0;

    while (pageAwayFromCurrent < MAX_PAGE_AMOUNT) {
        pageAwayFromCurrent++;

        const pagesBeforeAdding = visibleButtonsNumbers.length;
        const prevPage = currentPage - pageAwayFromCurrent;

        if (prevPage > 0) {
            visibleButtonsNumbers.unshift(prevPage);
        }

        if (visibleButtonsNumbers.length === buttonsLimit) {
            break;
        }

        const nextPage = currentPage + pageAwayFromCurrent;

        if (nextPage <= overallPagesCount) {
            visibleButtonsNumbers.push(nextPage);
        }

        if (
            visibleButtonsNumbers.length === buttonsLimit ||
            visibleButtonsNumbers.length === pagesBeforeAdding
        ) {
            break;
        }
    }

    return visibleButtonsNumbers;
}

const b = block('Pagination');

const Pagination: FC<IPaginationProps> = props => {
    const {
        className,
        itemsCount,
        currentPage,
        buttonsLimit,
        itemsPerPage,
        pageLinkBuilder,
    } = props;

    const history = useHistory();

    const handlePageClick = useCallback(
        page => {
            history.push(pageLinkBuilder(page, itemsPerPage));
        },
        [pageLinkBuilder, history, itemsPerPage],
    );

    const pagesCount = Math.ceil(itemsCount / itemsPerPage);

    if (pagesCount < 2 || currentPage > pagesCount) {
        return null;
    }

    const arrowAreVisible = pagesCount > buttonsLimit;

    const visibleButtonsNumbers = getVisibleButtonsNumbers(
        currentPage,
        buttonsLimit,
        pagesCount,
    );

    return (
        <ul className={b.mix(className)}>
            {arrowAreVisible && (
                <>
                    <li
                        className={b('pageItem', {disabled: currentPage === 1})}
                        onClick={handlePageClick.bind(null, 1)}
                    >
                        Первая
                    </li>

                    <li
                        className={b('pageItem', {disabled: currentPage === 1})}
                        onClick={handlePageClick.bind(null, currentPage - 1)}
                    >
                        Предыдущая
                    </li>
                </>
            )}

            {visibleButtonsNumbers.map(index => (
                <li
                    className={b('pageItem', {active: index === currentPage})}
                    key={index}
                    onClick={handlePageClick.bind(null, index)}
                >
                    {index}
                </li>
            ))}

            {arrowAreVisible && (
                <>
                    <li
                        className={b('pageItem', {
                            disabled: currentPage === pagesCount,
                        })}
                        onClick={handlePageClick.bind(null, currentPage + 1)}
                    >
                        Следующая
                    </li>

                    <li
                        className={b('pageItem', {
                            disabled: currentPage === pagesCount,
                        })}
                        onClick={handlePageClick.bind(null, pagesCount)}
                    >
                        Последняя
                    </li>
                </>
            )}
        </ul>
    );
};

export default Pagination;
