import {IBriefHotelInfo, IHotelOrderItem} from 'redux/reducers/types/hotel';
import {IAviaOrderItem, IBriefAviaInfo} from 'redux/reducers/types/avia';
import {IBriefTrainsInfo, ITrainOrderItem} from 'redux/reducers/types/train';
import {IBriefBusInfo, IBusOrderItem} from 'redux/reducers/types/bus';
import {Environment} from 'shared/types';

export interface IPrice {
    value: number;
    currency: string;
}

export enum OrderNotificationType {
    SUCCESS = 'NT_SUCCESS',
    REFUND = 'NT_REFUND',
}

export enum OrderNotificationTransport {
    EMAIL = 'NTT_EMAIL',
    SMS = 'NTT_SMS',
}

const ORDER_NOTIFICATION_TYPES = Object.values(OrderNotificationType);

export function isOrderNotificationType(
    notificationType: any,
): notificationType is OrderNotificationType {
    return ORDER_NOTIFICATION_TYPES.includes(notificationType);
}

export enum DisplayOrderType {
    DT_UNKNOWN = 'DT_UNKNOWN',
    DT_HOTEL = 'DT_HOTEL',
    DT_AVIA = 'DT_AVIA',
    DT_TRAIN = 'DT_TRAIN',
    DT_BUS = 'DT_BUS',
    DT_SUBURBAN = 'DT_SUBURBAN',
}

const DISPLAY_ORDER_TYPES = Object.values(DisplayOrderType);

export function isDisplayOrderTypeValid(
    displayOrderType?: any,
): displayOrderType is DisplayOrderType {
    return DISPLAY_ORDER_TYPES.includes(displayOrderType);
}

export interface IStoreUserInfo {
    hasAccess: boolean;
}

export interface IStoreSettings {
    isTechTexts: boolean;
}

export enum OrderStatus {
    OS_UNKNOWN = 'OS_UNKNOWN',
    OS_IN_PROGRESS = 'OS_IN_PROGRESS',
    OS_REFUNDED = 'OS_REFUNDED',
    OS_AWAITS_PAYMENT = 'OS_AWAITS_PAYMENT',
    OS_CANCELLED = 'OS_CANCELLED',
    OS_FULFILLED = 'OS_FULFILLED',
}

export const ORDER_STATUSES = Object.values(OrderStatus);

export function isOrderStatusValid(
    orderStatus?: any,
): orderStatus is OrderStatus {
    return ORDER_STATUSES.includes(orderStatus);
}

export enum OrderPartner {
    PT_UNKNOWN = 'PT_UNKNOWN',
    PT_HOTEL_TRAVELLINE = 'PT_HOTEL_TRAVELLINE',
    PT_HOTEL_DOLPHIN = 'PT_HOTEL_DOLPHIN',
    PT_HOTEL_BNOVO = 'PT_HOTEL_BNOVO',
    PT_HOTEL_EXPEDIA = 'PT_HOTEL_EXPEDIA',
    PT_HOTEL_BRONEVIK = 'PT_HOTEL_BRONEVIK',
    PT_AVIA_AEROFLOT = 'PT_AVIA_AEROFLOT',
    PT_TRAIN_RZHD = 'PT_TRAIN_RZHD',
    PT_MOVISTA = 'PT_MOVISTA',
    PT_IM_SUBURBAN = 'PT_IM_SUBURBAN',
    PT_AEROEXPRESS = 'PT_AEROEXPRESS',
    PT_BUS_BUSFOR = 'PT_BUS_BUSFOR',
    PT_BUS_ECOLINES = 'PT_BUS_ECOLINES',
    PT_BUS_ETRAFFIC = 'PT_BUS_ETRAFFIC',
    PT_BUS_NOY = 'PT_BUS_NOY',
    PT_BUS_OK = 'PT_BUS_OK',
    PT_BUS_RUSET = 'PT_BUS_RUSET',
    PT_BUS_SKS = 'PT_BUS_SKS',
    PT_BUS_UNITIKI = 'PT_BUS_UNITIKI',
    PT_BUS_YUGAVTOTRANS = 'PT_BUS_YUGAVTOTRANS',
}

const ORDER_PARTNERS = Object.values(OrderPartner);

export function isOrderPartnerValid(
    orderPartner?: any,
): orderPartner is OrderPartner {
    return ORDER_PARTNERS.includes(orderPartner);
}

export enum OrderCarrier {
    OC_UNKNOWN = '',
    OC_CPPK = 'cppk',
    OC_SZPPK = 'szppk',
    OC_MTPPK = 'mtppk',
    OC_BASHPPK = 'bashppk',
    OC_SODRUZHESTVO = 'sodruzhestvo',
    OC_AEROEXPRESS = 'aeroexpress',
}

export const ORDER_CARRIERS = Object.values(OrderCarrier);

export function isOrderCarrierValid(
    orderCarrier?: any,
): orderCarrier is OrderCarrier {
    return ORDER_CARRIERS.includes(orderCarrier);
}

export enum OrderReferralPartnerId {
    ORP_UNKNOWN = '',
    ORP_EMPTY = 'WL_EMPTY',
    ORP_S7 = 'WL_S7',
    ORP_TINKOFF = 'WL_TINKOFF',
    ORP_VTB = 'WL_VTB',
    ORP_RAIFFEISEN = 'WL_RAIFFEISEN',
    ORP_TELE2 = 'WL_TELE2',
    ORP_OTKRITIE = 'WL_OTKRITIE',
    ORP_BEELINE = 'WL_BEELINE',
}

export const ORDER_REFERRAL_PARTNER_IDS = Object.values(OrderReferralPartnerId);

export function isOrderReferralPartnerIdValid(
    orderReferralPartnerId?: any,
): orderReferralPartnerId is OrderReferralPartnerId {
    return ORDER_REFERRAL_PARTNER_IDS.includes(orderReferralPartnerId);
}

export enum SortTypes {
    SD_ASC = 'SD_ASC',
    SD_DESC = 'SD_DESC',
}

export const SORT_TYPES: SortTypes[] = [SortTypes.SD_DESC, SortTypes.SD_ASC];

export function isSortTypeValid(sortType?: any): sortType is SortTypes {
    return SORT_TYPES.includes(sortType);
}

export interface IBriefOrderInfo {
    order_id: string;
    pretty_id: string;
    display_order_type: DisplayOrderType;
    display_order_state: OrderStatus;
    order_item_state: string;
    invoice_state: string | null;
    partners: OrderPartner[];
    broken: boolean;
    clients_names: string;
    price_info: IPrice | null;
    avia_info: IBriefAviaInfo[];
    hotel_info: IBriefHotelInfo[];
    train_info: IBriefTrainsInfo[];
    bus_info: IBriefBusInfo[];
    created_at: string;
    deferred: boolean;
}

export enum PaymentScheduleType {
    PST_UNKNOWN = 'PST_UNKNOWN',
    PST_FULL = 'PST_FULL',
    PST_DEFERRED = 'PST_DEFERRED',
}

export const PAYMENT_SCHEDULE_TYPES = Object.values(PaymentScheduleType);

export function isPaymentScheduleTypeValid(
    paymentScheduleType?: any,
): paymentScheduleType is PaymentScheduleType {
    return PAYMENT_SCHEDULE_TYPES.includes(paymentScheduleType);
}

export enum OrderPaymentInfoReceiptTypes {
    /**
     * Приход
     */
    FRT_ACQUIRE = 'FRT_ACQUIRE',

    /**
     * Возврат
     */
    FRT_CLEAR = 'FRT_CLEAR',
}

export interface IOrderPaymentInfoItemReceipt {
    url: string;
    type: OrderPaymentInfoReceiptTypes;
}

export enum OrderPaymentStatus {
    IS_NEW = 'IS_NEW',
    IS_WAIT_FOR_PAYMENT = 'IS_WAIT_FOR_PAYMENT',
    IS_HOLD = 'IS_HOLD',
    IS_PAYMENT_NOT_AUTHORIZED = 'IS_PAYMENT_NOT_AUTHORIZED',
    IS_CLEARING = 'IS_CLEARING',
    IS_CLEARED = 'IS_CLEARED',
    IS_REFUNDING = 'IS_REFUNDING',
    IS_REFUNDED = 'IS_REFUNDED',
}

enum OrderPaymentFiscalItemType {
    TRAIN_TICKET = 'TRAIN_TICKET',
    TRAIN_SERVICE = 'TRAIN_SERVICE',
    TRAIN_FEE = 'TRAIN_FEE',
    TRAIN_INSURANCE = 'TRAIN_INSURANCE',
}

interface IOrderPaymentInvoiceItem {
    invoice_item_id: number;
    fiscal_item_id: number;
    fiscal_item_type: OrderPaymentFiscalItemType;
    money_amount: IPrice;
}

export enum EOrderPaymentRefundState {
    RS_NEW = 'RS_NEW',
    RS_CREATE = 'RS_CREATE',
    RS_START = 'RS_START',
    RS_IN_PROCESS = 'RS_IN_PROCESS',
    RS_SUCCESS = 'RS_SUCCESS',
    RS_FAILED = 'RS_FAILED',
    RS_ERROR = 'RS_ERROR',
}

export enum EOrderPaymentResizeState {
    RSS_UNKNOWN = 'RSS_UNKNOWN',
    RSS_SUCCESS = 'RSS_SUCCESS',
}

export enum ETrustRefundType {
    REFUND = 'trust_refund',
    RESIZE = 'trust_resize',
}

interface IOrderPaymentCommon {
    id: string;
    trust_refund_id: string;
    order_refund_id: string;
    created_at: string;
    confirmed_at: string | null;
    refund_items: {
        id: number;
        original_amount: IPrice;
        target_amount: IPrice;
    }[];
}

interface IOrderPaymentRefund extends IOrderPaymentCommon {
    type: ETrustRefundType.REFUND;
    state: EOrderPaymentRefundState;
}

interface IOrderPaymentResize extends IOrderPaymentCommon {
    type: ETrustRefundType.RESIZE;
    state: EOrderPaymentResizeState;
}

export interface IOrderPaymentInfoItem {
    trust_payment_id: string;
    paid_at: string | null;
    cancelled_at: string | null;
    status: OrderPaymentStatus;
    payment_resp_code: string;
    payment_resp_desc: string;
    rrn: string;
    card_number: string;
    scrooge_url: string;
    payment_url: string;
    original_price: IPrice;
    current_price: IPrice;
    invoice_items: IOrderPaymentInvoiceItem[] | null;
    trust_refunds: (IOrderPaymentRefund | IOrderPaymentResize)[] | null;
    receipts: IOrderPaymentInfoItemReceipt[];
    approval_code: string;
}

interface IOrderOwner {
    login: string;
    yandex_uid: string;
    passport_id: string;
    email: string;
    phone: string;
    ip: string;
    allows_subscription: boolean;
}

export enum AuthorizedUserRoles {
    OWNER = 'OWNER',
    VIEWER = 'VIEWER',
}

export interface IAuthorizedUser {
    login: string;
    role: AuthorizedUserRoles;
    yandex_uid: string;
    is_logged_in: boolean;
}

export interface IOrderFlags {
    has_masked_info: boolean;
    can_resend_refund_mail: boolean;
    can_resend_successful_mail: boolean;
    can_manual_refund_all_money: boolean;
    can_manual_refund_yandex_fee: boolean;
    can_manual_refund_by_fiscal_item: boolean;
    can_restore_dolphin_order: boolean;
    can_regenerate_vouchers: boolean;
    can_refund_hotel_order: boolean;
    can_refund_hotel_money_only: boolean;
    can_modify_hotel_order_details: boolean;
    can_retry_money_refund: boolean;
}

export enum OrderRefundType {
    RT_TRAIN_USER_REFUND = 'RT_TRAIN_USER_REFUND',
    RT_TRAIN_OFFICE_REFUND = 'RT_TRAIN_OFFICE_REFUND',
    RT_TRAIN_INSURANCE_AUTO_RETURN = 'RT_TRAIN_INSURANCE_AUTO_RETURN',
    RT_GENERIC_USER_REFUND = 'RT_GENERIC_USER_REFUND',
}

export interface IOrderRefund {
    id: string;
    state: string;
    type: OrderRefundType;
    created_at: string;
    updated_at: string;
    refunded_amount: IPrice;
}

export enum EPromoCodeApplicationResultType {
    UNKNOWN = 'UNKNOWN', // неизвестно
    SUCCESS = 'SUCCESS', // успешно применен
    NOT_FOUND = 'NOT_FOUND', // не найден запрошенный промокод с таким кодом (в заказе такого не будет)
    NOT_APPLICABLE = 'NOT_APPLICABLE', // не применим
    ALREADY_APPLIED = 'ALREADY_APPLIED', //уже был применен и невозможен к повторному применению
    EXPIRED = 'EXPIRED', // истек
}

export interface IPromoCodeApplicationResult {
    promo_code_id: string;
    promo_code_activation_id: string;
    code: string;
    type: EPromoCodeApplicationResultType;
    discount_amount?: IPrice;
}

export interface IOrderPriceInfo {
    price: IPrice;
    original_price: IPrice;
    discount_amount: IPrice;
    promo_code_application_results: IPromoCodeApplicationResult[];
}

export interface IYandexPlus {
    mode: 'TOPUP' | 'WITHDRAW';
    topup_info?: IYandexPlusTopupInfo;
    withdrawal_info?: IYandexPlusWithdrawalInfo;
}

export interface IYandexPlusWithdrawalInfo {
    amount: number;
}
export interface IYandexPlusTopupInfo {
    amount: number;
    purchase_token: string | null;
    state: EYandexPlusTopupState | null;
    topup_date: string | null;
}

export enum EYandexPlusTopupState {
    PS_UNKNOWN = 'PS_UNKNOWN',
    PS_NEW = 'PS_NEW',
    PS_STARTING_PAYMENT = 'PS_STARTING_PAYMENT',
    PS_WAIT_FOR_PAYMENT = 'PS_WAIT_FOR_PAYMENT',
    PS_CLEARED = 'PS_CLEARED',
    PS_NOT_CLEARED = 'PS_NOT_CLEARED',
    PS_FINANCIAL_EVENT_SENT = 'PS_FINANCIAL_EVENT_SENT',
    UNRECOGNIZED = 'UNRECOGNIZED',
}

export enum ETaxiPromoStatus {
    OTPS_NOT_ELIGIBLE = 'OTPS_NOT_ELIGIBLE', // заказ не участвует в акции=
    OTPS_ELIGIBLE = 'OTPS_ELIGIBLE', // заказ участвует в акции
    OTPS_SENDING_EMAIL = 'OTPS_SENDING_EMAIL', // код выдан, письмо в процессе отправки
    OTPS_EMAIL_SENT = 'OTPS_EMAIL_SENT', // код выдан, письмо успешно отправлено
}

export interface ITaxiPromo {
    email: string | null;
    email_scheduled_at: string | null;
    email_sent_at: string | null;
    status: ETaxiPromoStatus;
}

export enum EMirEligibility {
    MIR_ELIGIBILE = 'MIR_ELIGIBILE',
    MIR_BLACKLISTED = 'MIR_BLACKLISTED',
    MIR_WRONG_LOS = 'MIR_WRONG_LOS',
    MIR_WRONG_STAY_DATES = 'MIR_WRONG_STAY_DATES',
    MIR_PROMO_DISABLED = 'MIR_PROMO_DISABLED',
}

export interface IMirPromo {
    offer_eligibility: EMirEligibility;
    paid_with_mir: boolean | null;
    cashback_amount: number | null;
}

export enum EPaymentScheduleState {
    PS_FULLY_PAID = 'PS_FULLY_PAID',
    PS_PARTIALLY_PAID = 'PS_PARTIALLY_PAID',
}

export interface IPaymentSchedule {
    initial_invoice: IPaymentScheduleInvoice;
    items: IPaymentScheduleItem[];
}

export interface IPaymentScheduleItem {
    id: string;
    payment_ends_at: string;
    penalty_if_unpaid: IPrice;
    invoice: IPaymentScheduleInvoice;
}

export interface IPaymentScheduleInvoice {
    total_amount: IPrice;
    state: EPaymentScheduleState;
    payment_attempts: IPaymentAttempt[];
}

export interface IPaymentAttempt {
    receipts: IOrderPaymentInfoItemReceipt[];
}

export interface IOrderInfo {
    id: string;
    status: OrderStatus;
    display_order_type: DisplayOrderType;
    admin_action_token: string;
    created_at: string;
    updated_at: string;
    yandex_order_id: string;

    /**
     * Сссылка на заказ в Путешествиях
     */
    url: string;
    partners: OrderPartner[];
    hotel_order_items: IHotelOrderItem[];
    avia_order_items: IAviaOrderItem[];
    train_order_items: ITrainOrderItem[];
    bus_order_items: IBusOrderItem[];
    payment_info: IOrderPaymentInfoItem[];
    promo_campaigns?: {
        taxi2020?: ITaxiPromo;
        mir2020?: IMirPromo;
        yandex_plus?: IYandexPlus;
    };
    owner: IOrderOwner | null;
    flags: IOrderFlags;
    authorized_users: IAuthorizedUser[];
    star_trek_ticket_ids: string[] | null;
    order_refunds: IOrderRefund[] | null;
    workflow_id: string;
    user_action_scheduled: boolean;
    broken: boolean;
    order_price_info: IOrderPriceInfo;
    vouchers: IOrderVoucher[] | null;
    payment_schedule?: IPaymentSchedule | null;
    is_post_pay_eligible: boolean;
    is_post_paid: boolean;
}

export interface IFetchError {
    status: number;
    error: string;
    message: string;
    stack?: string;
}

export type IStoreFetchError = IFetchError | null;

export enum OrderLogLevel {
    IRRELEVANT = '',
    INFO = 'INFO',
    WARN = 'WARN',
    ERROR = 'ERROR',
    DEBUG = 'DEBUG',
}

const ORDER_LOG_LEVELS: OrderLogLevel[] = [
    OrderLogLevel.IRRELEVANT,
    OrderLogLevel.INFO,
    OrderLogLevel.WARN,
    OrderLogLevel.ERROR,
    OrderLogLevel.DEBUG,
];

export function isOrderLogLevel(level: any): level is OrderLogLevel {
    return ORDER_LOG_LEVELS.includes(level);
}

export interface IOrderLog {
    level: OrderLogLevel;
    logger: string;
    message: string;
    message_id: string;
    owner_uuid: string;
    timestamp: string;
    context: string;
    host_name: string;
}

export interface IOrderPayloadInfo {
    order_item_id: string;
    payload: any;
    version: number;
}

export interface IOrderStartrekTicket {
    link: string;
    image: {
        width: number;
        height: number;
        src: string;
    };
    id: {
        value: string;
        color: string;
        strike: boolean;
    };
    status: {
        value: string;
        color: string;
    };
    title: {
        value: string;
    };
    user: {
        value: string;
    };
}

export interface IOrderVoucher {
    id: string;
    name: string;
    url: string;
}

export interface ICalculatedHotelOrderRefund {
    refund_amount_by_rules: IPrice;
    total_amount: IPrice;
    paid_amount: IPrice;
}

export interface ICalculatedHotelMoneyOnlyRefund {
    total_amount: IPrice;
    remaining_amount: IPrice;
}

export interface IHotelGuest {
    firstName: string;
    lastName: string;
    age?: number;
    child?: boolean;
}

export interface IStoreOrder {
    item: {
        value: IOrderInfo | null;
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    logs: {
        value: IOrderLog[];
        overallItemsCount: number;
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    workflow: {
        value: any;
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    payloads: {
        items: {
            value: IOrderPayloadInfo[] | null;
            isLoading: boolean;
            isFetched: boolean;
            error: IStoreFetchError;
        };
        edit: {
            isSuccess: boolean;
            isLoading: boolean;
            isFetched: boolean;
            error: IStoreFetchError;
        };
    };
    emailEdition: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    phoneEdition: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    sendingNotification: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    retryingMoneyRefund: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    updatingTrainTickets: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    refund: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    startrekTickets: {
        value: IOrderStartrekTicket[] | null;
        isLoading: boolean;
        isFetched: boolean;
        error: unknown;
    };
    vouchers: {
        value: IOrderVoucher[] | null;
    };
    restoreDolphinOrder: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    regenerateVouchers: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    calculateHotelOrderRefund: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
        calculatedRefund: ICalculatedHotelOrderRefund | null;
    };
    refundHotelOrder: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    calculateHotelMoneyOnlyRefund: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
        calculatedRefund: ICalculatedHotelMoneyOnlyRefund | null;
    };
    refundHotelMoneyOnly: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
    modifyHotelOrderDetails: {
        isLoading: boolean;
        isFetched: boolean;
        error: IStoreFetchError;
    };
}

export interface IStoreOrders {
    value: IBriefOrderInfo[];
    isLoading: boolean;
    isFetched: boolean;
    error: IStoreFetchError;
    overallItemsCount: number;

    isOptionsLoading: boolean;
    isOptionsFetched: boolean;
    optionsError: IStoreFetchError;
    stateOptions: OrderStatus[];
    displayTypeOptions: DisplayOrderType[];
    partnerOptions: OrderPartner[];
}

export interface IStore {
    env: Environment;
    userInfo: IStoreUserInfo;
    settings: IStoreSettings;
    order: IStoreOrder;
    orders: IStoreOrders;
}
