import {takeLatest, call, put} from 'redux-saga/effects';
import {ActionType, getType} from 'typesafe-actions';
import {AxiosResponse} from 'axios';
import apiClient from 'services/apiClient';
import {
    ICalculateHotelOrderRefundRequestParams,
    calculateHotelOrderRefundRequest,
    calculateHotelMoneyOnlyRefundRequest,
    calculateHotelMoneyOnlyRefundRequestFailure,
    calculateHotelMoneyOnlyRefundRequestSuccess,
} from 'redux/reducers/order/actions';

import {ICalculatedHotelMoneyOnlyRefund} from 'redux/reducers/types';

import isAxiosError from 'lib/isAxiosError';

function requestCalculateHotelMoneyOnlyRefund(
    data: Omit<ICalculateHotelOrderRefundRequestParams, 'addToast'>,
): Promise<AxiosResponse> {
    return apiClient.get(
        `/api/orders/${data.orderId}/calculateHotelMoneyOnlyRefund`,
    );
}

function* handleFetch(
    action: ActionType<typeof calculateHotelOrderRefundRequest>,
) {
    const {
        payload: {addToast, ...callParams},
    } = action;

    try {
        const {
            data: calculatedHotelMoneyOnlyRefund,
        }: AxiosResponse<ICalculatedHotelMoneyOnlyRefund> = yield call(
            requestCalculateHotelMoneyOnlyRefund,
            callParams,
        );

        yield put(
            calculateHotelMoneyOnlyRefundRequestSuccess(
                calculatedHotelMoneyOnlyRefund,
            ),
        );

        addToast('Запрощены данные для возврата денежных средств', {
            appearance: 'success',
            autoDismiss: true,
        });
    } catch (error) {
        addToast('Ошибка при загрузке данных для возврата денежных средств', {
            appearance: 'error',
            autoDismiss: true,
        });

        if (isAxiosError(error) && error.response && error.response.data) {
            yield put(
                calculateHotelMoneyOnlyRefundRequestFailure(
                    error.response.data,
                ),
            );
        }
    }
}

export default function* calculateHotelMoneyOnlyRefund() {
    yield takeLatest(
        getType(calculateHotelMoneyOnlyRefundRequest),
        handleFetch,
    );
}
