import {takeLatest, call, put} from 'redux-saga/effects';
import {ActionType, getType} from 'typesafe-actions';
import {AxiosResponse} from 'axios';
import apiClient from 'services/apiClient';
import {
    ICalculateHotelOrderRefundRequestParams,
    calculateHotelOrderRefundRequest,
    calculateHotelOrderRefundRequestSuccess,
    calculateHotelOrderRefundRequestFailure,
} from 'redux/reducers/order/actions';

import {ICalculatedHotelOrderRefund} from 'redux/reducers/types';

import isAxiosError from 'lib/isAxiosError';

function requestCalculateHotelOrderRefund(
    data: Omit<ICalculateHotelOrderRefundRequestParams, 'addToast'>,
): Promise<AxiosResponse> {
    return apiClient.get(
        `/api/orders/${data.orderId}/calculateHotelOrderRefund`,
    );
}

function* handleFetch(
    action: ActionType<typeof calculateHotelOrderRefundRequest>,
) {
    const {
        payload: {addToast, ...callParams},
    } = action;

    try {
        const {
            data: calculatedHotelOrderRefund,
        }: AxiosResponse<ICalculatedHotelOrderRefund> = yield call(
            requestCalculateHotelOrderRefund,
            callParams,
        );

        yield put(
            calculateHotelOrderRefundRequestSuccess(calculatedHotelOrderRefund),
        );

        addToast('Запрощены данные для отмены заказа', {
            appearance: 'success',
            autoDismiss: true,
        });
    } catch (error) {
        addToast('Ошибка при загрузке данных для отмены заказа', {
            appearance: 'error',
            autoDismiss: true,
        });

        if (isAxiosError(error) && error.response && error.response.data) {
            yield put(
                calculateHotelOrderRefundRequestFailure(error.response.data),
            );
        }
    }
}

export default function* calculateHotelOrderRefund() {
    yield takeLatest(getType(calculateHotelOrderRefundRequest), handleFetch);
}
