import {takeLatest, call, put, select, delay} from 'redux-saga/effects';
import {ActionType, getType} from 'typesafe-actions';
import {AxiosResponse} from 'axios';
import apiClient from 'services/apiClient';
import {
    OrderRefundRequestParams,
    orderRefundRequest,
    orderRefundRequestFailure,
    orderRefundRequestSuccess,
    orderRequestSuccess,
    orderRefundRequestTimeout,
} from 'redux/reducers/order/actions';
import {IFetchOrderResponse, requestOrder} from 'redux/sagas/order/orderSaga';

import {IStore} from 'redux/reducers/types';

import isAxiosError from 'lib/isAxiosError';

const MAX_POLLS_TRIES = 20;
const POLL_RETRY_DELAY_MS = 2000;

function requestOrderRefund(
    data: Omit<OrderRefundRequestParams, 'addToast'>,
): Promise<AxiosResponse> {
    return apiClient.post(`/api/orders/${data.orderId}/refund`, data);
}

function* handleFetch(action: ActionType<typeof orderRefundRequest>) {
    const {
        payload: {addToast, ...refundRequestParams},
    } = action;

    const {
        order: {
            item: {value: order},
        },
    }: IStore = yield select((store: IStore) => store);

    if (!order) {
        return;
    }

    try {
        yield call(requestOrderRefund, refundRequestParams);

        let isOrderQuerying = true;
        let pollsCounter = 0;

        while (isOrderQuerying && pollsCounter++ < MAX_POLLS_TRIES) {
            const {
                data: {
                    order: updatedOrder,
                    order: {user_action_scheduled},
                },
            }: AxiosResponse<IFetchOrderResponse> = yield call(requestOrder, {
                orderId: order.id,
                needToFetchPersonalData: !order.flags.has_masked_info,
            });

            if (!user_action_scheduled) {
                isOrderQuerying = false;

                yield put(orderRequestSuccess(updatedOrder));
            } else {
                yield delay(POLL_RETRY_DELAY_MS);
            }
        }

        if (isOrderQuerying) {
            addToast('Превышено время ожидания. Попробуйте ещё раз.', {
                appearance: 'error',
                autoDismiss: true,
            });

            yield put(orderRefundRequestTimeout());

            return;
        }

        yield put(orderRefundRequestSuccess());

        addToast('Возврат прошел успешно', {
            appearance: 'success',
            autoDismiss: true,
        });
    } catch (error) {
        addToast('Ошибка при возврате', {
            appearance: 'error',
            autoDismiss: true,
        });

        if (isAxiosError(error) && error.response && error.response.data) {
            yield put(orderRefundRequestFailure(error.response.data));
        }
    }
}

export default function* orderRefund() {
    yield takeLatest(getType(orderRefundRequest), handleFetch);
}
