import {takeLatest, call, put} from 'redux-saga/effects';
import {ActionType, getType} from 'typesafe-actions';
import axios, {AxiosResponse} from 'axios';
import {
    IOrderStartrekTicketsRequestParams,
    orderStartrekTicketsRequest,
    orderStartrekTicketsRequestFailure,
    orderStartrekTicketsRequestSuccess,
} from 'redux/reducers/order/actions';

import {IOrderStartrekTicket} from 'redux/reducers/types';

import isAxiosError from 'lib/isAxiosError';

interface IStartrekTicketsResponse {
    debug: {
        duration: number;
        view: string;
        method: string;
    };
    drop_client_cache: boolean;
    data: Record<
        string,
        {
            value: {
                type: string;
                src: string;
                width: number;
                height: number;
                text: string;
                value: string;
                color: string;
                action: {
                    event: string;
                    type: string;
                    url: string;
                };
                strike: boolean;
                login?: string;
            }[];
            type: string;
            separator: string;
            ttl: number;
            completed: boolean;
        }
    >;
}

function prepareStartrekTicketsResponse(
    startrekTicketsData: IStartrekTicketsResponse,
): IOrderStartrekTicket[] {
    return Object.values(startrekTicketsData.data).map(ticketData => {
        const [imageInfo, idInfo, statusInfo, titleInfo, userInfo] =
            ticketData.value;

        return {
            link: idInfo.action.url,
            completed: ticketData.completed,
            image: {
                width: imageInfo.width,
                height: imageInfo.height,
                src: imageInfo.src,
            },
            id: {
                value: idInfo.value,
                color: idInfo.color,
                strike: idInfo.strike,
            },
            status: {
                value: statusInfo.value,
                color: statusInfo.color,
            },
            title: {
                value: titleInfo.value,
            },
            user: {
                value: userInfo?.login || '',
            },
        };
    });
}

function requestOrderStartrekTickets({
    ticketsIds,
}: IOrderStartrekTicketsRequestParams): Promise<
    AxiosResponse<IStartrekTicketsResponse>
> {
    return axios({
        method: 'post',
        url: 'https://wf.yandex-team.ru/magiclinks/v1/links',
        data: {
            links: ticketsIds.map(
                ticketId => `https://st.yandex-team.ru/${ticketId}`,
            ),
        },
        withCredentials: true,
    });
}

function* handleFetch(action: ActionType<typeof orderStartrekTicketsRequest>) {
    try {
        const {payload} = action;

        const {
            data: starteckTicketsData,
        }: AxiosResponse<IStartrekTicketsResponse> = yield call(
            requestOrderStartrekTickets,
            payload,
        );

        yield put(
            orderStartrekTicketsRequestSuccess(
                prepareStartrekTicketsResponse(starteckTicketsData),
            ),
        );
    } catch (error) {
        if (isAxiosError(error) && error.response && error.response.data) {
            yield put(orderStartrekTicketsRequestFailure(error.response.data));
        }
    }
}

export default function* orderWorkflow() {
    yield takeLatest(getType(orderStartrekTicketsRequest), handleFetch);
}
