import {takeLatest, call, put, select, delay} from 'redux-saga/effects';
import {ActionType, getType} from 'typesafe-actions';
import {AxiosResponse} from 'axios';
import apiClient from 'services/apiClient';
import {
    IOrderUpdateTrainTicketsRequestParams,
    orderRequestSuccess,
    orderUpdateTrainTicketsRequest,
    orderUpdateTrainTicketsRequestFailure,
    orderUpdateTrainTicketsRequestSuccess,
    orderUpdateTrainTicketsRequestTimeout,
} from 'redux/reducers/order/actions';
import {IFetchOrderResponse, requestOrder} from 'redux/sagas/order/orderSaga';

import {IStore} from 'redux/reducers/types';

import isAxiosError from 'lib/isAxiosError';

const MAX_POLLS_TRIES = 20;
const POLL_RETRY_DELAY_MS = 2000;

function requestOrderUpdateTrainTickets(
    data: Omit<IOrderUpdateTrainTicketsRequestParams, 'addToast'>,
): Promise<AxiosResponse<any>> {
    return apiClient.get(
        `/api/orders/${data.orderId}/updateTrainTicketsStatus`,
    );
}

function* handleFetch(
    action: ActionType<typeof orderUpdateTrainTicketsRequest>,
) {
    const {
        payload: {addToast, ...updateTrainTicketsParams},
    } = action;

    const {
        order: {
            item: {value: order},
        },
    }: IStore = yield select((store: IStore) => store);

    if (!order) {
        return;
    }

    try {
        yield call(requestOrderUpdateTrainTickets, updateTrainTicketsParams);

        let isOrderQuerying = true;
        let pollsCounter = 0;

        while (isOrderQuerying && pollsCounter++ < MAX_POLLS_TRIES) {
            const {
                data: {
                    order: updatedOrder,
                    order: {user_action_scheduled},
                },
            }: AxiosResponse<IFetchOrderResponse> = yield call(requestOrder, {
                orderId: order.id,
                needToFetchPersonalData: !order.flags.has_masked_info,
            });

            if (!user_action_scheduled) {
                isOrderQuerying = false;

                yield put(orderRequestSuccess(updatedOrder));
            } else {
                yield delay(POLL_RETRY_DELAY_MS);
            }
        }

        if (isOrderQuerying) {
            addToast('Превышено время ожидания. Попробуйте ещё раз.', {
                appearance: 'error',
                autoDismiss: true,
            });

            yield put(orderUpdateTrainTicketsRequestTimeout());

            return;
        }

        yield put(orderUpdateTrainTicketsRequestSuccess());

        addToast('Статус обновлен', {
            appearance: 'success',
            autoDismiss: true,
        });
    } catch (error) {
        addToast('Ошибка обновления статуса', {
            appearance: 'error',
            autoDismiss: true,
        });

        if (isAxiosError(error) && error.response && error.response.data) {
            yield put(
                orderUpdateTrainTicketsRequestFailure(error.response.data),
            );
        }
    }
}

export default function* orderUpdateTrainTickets() {
    yield takeLatest(getType(orderUpdateTrainTicketsRequest), handleFetch);
}
