package ru.yandex.travel.acceptance;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.StandardEnvironment;

import ru.yandex.travel.acceptance.orders.vault.VaultClient;
import ru.yandex.travel.acceptance.orders.vault.VaultPropertiesLoader;

@Slf4j
public class AcceptanceUtils {
    // for local dev purposes
    public static String readLocalYavToken() {
        try {
            String home = System.getProperty("user.home");
            return Files.readString(Path.of(home, ".yav", "token")).trim();
        } catch (Exception e) {
            log.info("No local yav token", e);
            return null;
        }
    }

    public static Map<String, String> tryRetrievingVaultSecrets(String version, Collection<String> keys) {
        // YAV Oauth Token: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=ce68fbebc76c4ffda974049083729982
        String vaultOauth = System.getenv("VAULT_OAUTH_TOKEN");
        if (Strings.isNullOrEmpty(vaultOauth)) {
            vaultOauth = AcceptanceUtils.readLocalYavToken();
        }
        if (!Strings.isNullOrEmpty(vaultOauth)) {
            try {
                VaultPropertiesLoader propertiesLoader = new VaultPropertiesLoader(vaultOauth, VaultClient.Type.PROD);
                return propertiesLoader.getProperties(version, keys);
            } catch (Exception e) {
                // An error occurred while fetching properties from vault
                log.error("Yav properties lookup failed", e);
            }
        }
        return Collections.emptyMap();
    }

    public static void setPropertiesFromVault(String secretKey, Map<String, String> secretToEnvPropertyMap,
                                              StandardEnvironment environment) {
        Map<String, String> secretValueMap = tryRetrievingVaultSecrets(secretKey, secretToEnvPropertyMap.keySet());
        Map<String, Object> envPropertySecretPropertyMap =
                secretValueMap.entrySet()
                        .stream()
                        .reduce(new HashMap<>(),
                                (map, entry) -> {
                                    map.put(secretToEnvPropertyMap.get(entry.getKey()), entry.getValue());
                                    return map;
                                },
                                (a, b) -> a);
        environment.getPropertySources().addLast(new MapPropertySource("vault-properties",
                envPropertySecretPropertyMap));
    }
}
