package ru.yandex.travel.acceptance.orders.invoice.trust;

import java.util.Map;
import java.util.UUID;

import com.google.common.collect.ImmutableMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.jpa.repository.JpaRepository;

import ru.yandex.bolts.collection.Option;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.SimpleTrustRefund;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.repository.SimpleTrustRefundRepository;
import ru.yandex.travel.orders.repository.TrustInvoiceRepository;
import ru.yandex.travel.workflow.BasicMessagingContext;
import ru.yandex.travel.workflow.MessagingContextFactory;
import ru.yandex.travel.workflow.WorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowEventHandlerMatcher;

@Configuration
public class TrustInvoiceAcceptanceConfiguration {

    @Autowired
    @Qualifier("trustInvoiceWorkflowEventHandler")
    WorkflowEventHandler<Invoice> trustInvoiceWorkflowEventHandler;

    @Autowired
    @Qualifier("trustRefundWorkflowEventHandler")
    WorkflowEventHandler<SimpleTrustRefund> trustRefundWorkflowEventHandler;

    @Bean
    public CheckerRegistry checkerRegistry() {
        return new CheckerRegistry();
    }


    @Bean
    public WorkflowEventHandlerMatcher workflowEventHandlerMatcher(CheckerRegistry checkerRegistry) {
        Map<String, WorkflowEventHandler> handlers = ImmutableMap.<String, WorkflowEventHandler>builder()
                .put(WellKnownWorkflowEntityType.TRUST_INVOICE.getDiscriminatorValue(),
                        trustInvoiceWorkflowEventHandler)
                .put(WellKnownWorkflowEntityType.TRUST_REFUND.getDiscriminatorValue(), trustRefundWorkflowEventHandler)
                .build();
        return (workflow, event) -> {
            if (TestWorkflows.CHECKER.getEntityType().equals(workflow.getEntityType())) {
                return Option.ofNullable(checkerRegistry.getCheckerHandlerForWorkflow(workflow.getId()));
            } else {
                return Option.ofNullable(handlers.get(workflow.getEntityType()));
            }
        };
    }

    @Bean
    public MessagingContextFactory messagingContextFactory(TrustInvoiceRepository trustInvoiceRepository,
                                                           SimpleTrustRefundRepository simpleTrustRefundRepository) {
        Map<String, JpaRepository<?, UUID>> repositories = ImmutableMap.<String, JpaRepository<?, UUID>>builder()
                .put(WellKnownWorkflowEntityType.TRUST_INVOICE.getDiscriminatorValue(), trustInvoiceRepository)
                .put(WellKnownWorkflowEntityType.TRUST_REFUND.getDiscriminatorValue(), simpleTrustRefundRepository)
                .put(TestWorkflows.CHECKER.getEntityType(), trustInvoiceRepository)
                .build();
        return (workflow, attempt) -> {
            JpaRepository<?, UUID> repository = repositories.get(workflow.getEntityType());
            if (repository == null) {
                return Option.empty();
            }
            return Option.of(new BasicMessagingContext<>(workflow.getId(),
                    repository.getOne(workflow.getEntityId()), attempt, workflow.getWorkflowVersion()));
        };
    }
}
