package ru.yandex.travel.acceptance.orders.invoice.trust;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentClear;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentRefund;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class TrustInvoiceHoldClearRefundCheckerHandler extends TrustInvoiceCheckerBase {

    @Autowired
    public TrustInvoiceHoldClearRefundCheckerHandler(TrustPayer trustPayer) {
        super(trustPayer);
    }

    @HandleEvent
    public void handleInvoicePaymentStarted(TInvoicePaymentStarted message, MessagingContext<Invoice> ctx) {
        try {
            trustPayer.pay(ctx.getWorkflowEntity().getPaymentUrl(), createCardParams(),
                    ctx.getWorkflowEntity().getReturnPath());
        } catch (Exception e) {
            log.error("Error paying for invoice {}", ctx.getWorkflowEntity().getId(), e);
            fail();
        }
    }

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired message, MessagingContext<Invoice> ctx) {
        log.info("Money acquired");
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getWorkflow().getId(), TPaymentClear.newBuilder().build());

    }

    @HandleEvent
    public void handleInvoiceCleared(TInvoiceCleared message, MessagingContext<Invoice> ctx) {
        log.info("Money cleared");
        // refunding half of the money, to have a partial refund
        TPaymentRefund.Builder refundBuilder = TPaymentRefund.newBuilder();
        Map<Long, Money> refundMap = ctx.getWorkflowEntity().getInvoiceItems().stream().collect(Collectors.toMap(
                InvoiceItem::getFiscalItemId,
                ii -> Money.of(ii.getPrice().divide(BigDecimal.valueOf(2), RoundingMode.CEILING),
                        ctx.getWorkflowEntity().getCurrency()))
        );
        refundBuilder.putAllTargetFiscalItems(RefundingUtils.convertTargetFiscalItemsToProto(refundMap));
        refundBuilder.setReason("Cancel booking");

        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getWorkflow().getId(), refundBuilder.build());
    }

    @HandleEvent
    public void handleInvoiceRefunded(TInvoiceRefunded message, MessagingContext<Invoice> ctx) {
        log.info("Invoice refunded");
        pass();
    }

    @HandleEvent
    public void handleWorkflowCrashedEvent(TWorkflowCrashed message, MessagingContext<?> messagingContext) {
        log.info("Test failed as workflow {} for entity {} of type {} crashed", message.getWorkflowId(),
                message.getEntityId(), message.getEntityType());
        fail();
    }
}
