package ru.yandex.travel.acceptance.orders.invoice.trust;

import java.net.MalformedURLException;
import java.net.URL;

import lombok.extern.slf4j.Slf4j;
import org.openqa.selenium.By;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.openqa.selenium.support.ui.WebDriverWait;
import org.springframework.stereotype.Service;

@Service
@Slf4j
public class TrustPayer {
    public void pay(String paymentUrl, CardParams card, String returnPath) {
        RuntimeException lastError;
        int maxAttempts = 3;
        int attempt = 0;
        do {
            try {
                tryPaymentOnce(paymentUrl, card, returnPath);
                return;
            } catch (RuntimeException e) {
                // e.g.: org.openqa.selenium.WebDriverException: Session timed out or not found
                int attemptsLeft = maxAttempts - attempt - 1;
                lastError = e;
                log.warn("Selenium-based payment attempt has failed (attempts left {}): {}",
                        attemptsLeft, e.getMessage(), e);
            }
        } while (++attempt < maxAttempts);
        throw lastError;
    }

    private void tryPaymentOnce(String paymentUrl, CardParams card, String returnPath) {

        log.info("Opening URL: {}", paymentUrl);

        WebDriver webDriver = createRemoteWebDriver();

        webDriver.navigate().to(paymentUrl);

        log.info("Waiting for the payment form to load.");

        new WebDriverWait(webDriver, 30).until(
                ExpectedConditions.presenceOfElementLocated(By.id("pcidss-frame")));

        log.info("Filling payment data...");

        webDriver.switchTo().frame("pcidss-frame");
        fillThePaymentForm(card, webDriver);
        webDriver.switchTo().defaultContent();

        WebElement payButton = webDriver.findElement(By.xpath("//div[@class='footer']//button"));
        log.info("Submitting payment form");
        payButton.click();

        new WebDriverWait(webDriver, 60).until(ExpectedConditions.urlContains(returnPath));
    }

    private WebDriver createRemoteWebDriver() {
        DesiredCapabilities capabilities = DesiredCapabilities.chrome();
        capabilities.setVersion("77.0");
        WebDriver webDriver;
        try {
            webDriver = new RemoteWebDriver(
                    new URL("http://travel-front@sw.yandex-team.ru:80/v0"),
                    capabilities
            );
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
        return webDriver;
    }

    private void fillThePaymentForm(CardParams card, WebDriver webDriver) {
        webDriver.findElement(By.cssSelector(".card_number-input input")).sendKeys(card.getCardNumber());
        webDriver.findElement(By.cssSelector(".card_valid-date input")).sendKeys(
                card.getExpiresAtMonth() + card.getExpiresAtYear());
        webDriver.findElement(By.cssSelector(".card_back input")).sendKeys(card.getCvc());
    }
}
