package ru.yandex.travel.acceptance.orders.orderitem.expedia;

import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.proto.THotelRefundToken;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TReservationStart;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class CheckerHandler extends AnnotatedWorkflowEventHandler<ExpediaOrderItem> {
    private final CountDownLatch latch;
    private final Queue<Class<? extends Message>> expectedMessages;
    @Getter
    private final List<Class<? extends Message>> expectedMessagesArchive;
    private final AtomicBoolean errorsEncountered;
    private final RefundCalculationService refundCalculationService;
    private final TransactionTemplate transactionTemplate;
    @Getter
    private final UUID workflowToRun;
    private final AtomicBoolean initialized;
    @Getter
    private final ExpectedTestResult.Outcome outcome;

    static class ChainBuilder {
        private final CheckerHandler handler;

        ChainBuilder(CheckerHandler handler) {
            this.handler = handler;
        }

        public CheckerHandler andThenDone() {
            Preconditions.checkState(handler.initialized.compareAndSet(false, true),
                    "The handler has already been initialized");
            return handler;
        }

        public ChainBuilder andThen(Class<? extends Message> message) {
            return getsMessage(message);
        }

        public ChainBuilder getsMessage(Class<? extends Message> message) {
            handler.expectedMessages.add(message);
            handler.expectedMessagesArchive.add(message);
            return this;
        }
    }

    CheckerHandler(RefundCalculationService refundCalculationService, TransactionTemplate transactionTemplate,
                   UUID workflowToRun, ExpectedTestResult.Outcome outcome) {
        this.workflowToRun = workflowToRun;
        this.refundCalculationService = refundCalculationService;
        this.transactionTemplate = transactionTemplate;
        this.outcome = outcome;
        this.expectedMessages = new LinkedList<>();
        this.expectedMessagesArchive = new ArrayList<>();
        this.errorsEncountered = new AtomicBoolean(false);
        this.latch = new CountDownLatch(1);
        this.initialized = new AtomicBoolean(false);
    }

    ChainBuilder initializer() {
        Preconditions.checkState(!initialized.get(), "The handler has already been initialized");
        return new ChainBuilder(this);
    }

    @HandleEvent
    public void handleWorkflowCrashedEvent(TWorkflowCrashed message, MessagingContext<OrderItem> messagingContext) {
        log.info("Test failed as workflow {} for entity {} of type {} crashed", message.getWorkflowId(),
                message.getEntityId(), message.getEntityType());
        setError();
    }

    @HandleEvent
    public void handleReserved(TServiceReserved message, MessagingContext<ExpediaOrderItem> messagingContext) {
        checkMessageInSequence(message.getServiceId(), message);
        log.info("Itinerary is reserved. Let's confirm the reservation now");
        ExpediaOrderItem item = messagingContext.getWorkflowEntity();
        messagingContext.scheduleExternalEvent(item.getWorkflow().getId(), TConfirmationStart.newBuilder().build());
    }

    @HandleEvent
    public void handleRefunded(TServiceRefunded message, MessagingContext<ExpediaOrderItem> messagingContext) {
        checkMessageInSequence(message.getServiceId(), message);
        log.info("Itinerary is refunded. We're done");
    }


    @HandleEvent
    public void handleCancelled(TServiceCancelled message, MessagingContext<ExpediaOrderItem> messagingContext) {
        checkMessageInSequence(message.getServiceId(), message);
        ExpediaOrderItem item = messagingContext.getWorkflowEntity();
        log.info("Itinerary is canceled. Reason=" + item.getItinerary().getOrderCancellationDetails().getReason());
    }

    @HandleEvent
    public void handleConfirmed(TServiceConfirmed message, MessagingContext<ExpediaOrderItem> messagingContext) {
        checkMessageInSequence(message.getServiceId(), message);
        log.info("Itinerary is confirmed. Hurray! Let's refund it now");
        ExpediaOrderItem item = messagingContext.getWorkflowEntity();
        RefundCalculationService.RefundCalculationWrapper<THotelRefundToken> refundCalculation =
                refundCalculationService.calculateRefundForHotelItemFromRules(item, ProtoCurrencyUnit.RUB);
        messagingContext.scheduleExternalEvent(item.getWorkflow().getId(),
                TRefundStart.newBuilder().setToken(refundCalculation.getRefundToken()).build());
    }

    public void run(WorkflowMessageSender workflowService) {
        transactionTemplate.execute(ignored -> {
            workflowService.scheduleEvent(this.workflowToRun, TReservationStart.newBuilder().build());
            return null;
        });
    }

    private void checkMessageInSequence(String itemId, Message message) {
        if (expectedMessages.isEmpty()) {
            log.error(String.format("Expected empty test sequence for order item with id=%s", itemId));
            setError();
        } else {
            if (!expectedMessages.peek().equals(message.getClass())) {
                log.error("Unexpected message for order item with id={}: expected {}, got {}",
                        itemId, expectedMessages.peek().getSimpleName(), message.getClass().getSimpleName());
                setError();
            } else {
                expectedMessages.poll();
                if (expectedMessages.isEmpty()) {
                    log.error("Got a new message while being in terminal state");
                    latch.countDown();
                }
            }
        }
    }

    private void setError() {
        if (!errorsEncountered.getAndSet(true)) {
            latch.countDown();
        }
    }

    public boolean waitTestResult(Duration awaitDuration) {
        try {
            boolean notInterrupted = latch.await(awaitDuration.toNanos(), TimeUnit.NANOSECONDS);
            return !errorsEncountered.get() && notInterrupted;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }
}
