package ru.yandex.travel.acceptance.orders.orderitem.expedia;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.travel.acceptance.orders.invoice.trust.TestWorkflows;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.repository.ExpediaOrderItemRepository;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.workflow.BasicMessagingContext;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.MessagingContextFactory;
import ru.yandex.travel.workflow.WorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowEventHandlerMatcher;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

@Component
@RequiredArgsConstructor
@Slf4j
public class CheckerHandlerFactory {
    private final RefundCalculationService refundCalculationService;
    private final PayloadPreparer payloadPreparer;
    private final ExpediaOrderItemRepository expediaOrderItemRepository;
    private final WorkflowRepository workflowRepository;
    private final TransactionTemplate transactionTemplate;
    private final ExpediaAcceptanceProperties properties;

    private final Map<UUID, CheckerHandler> checkerWorkflowIdsToHandlers = new HashMap<>();
    private final Map<UUID, UUID> checkerWorkflowIdsToItemIds = new HashMap<>();

    @Autowired
    @Qualifier("expediaWorkflowEventHandler")
    WorkflowEventHandler<ExpediaOrderItem> expediaWorkflowEventHandler;


    public CheckerHandler.ChainBuilder testExpect(ExpectedTestResult.Outcome outcome) {
        return transactionTemplate.execute(ignored -> {
            Workflow checker = new Workflow();
            UUID checkerId = UUID.randomUUID();
            log.info("Checker workflow will run as " + checkerId);
            checker.setEntityType(TestWorkflows.CHECKER.getEntityType());
            checker.setId(checkerId);
            checker.setState(EWorkflowState.WS_RUNNING);
            workflowRepository.saveAndFlush(checker);

            ExpediaOrderItem item = prepare(outcome, checkerId);
            checkerWorkflowIdsToItemIds.put(checkerId, item.getId());
            expediaOrderItemRepository.saveAndFlush(item);
            CheckerHandler checkerHandler = new CheckerHandler(refundCalculationService, transactionTemplate,
                    item.getWorkflow().getId(), outcome);
            checkerWorkflowIdsToHandlers.put(checkerId, checkerHandler);
            return checkerHandler.initializer();
        });
    }

    public WorkflowEventHandlerMatcher getHandlerMatcher() {
        return (workflow, event) -> {
            if (workflow.getEntityType().equals(TestWorkflows.CHECKER.getEntityType()) && checkerWorkflowIdsToHandlers.containsKey(workflow.getId())) {
                return Option.of(checkerWorkflowIdsToHandlers.get(workflow.getId()));
            } else if (workflow.getEntityType().equals(WellKnownWorkflowEntityType.EXPEDIA_ORDER_ITEM.getDiscriminatorValue())) {
                return Option.of(expediaWorkflowEventHandler);
            } else {
                return Option.empty();
            }
        };
    }

    public MessagingContextFactory getMessagingContextFactory() {
        return (workflow, attempt) -> {
            if (workflow.getEntityType().equals(TestWorkflows.CHECKER.getEntityType()) && checkerWorkflowIdsToItemIds.containsKey(workflow.getId())) {
                UUID orderItemId = checkerWorkflowIdsToItemIds.get(workflow.getId());
                return Option.of(new BasicMessagingContext<>(workflow.getId(),
                        expediaOrderItemRepository.getOne(orderItemId), attempt, workflow.getWorkflowVersion()));
            }
            if (workflow.getEntityType().equals(WellKnownWorkflowEntityType.EXPEDIA_ORDER_ITEM.getDiscriminatorValue())) {
                return Option.of(new BasicMessagingContext<>(workflow.getId(),
                        expediaOrderItemRepository.getOne(workflow.getEntityId()), attempt, workflow.getWorkflowVersion()));
            } else {
                return Option.empty();
            }
        };
    }

    public boolean waitAll() {
        return checkerWorkflowIdsToHandlers.values().stream().allMatch(handler -> handler.waitTestResult(properties.getMaxTestDuration()));
    }

    public String waitAndDescribeAll() {
        return checkerWorkflowIdsToHandlers.entrySet().stream()
                .map(e -> {
                    var handler = e.getValue();
                    boolean result = handler.waitTestResult(properties.getMaxTestDuration());
                    List<String> expectedMessages = handler.getExpectedMessagesArchive().stream()
                            .map(Class::getSimpleName)
                            .collect(toList());
                    return String.format("Checker id %s, workflow id %s, expected messages %s, %s -> %s",
                            e.getKey(), handler.getWorkflowToRun(), expectedMessages, handler.getOutcome(), result);
                })
                .collect(joining("; "));
    }

    private ExpediaOrderItem prepare(ExpectedTestResult.Outcome outcome,
                                     UUID checkerId) {
        ExpediaOrderItem item = new ExpediaOrderItem();
        item.setOrderWorkflowId(checkerId);
        item.setState(EExpediaItemState.IS_NEW);
        item.setItinerary(payloadPreparer.prepare(outcome));
        item = expediaOrderItemRepository.saveAndFlush(item);
        Workflow itemWorkflow = Workflow.createWorkflowForEntity(item, checkerId);
        workflowRepository.saveAndFlush(itemWorkflow);
        item = expediaOrderItemRepository.saveAndFlush(item);
        return item;
    }
}
