package ru.yandex.travel.acceptance.orders.orderitem.expedia;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.text.MessageFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.io.Resources;
import org.assertj.core.util.Lists;
import org.javamoney.moneta.Money;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.partners.expedia.ApiVersion;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaRefundRulesBuilder;
import ru.yandex.travel.hotels.common.partners.expedia.Helpers;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.PropertyAvailability;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.RoomAvailability;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.RoomPriceCheck;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.SalesChannel;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.ShoppingRate;
import ru.yandex.travel.hotels.common.refunds.RefundRules;

@Component
public class PayloadPreparer {
    @Autowired
    private ExpediaClient expediaClient;

    @Autowired
    private ExpediaAcceptanceProperties testProperties;


    public ExpediaHotelItinerary prepare(ExpectedTestResult.Outcome outcome) {
        ExpediaHotelItinerary itinerary = loadPayloadTemplate();
        String affiliateId = UUID.randomUUID().toString().substring(0, 28);
        itinerary.setAffiliateId(affiliateId);
        String shopSesionId = UUID.randomUUID().toString();
        String bookSessionId = UUID.randomUUID().toString();
        String sessionIdForExpeida = MessageFormat.format("{0}:{1}", shopSesionId, bookSessionId);
        BigDecimal fxRate = BigDecimal.valueOf(testProperties.getFxRate(), testProperties.getFxRatePrecision());
        LocalDate checkinDate = LocalDate.now().plusDays(testProperties.getCheckinAfterDays());
        LocalDate checkoutDate = checkinDate.plusDays(testProperties.getLengthOfStay());
        String occupancy = testProperties.getOccupancy();
        PropertyAvailability shoppedOffer = shop(shopSesionId, checkinDate, checkoutDate, occupancy);
        RoomAvailability room = shoppedOffer.getRooms().stream()
                .filter(r -> r.getRates().stream().anyMatch(ShoppingRate::isRefundable)).findAny().get();
        ShoppingRate rate =
                room.getRates().stream().filter(ShoppingRate::isRefundable).findAny().get();
        String priceCheckToken =
                Helpers.retrievePriceCheckToken(rate.getBedGroups().values().iterator().next().getLinks().getPriceCheck().getHref());
        RoomPriceCheck priceCheck = checkPrice(shoppedOffer.getPropertyId(), room.getId(), rate.getId(),
                priceCheckToken, itinerary.getCustomerIp(),
                itinerary.getCustomerUserAgent(), sessionIdForExpeida);
        String reservarionToken = Helpers.retrieveReservationToken(priceCheck.getLinks().getBook().getHref());
        rate = rate.toBuilder().occupancyPricing(priceCheck.getOccupancyPricing()).build();
        List<Guest> guests = generateGuests(occupancy);
        OrderDetails orderDetails = OrderDetails.builder().checkinDate(checkinDate).checkoutDate(checkoutDate).build();
        RefundRules refundRules = ExpediaRefundRulesBuilder.build(rate, occupancy, Duration.ofHours(2));
        itinerary.setRefundRules(refundRules);
        itinerary.setOrderDetails(orderDetails);
        itinerary.setCustomerSessionId(sessionIdForExpeida);
        itinerary.setHotelId(shoppedOffer.getPropertyId());
        itinerary.setRoomId(room.getId());
        itinerary.setRateId(rate.getId());
        itinerary.setOccupancy(occupancy);
        itinerary.setGuests(guests);
        itinerary.setExpediaReservationToken(reservarionToken);
        itinerary.setApiVersion(ApiVersion.V2_4);
        itinerary.setFiscalPrice(Money.of(Helpers.round(priceCheck.getOccupancyPricing().get(occupancy)).getTotals().getInclusive().getBillableCurrency().getValue(), "RUB"));
        if (outcome != ExpectedTestResult.Outcome.FX_RATE_CHECK_FAILURE) {
            itinerary.setExpiresAtInstant(Instant.now().plus(Duration.ofDays(1)));
        } else {
            itinerary.setExpiresAtInstant(Instant.now().minus(Duration.ofMinutes(1)));
        }
        return itinerary;
    }

    private List<Guest> generateGuests(String occupancyString) {
        String[] parts = occupancyString.split("-");
        Preconditions.checkArgument(parts.length == 1 || parts.length == 2, "Unexpected");
        int numAdults = Integer.parseInt(parts[0]);
        List<Integer> childAges;
        if (parts.length == 1) {
            childAges = Lists.emptyList();
        } else {
            childAges =
                    Arrays.stream(parts[1].split(",")).mapToInt(Integer::parseInt).boxed().collect(Collectors.toList());
        }
        List<Guest> result = new ArrayList<>(numAdults + childAges.size());
        for (int i = 1; i <= numAdults; i++) {
            Guest guest = new Guest();
            guest.setFirstName("ADULT");
            guest.setLastName("GUEST");
            guest.setChild(false);
            guest.setAge(null);
            result.add(guest);
        }
        for (int age : childAges) {
            Guest guest = new Guest();
            guest.setFirstName("CHILD");
            guest.setLastName("GUEST");
            guest.setChild(true);
            guest.setAge(age);
            result.add(guest);
        }
        return result;
    }

    private ExpediaHotelItinerary loadPayloadTemplate() {
        try {
            String template = Resources.toString(Resources.getResource("expediaPayloadTemplate.json"),
                    Charset.defaultCharset());
            TJson json = TJson.newBuilder()
                    .setValue(template)
                    .build();
            return ProtoUtils.fromTJson(json, ExpediaHotelItinerary.class);
        } catch (IOException ex) {
            throw new RuntimeException("Unable to load payload template");
        }
    }

    private PropertyAvailability shop(String shopSessionId, LocalDate checkinDate, LocalDate checkoutDate,
                                      String occupancy) {
        try {
            Map<String, PropertyAvailability> foundOffers =
                    expediaClient.findAvailabilities(testProperties.getHotelIds(), checkinDate, checkoutDate, occupancy,
                            "USD", true, testProperties.getSearchIpAddess(), shopSessionId, shopSessionId, SalesChannel.CACHE)
                            .toCompletableFuture().get();
            if (foundOffers.isEmpty()) {
                throw new RuntimeException("No availability in configured hotels in configured dates, unable to " +
                        "prepare order");
            }
            var filtered = foundOffers.values().stream()
                    .filter(pa -> pa.getRooms().stream()
                            .anyMatch(room -> room.getRates().stream()
                                    .anyMatch(ShoppingRate::isRefundable))).findAny();
            if (filtered.isEmpty()) {
                throw new RuntimeException("No refundable offers in configured hotels in configured dates, unable to " +
                        "prepare order");
            } else {
                return filtered.get();
            }
        } catch (Exception e) {
            throw new RuntimeException("Unable to execute shopping request", e);
        }
    }

    private RoomPriceCheck checkPrice(String propertyId, String roomId, String rateId, String token, String clientIp,
                                      String userAgent, String sessionId) {
        try {
            return expediaClient.checkPrice(propertyId, roomId, rateId, token, clientIp, userAgent, sessionId).toCompletableFuture().get();
        } catch (Exception e) {
            throw new RuntimeException("Unable to check price", e);
        }

    }
}
