package ru.yandex.travel.acceptance.orders.orderitem.train;

import java.time.Duration;
import java.util.Queue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.protobuf.Message;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class BaseTrainCheckerHandler extends AnnotatedWorkflowEventHandler<TrainOrderItem> {
    private final CountDownLatch latch;
    private final Queue<Class> expectedMessages;
    private final AtomicBoolean errorsEncountered;
    private final EOrderItemState expectedState;
    private EOrderItemState actualState;

    BaseTrainCheckerHandler(EOrderItemState expectedState, Queue<Class> expectedMessages) {
        this.expectedMessages = expectedMessages;
        this.errorsEncountered = new AtomicBoolean(false);
        this.latch = new CountDownLatch(1);
        this.expectedState = expectedState;
    }

    @HandleEvent
    public void handleWorkflowCrashedEvent(TWorkflowCrashed message, MessagingContext<TrainOrderItem> messagingContext) {
        log.info("Test failed as workflow {} for entity {} of type {} crashed", message.getWorkflowId(),
                message.getEntityId(), message.getEntityType());
        storeActualState(messagingContext.getWorkflowEntity());
        setError();
    }

    protected void storeActualState(TrainOrderItem item) {
        actualState = item.getState();
    }

    protected void checkMessageInSequence(Message message) {
        if (expectedMessages.isEmpty()) {
            log.error("Expected empty test sequence for order item: got {}", message.getClass().getSimpleName());
            setError();
        } else {
            if (!expectedMessages.peek().equals(message.getClass())) {
                log.error("Unexpected message for order item: expected {}, got {}",
                        expectedMessages.peek().getSimpleName(), message.getClass().getSimpleName());
                setError();
            } else {
                expectedMessages.poll();
                if (expectedMessages.isEmpty()) {
                    latch.countDown();
                }
            }
        }
    }

    protected void setError() {
        if (!errorsEncountered.getAndSet(true)) {
            latch.countDown();
        }
    }

    public boolean waitTestResult(Duration awaitDuration) {
        try {
            boolean notInterrupted = latch.await(awaitDuration.toNanos(), TimeUnit.NANOSECONDS);
            log.info("Test flow ended. actualState = {}, expectedState = {}", actualState, expectedState);
            return !errorsEncountered.get() && notInterrupted && actualState == expectedState;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }
}
