package ru.yandex.travel.acceptance.orders.orderitem.train;

import java.io.IOException;
import java.nio.charset.Charset;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import com.google.common.io.Resources;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.travel.acceptance.orders.invoice.trust.TestWorkflows;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.orders.entities.TrainInsuranceRefund;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TrainInsuranceRefundRepository;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.repository.TrainTicketRefundRepository;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TReservationStart;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.workflow.BasicMessagingContext;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.MessagingContextFactory;
import ru.yandex.travel.workflow.WorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowEventHandlerMatcher;
import ru.yandex.travel.workflow.WorkflowProcessService;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static java.util.stream.Collectors.toList;

@Component
@RequiredArgsConstructor
@Slf4j
public class TrainCheckerHandlerManager {
    private final RefundCalculationService refundCalculationService;
    private final TrainOrderItemRepository trainOrderItemRepository;
    private final TrainTicketRefundRepository trainTicketRefundRepository;
    private final TrainInsuranceRefundRepository trainInsuranceRefundRepository;
    private final OrderRepository trainOrderRepository;
    private final WorkflowRepository workflowRepository;
    private final TransactionTemplate transactionTemplate;
    private final TrainAcceptanceProperties properties;

    private final Map<UUID, BaseTrainCheckerHandler> checkerWorkflowIdsToHandlers = new HashMap<>();
    private final Map<UUID, UUID> checkerWorkflowIdsToItemIds = new HashMap<>();
    private static final ZoneId UTC = ZoneId.of("UTC");

    @Autowired
    @Qualifier("trainOrderItemWorkflowEventHandler")
    WorkflowEventHandler<TrainOrderItem> trainOrderItemWorkflowEventHandler;

    @Autowired
    @Qualifier("trainInsuranceRefundWorkflowEventHandler")
    WorkflowEventHandler<TrainInsuranceRefund> trainInsuranceRefundWorkflowEventHandler;

    @Autowired
    @Qualifier("trainTicketRefundWorkflowEventHandler")
    WorkflowEventHandler<TrainTicketRefund> trainTicketRefundWorkflowEventHandler;

    public BaseTrainCheckerHandler runConfirmRefundChecker(WorkflowProcessService workflowProcessService) {
        BaseTrainCheckerHandler checkerHandler = new TrainConfirmRefundCheckerHandler(refundCalculationService);
        createAndRunChecker(checkerHandler, workflowProcessService);
        return checkerHandler;
    }

    public BaseTrainCheckerHandler runReserveCancelChecker(WorkflowProcessService workflowProcessService) {
        BaseTrainCheckerHandler checkerHandler = new TrainReserveCancelCheckerHandler();
        createAndRunChecker(checkerHandler, workflowProcessService);
        return checkerHandler;
    }

    private void createAndRunChecker(BaseTrainCheckerHandler checkerHandler, WorkflowProcessService workflowProcessService) {
        transactionTemplate.execute(ignored -> {
            UUID checkerId = UUID.randomUUID();
            Workflow checker = new Workflow();
            checker.setEntityType(TestWorkflows.CHECKER.getEntityType());
            checker.setId(checkerId);
            checker.setState(EWorkflowState.WS_RUNNING);
            workflowRepository.saveAndFlush(checker);

            TrainOrderItem orderItem = createOrderItem(checkerId);
            checkerWorkflowIdsToHandlers.put(checkerId, checkerHandler);
            workflowProcessService.scheduleEvent(orderItem.getWorkflow().getId(),
                    TReservationStart.newBuilder().build());
            return null;
        });
    }

    private TrainOrderItem createOrderItem(UUID checkerId) {
        TrainOrder order = new TrainOrder();
        order.setId(UUID.randomUUID());
        order.setPrettyId(order.getId().toString());
        order.setEmail("a@a.ru");
        order.setCurrency(ProtoCurrencyUnit.RUB);
        order.setState(ETrainOrderState.OS_NEW);
        TrainOrderItem orderItem = new TrainOrderItem();
        orderItem.setOrderWorkflowId(checkerId);
        orderItem.setState(EOrderItemState.IS_NEW);
        TrainReservation payload = loadPayloadTemplate();
        Instant departure = LocalDate.now(UTC).atTime(properties.getDepartureTime())
                .plus(properties.getDepartureDayAfterNow(), ChronoUnit.DAYS).atZone(UTC).toInstant();
        payload.getReservationRequestData().setDepartureTime(departure);
        orderItem.setReservation(payload);
        order.addOrderItem(orderItem);
        Workflow orderWorkflow = Workflow.createWorkflowForEntity(order, checkerId);
        workflowRepository.saveAndFlush(orderWorkflow);
        order = trainOrderRepository.saveAndFlush(order);
        orderItem = (TrainOrderItem) order.getOrderItems().get(0);
        Workflow itemWorkflow = Workflow.createWorkflowForEntity(orderItem, checkerId);
        workflowRepository.saveAndFlush(itemWorkflow);
        orderItem = trainOrderItemRepository.saveAndFlush(orderItem);
        checkerWorkflowIdsToItemIds.put(checkerId, orderItem.getId());
        return orderItem;
    }

    public WorkflowEventHandlerMatcher getHandlerMatcher() {
        return (workflow, event) -> {
            if (workflow.getEntityType().equals(TestWorkflows.CHECKER.getEntityType()) && checkerWorkflowIdsToHandlers.containsKey(workflow.getId())) {
                return Option.of(checkerWorkflowIdsToHandlers.get(workflow.getId()));
            } else if (workflow.getEntityType().equals(WellKnownWorkflowEntityType.TRAIN_ORDER_ITEM.getDiscriminatorValue())) {
                return Option.of(trainOrderItemWorkflowEventHandler);
            } else if (workflow.getEntityType().equals(WellKnownWorkflowEntityType.TRAIN_TICKET_REFUND.getDiscriminatorValue())) {
                return Option.of(trainTicketRefundWorkflowEventHandler);
            } else if (workflow.getEntityType().equals(WellKnownWorkflowEntityType.TRAIN_INSURANCE_REFUND.getDiscriminatorValue())) {
                return Option.of(trainInsuranceRefundWorkflowEventHandler);
            } else {
                return Option.empty();
            }
        };
    }

    public MessagingContextFactory getMessagingContextFactory() {
        return (workflow, attempt) -> {
            if (workflow.getEntityType().equals(TestWorkflows.CHECKER.getEntityType()) && checkerWorkflowIdsToItemIds.containsKey(workflow.getId())) {
                UUID orderItemId = checkerWorkflowIdsToItemIds.get(workflow.getId());
                return Option.of(new BasicMessagingContext<>(workflow.getId(),
                        trainOrderItemRepository.getOne(orderItemId), attempt, workflow.getWorkflowVersion()));
            } else if (workflow.getEntityType().equals(WellKnownWorkflowEntityType.TRAIN_ORDER_ITEM.getDiscriminatorValue())) {
                return Option.of(new BasicMessagingContext<>(workflow.getId(),
                        trainOrderItemRepository.getOne(workflow.getEntityId()), attempt, workflow.getWorkflowVersion()));
            } else if (workflow.getEntityType().equals(WellKnownWorkflowEntityType.TRAIN_TICKET_REFUND.getDiscriminatorValue())) {
                return Option.of(new BasicMessagingContext<>(workflow.getId(),
                        trainTicketRefundRepository.getOne(workflow.getEntityId()), attempt, workflow.getWorkflowVersion()));
            } else if (workflow.getEntityType().equals(WellKnownWorkflowEntityType.TRAIN_INSURANCE_REFUND.getDiscriminatorValue())) {
                return Option.of(new BasicMessagingContext<>(workflow.getId(),
                        trainInsuranceRefundRepository.getOne(workflow.getEntityId()), attempt, workflow.getWorkflowVersion()));
            } else {
                return Option.empty();
            }
        };
    }

    public boolean waitAll() {
        return checkerWorkflowIdsToHandlers.values().stream().allMatch(handler -> handler.waitTestResult(properties.getMaxTestDuration()));
    }

    public String waitAndDescribeAll() {
        return checkerWorkflowIdsToHandlers.values().stream()
                .map(handler -> handler.getClass().getSimpleName() + " -> " +
                        handler.waitTestResult(properties.getMaxTestDuration()))
                .collect(toList())
                .toString();
    }

    private TrainReservation loadPayloadTemplate() {
        try {
            String template = Resources.toString(Resources.getResource("trainPayloadTemplate.json"),
                    Charset.defaultCharset());
            TJson json = TJson.newBuilder()
                    .setValue(template)
                    .build();
            return ProtoUtils.fromTJson(json, TrainReservation.class);
        } catch (IOException ex) {
            throw new RuntimeException("Unable to load payload template");
        }
    }
}
