package ru.yandex.travel.acceptance.orders.orderitem.train;

import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainOrderUserRefund;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.train.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.train.proto.TTrainCalculateRefundReqInfo;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class TrainConfirmRefundCheckerHandler extends BaseTrainCheckerHandler {
    private final RefundCalculationService refundCalculationService;

    TrainConfirmRefundCheckerHandler(RefundCalculationService refundCalculationService) {
        super(EOrderItemState.IS_REFUNDED, new LinkedList<>(List.of(
                TServiceReserved.class, TServiceConfirmed.class, TServiceRefunded.class,
                ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded.class)));
        this.refundCalculationService = refundCalculationService;
    }

    @HandleEvent
    public void handleReserved(TServiceReserved message, MessagingContext<TrainOrderItem> messagingContext) {
        log.info("Order is reserved. Let's confirm the reservation now");
        TrainOrderItem item = messagingContext.getWorkflowEntity();
        storeActualState(item);
        checkMessageInSequence(message);
        messagingContext.scheduleExternalEvent(item.getWorkflow().getId(), TConfirmationStart.newBuilder().build());
    }

    @HandleEvent
    public void handleRefunded(TServiceRefunded message, MessagingContext<TrainOrderItem> messagingContext) {
        log.info("Order is refunded. We're done");
        storeActualState(messagingContext.getWorkflowEntity());
        checkMessageInSequence(message);
    }

    @HandleEvent
    public void handleCancelled(TServiceCancelled message, MessagingContext<TrainOrderItem> messagingContext) {
        log.info("Order is canceled.");
        storeActualState(messagingContext.getWorkflowEntity());
        checkMessageInSequence(message);
    }

    @HandleEvent
    public void handleConfirmed(TServiceConfirmed message, MessagingContext<TrainOrderItem> messagingContext) {
        log.info("Order is confirmed. Hurray! Let's refund it now");
        TrainOrderItem item = messagingContext.getWorkflowEntity();
        storeActualState(item);
        checkMessageInSequence(message);
        TTrainCalculateRefundReqInfo reqInfo = TTrainCalculateRefundReqInfo.newBuilder()
                .addAllBlankIds(item.getPayload().getPassengers().stream().map(x -> x.getTicket().getBlankId())
                        .collect(Collectors.toSet()))
                .build();
        String refundToken = refundCalculationService
                .calculateRefundForTrainItem(item, Set.copyOf(reqInfo.getBlankIdsList()), ProtoCurrencyUnit.RUB)
                .getRefundToken();
        TrainOrderUserRefund refund = TrainOrderUserRefund.createForOrder(item.getOrder());
        refund.setState(EOrderRefundState.RS_WAITING_SERVICE_REFUND);
        messagingContext.scheduleExternalEvent(item.getWorkflow().getId(), TRefundStart.newBuilder()
                .setOrderRefundId(ProtoUtils.toStringOrEmpty(refund.getId()))
                .setToken(refundToken).build());
    }
}
