package ru.yandex.travel.acceptance.orders.vault;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Dsl;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException;

@Slf4j
public class VaultClient {

    private static final Set<Integer> SUCCESSFUL_RESPONSE_CODES = ImmutableSet.of(200);

    private String host;
    private String oauthToken;
    private AsyncHttpClient httpClient;
    private ObjectMapper objectMapper;

    public VaultClient(String oauthToken, Type type) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(oauthToken), "OAuth token must be provided");
        this.host = type.getHost();
        this.oauthToken = oauthToken;
        this.httpClient = Dsl.asyncHttpClient(Dsl.config()
                .setThreadFactory(new ThreadFactoryBuilder().setDaemon(true).build())
                .setThreadPoolName("httpVaultClient"));
        this.objectMapper = new ObjectMapper();
    }

    public VersionResponse getVersion(String versionId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(versionId), "Version ID must be provided");
        RequestBuilder requestBuilder = new RequestBuilder();
        requestBuilder.setHeader("Authorization", oauthToken)
                .setUrl(host + "/1/versions/" + versionId);
        return sync(httpClient.executeRequest(requestBuilder).toCompletableFuture()
                .thenApply(r -> parseResponse(r, versionId, VersionResponse.class)));
    }

    private <T> T sync(CompletableFuture<T> future) {
        try {
            return future.get();
        } catch (InterruptedException e) {
            log.error("Vault call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new HttpApiRetryableException(e.getMessage());
        } catch (ExecutionException e) {
            if (e.getCause() == null) {
                throw new RuntimeException("No root cause for ExecutionException found", e);
            } else {

            }
            Throwable cause = e.getCause();
            if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            } else {
                throw new RuntimeException(e.getCause());
            }
        }
    }

    private <T> T parseResponse(Response response, String versionId, Class<T> resultClass) {
        if (SUCCESSFUL_RESPONSE_CODES.contains(response.getStatusCode())) {
            try {
                return objectMapper.readValue(response.getResponseBody(), resultClass);
            } catch (IOException e) {
                log.error("Unable to parse response", e);
                throw new RuntimeException("Unable to parse response", e);
            }
        } else if (response.getStatusCode() == 401) {
            throw new RuntimeException(MessageFormat.format("Not authorized access to secret version {0}", versionId));
        } else {
            throw new RuntimeException(MessageFormat.format("Do not know how to handle status {0} from vault",
                    response.getStatusCode()));
        }
    }

    @RequiredArgsConstructor
    @Getter
    public enum Type {
        PROD("https://vault-api.passport.yandex.net"),
        TESTING("https://vault-api-test.passport.yandex.net");
        private final String host;
    }
}
