package ru.yandex.travel.orders;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import com.google.common.base.Strings;
import com.google.protobuf.Message;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.javamoney.moneta.Money;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.mock.mockito.MockBean;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.TOffsetPage;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TAuthorizeForOrderReq;
import ru.yandex.travel.orders.proto.TAuthorizeForOrderRsp;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TListOrdersReq;
import ru.yandex.travel.orders.proto.TListOrdersRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TReserveRsp;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.services.partners.BillingPartnerService;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservation;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.WorkflowProcessService;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;
import static org.mockito.Mockito.verify;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class OrdersServiceTest extends AbstractGrpcTest {
    public static final String DEFAULT_TEST_EMAIL = "robot-travel-dev@yandex-team.ru";
    public static final String DEFAULT_TEST_PHONE = "+7-(495)-000-00-000";

    @Autowired
    private OrdersService ordersService;
    @MockBean
    private WorkflowProcessService workflowProcessService;

    @MockBean
    private WorkflowMessageSender workflowMessageSender;

    @MockBean
    private BillingPartnerService billingPartnerService;

    @Test
    public void testCreateOrderLoggedIn() {
        mockDefaultUser();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();

        TCreateOrderReq createOrderReq = buildTestCreateOrderReq().build();
        TCreateOrderRsp createOrderRsp = clientStub.createOrder(createOrderReq);

        assertThat(createOrderRsp.getNewOrder().getServiceCount()).isEqualTo(1);
        assertThat(createOrderRsp.getNewOrder().getService(0).getServiceInfo().getPayload().getValue()).isNotNull();
        assertThat(createOrderRsp.getNewOrder().getOwner().getYandexUid()).isEqualTo(DEFAULT_TEST_YANDEX_UID);
        assertThat(createOrderRsp.getNewOrder().getOwner().getPassportId()).isEqualTo(DEFAULT_TEST_PASSPORT_ID);
        assertThat(createOrderRsp.getNewOrder().getOwner().getLogin()).isEqualTo(DEFAULT_TEST_LOGIN);
    }

    @Test
    public void testCreateOrderLoggedOut() {
        mockDefaultUserLoggedOut();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();

        TCreateOrderReq createOrderReq = buildTestCreateOrderReq(DEFAULT_TEST_YANDEX_UID, null, null).build();
        TCreateOrderRsp createOrderRsp = clientStub.createOrder(createOrderReq);

        assertThat(createOrderRsp.getNewOrder().getServiceCount()).isEqualTo(1);
        assertThat(createOrderRsp.getNewOrder().getService(0).getServiceInfo().getPayload().getValue()).isNotNull();
        assertThat(createOrderRsp.getNewOrder().getOwner().getYandexUid()).isEqualTo(DEFAULT_TEST_YANDEX_UID);
        assertThat(createOrderRsp.getNewOrder().getOwner().getPassportId()).isEmpty();
        assertThat(createOrderRsp.getNewOrder().getOwner().getLogin()).isEmpty();
    }

    @Test
    public void testCannotCreateOrderWithMismatchedOwnerUid() {
        mockDefaultUserLoggedOut();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        TCreateOrderReq createOrderReq = buildTestCreateOrderReq("other", null, null).build();
        assertThatThrownBy(() -> clientStub.createOrder(createOrderReq))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.INVALID_ARGUMENT);
                });
    }

    @Test
    public void testCannotCreateOrderWithMismatchedPassportData() {
        // case 1: user is logged in but the order is created without the owner passport
        mockDefaultUserLoggedOut();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        TCreateOrderReq createOrderReq = buildTestCreateOrderReq().build();
        assertThatThrownBy(() -> clientStub.createOrder(createOrderReq))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.INVALID_ARGUMENT);
                });
        // case 1: user is NOT logged in but the order is created with some owner passport
        mockDefaultUser();
        TCreateOrderReq createOrderReq2 = buildTestCreateOrderReq(DEFAULT_TEST_YANDEX_UID, null, null).build();
        assertThatThrownBy(() -> clientStub.createOrder(createOrderReq2))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.INVALID_ARGUMENT);
                });
    }

    @Test
    public void testCreateOrderWithDuplicateKey() {
        mockDefaultUser();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();

        TCreateOrderReq createOrderReq = buildTestCreateOrderReq().build();
        TCreateOrderRsp createOrderRsp = clientStub.createOrder(createOrderReq);

        assertThat(createOrderRsp.getNewOrder().getServiceCount()).isEqualTo(1);
        assertThat(createOrderRsp.getNewOrder().getService(0).getServiceInfo().getPayload().getValue()).isNotNull();

        TCreateOrderRsp createOrderRsp2 = clientStub.createOrder(createOrderReq);
        assertThat(createOrderRsp).isEqualTo(createOrderRsp2);
    }

    @Test
    public void testGetOrderInfo() {
        mockDefaultUser();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();

        TCreateOrderReq createOrderReq = buildTestCreateOrderReq().setLabel("Label1").build();
        TCreateOrderRsp createOrderRsp = clientStub.createOrder(createOrderReq);

        String orderId = createOrderRsp.getNewOrder().getOrderId();
        assertThat(orderId).isNotEmpty();
        String prettyId = createOrderRsp.getNewOrder().getPrettyId();
        assertThat(prettyId).isNotEmpty();
        assertThat(createOrderRsp.getNewOrder().getLabel()).isEqualTo("Label1");

        TGetOrderInfoReq req = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp rsp = clientStub.getOrderInfo(req);

        assertThat(rsp.getResult()).isNotNull();
        assertThat(rsp.getResult().getOrderId()).isEqualTo(orderId);
        assertThat(rsp.getResult().getPrettyId()).isEqualTo(prettyId);
        assertThat(rsp.getResult().getCreatedAt()).isNotNull();
        assertThat(rsp.getResult().getUpdatedAt()).isNotNull();
        assertThat(rsp.getResult().getLabel()).isEqualTo("Label1");

        TGetOrderInfoReq req2 = TGetOrderInfoReq.newBuilder().setPrettyId(prettyId).build();
        TGetOrderInfoRsp rsp2 = clientStub.getOrderInfo(req2);

        assertThat(rsp2.getResult()).isNotNull();
        assertThat(rsp2.getResult().getOrderId()).isEqualTo(orderId);
        assertThat(rsp2.getResult().getPrettyId()).isEqualTo(prettyId);
        assertThat(rsp2.getResult().getCreatedAt()).isNotNull();
        assertThat(rsp2.getResult().getUpdatedAt()).isNotNull();
    }

    @Test
    public void testOwnerOrderAccess() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockDefaultUser();
        String orderId = clientStub.createOrder(buildTestCreateOrderReq().build()).getNewOrder().getOrderId();
        clientStub.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        mockUser("otherCookie", DEFAULT_TEST_PASSPORT_ID); // logged in as same user, but with different cookie
        clientStub.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        mockUser("otherCookie", "other passport"); // different cookie, different passport -> no access
        assertThatThrownBy(() -> clientStub.getOrderInfo((TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.PERMISSION_DENIED);
                });
    }

    @Test
    public void testAuthorizeWithNoSecretCallToGetIds() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockDefaultUser();
        TOrderInfo orderInfo = clientStub.createOrder(buildTestCreateOrderReq().build()).getNewOrder();
        TAuthorizeForOrderRsp resp =
                clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder().setOrderId(orderInfo.getOrderId()).build());
        assertThat(resp.getAuthorized()).isTrue();
        assertThat(resp.getOrderId()).isEqualTo(orderInfo.getOrderId());
        assertThat(resp.getPrettyId()).isEqualTo(orderInfo.getPrettyId());
        resp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder().setPrettyId(orderInfo.getPrettyId()).build());
        assertThat(resp.getAuthorized()).isTrue();
        assertThat(resp.getOrderId()).isEqualTo(orderInfo.getOrderId());
        assertThat(resp.getPrettyId()).isEqualTo(orderInfo.getPrettyId());
        mockAnonymousUser("other", "other");
        resp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder().setOrderId(orderInfo.getOrderId()).build());
        assertThat(resp.getAuthorized()).isFalse();
        assertThat(resp.getOrderId()).isEqualTo(orderInfo.getOrderId());
        assertThat(resp.getPrettyId()).isEqualTo(orderInfo.getPrettyId());
        resp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder().setPrettyId(orderInfo.getPrettyId()).build());
        assertThat(resp.getAuthorized()).isFalse();
        assertThat(resp.getOrderId()).isEqualTo(orderInfo.getOrderId());
        assertThat(resp.getPrettyId()).isEqualTo(orderInfo.getPrettyId());
    }

    @Test
    public void testAuthorizeByEmail() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockDefaultUser();
        TOrderInfo orderInfo = clientStub.createOrder(buildTestCreateOrderReq().build()).getNewOrder();
        mockAnonymousUser("other", "other");
        assertThatThrownBy(() -> clientStub.getOrderInfo((TGetOrderInfoReq.newBuilder().setOrderId(orderInfo.getOrderId()).build())))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.PERMISSION_DENIED);
                });
        TAuthorizeForOrderRsp resp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder()
                .setOrderId(orderInfo.getOrderId())
                .setSecret(DEFAULT_TEST_EMAIL).build());
        assertThat(resp.getAuthorized()).isTrue();
        clientStub.getOrderInfo((TGetOrderInfoReq.newBuilder().setOrderId(orderInfo.getOrderId()).build()));
    }

    @Test
    public void testAuthorizeByPhone() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockDefaultUser();
        TOrderInfo orderInfo = clientStub.createOrder(buildTestCreateOrderReq().build()).getNewOrder();
        mockAnonymousUser("other", "other");
        assertThatThrownBy(() -> clientStub.getOrderInfo((TGetOrderInfoReq.newBuilder().setOrderId(orderInfo.getOrderId()).build())))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.PERMISSION_DENIED);
                });
        TAuthorizeForOrderRsp resp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder()
                .setOrderId(orderInfo.getOrderId())
                .setSecret(DEFAULT_TEST_PHONE).build());
        assertThat(resp.getAuthorized()).isTrue();
        clientStub.getOrderInfo((TGetOrderInfoReq.newBuilder().setOrderId(orderInfo.getOrderId()).build()));
    }

    @Test
    public void testAuthorizeByFuzzyPhone() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockDefaultUser();
        TOrderInfo orderInfo = clientStub.createOrder(buildTestCreateOrderReq().build()).getNewOrder();
        mockAnonymousUser("other", "other");
        assertThatThrownBy(() -> clientStub.getOrderInfo((TGetOrderInfoReq.newBuilder().setOrderId(orderInfo.getOrderId()).build())))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.PERMISSION_DENIED);
                });
        TAuthorizeForOrderRsp resp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder()
                .setOrderId(orderInfo.getOrderId())
                .setSecret("#7( 495 ) 000-00000").build());
        assertThat(resp.getAuthorized()).isTrue();
        clientStub.getOrderInfo((TGetOrderInfoReq.newBuilder().setOrderId(orderInfo.getOrderId()).build()));
    }

    @Test
    public void testAuthorizationDoesNotAllowListing() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockDefaultUser();
        TOrderInfo orderInfo = clientStub.createOrder(buildTestCreateOrderReq().build()).getNewOrder();
        mockAnonymousUser("other", "other");
        TAuthorizeForOrderRsp resp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder()
                .setOrderId(orderInfo.getOrderId())
                .setSecret(DEFAULT_TEST_EMAIL).build());
        assertThat(resp.getAuthorized()).isTrue();
        assertThat(
                clientStub.listOrders(TListOrdersReq.newBuilder()
                        .addTypes(EDisplayOrderType.DT_HOTEL)
                        .setPage(TOffsetPage.newBuilder().setOffset(0).setLimit(10).build())
                        .build())
                        .getOrderInfoCount()).isEqualTo(0);
    }

    @Test
    public void testAuthorizationFailure() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockDefaultUser();
        TOrderInfo orderInfo = clientStub.createOrder(buildTestCreateOrderReq().build()).getNewOrder();
        mockAnonymousUser("other", "other");
        TAuthorizeForOrderRsp resp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder()
                .setOrderId(orderInfo.getOrderId())
                .setSecret("something-wrong").build());
        assertThat(resp.getAuthorized()).isFalse();
        assertThatThrownBy(() -> clientStub.getOrderInfo((TGetOrderInfoReq.newBuilder().setOrderId(orderInfo.getOrderId()).build())))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.PERMISSION_DENIED);
                });
    }

    @Test
    public void testCreateOrderAsLoggedInThanLogoutLeadsToNoAccess() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockUser("someUid", "somePassport", "someLogin");
        TOrderInfo orderInfo =
                clientStub.createOrder(buildTestCreateOrderReq("someUid", "somePassport", "someLogin").build()).getNewOrder();
        TAuthorizeForOrderRsp rsp =
                clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder().setOrderId(orderInfo.getOrderId()).build());
        assertThat(rsp.getAuthorized()).isTrue();
        mockAnonymousUser("other", "other");
        rsp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder().setOrderId(orderInfo.getOrderId()).build());
        assertThat(rsp.getAuthorized()).isFalse();
        rsp = clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder().setOrderId(orderInfo.getOrderId()).setSecret(DEFAULT_TEST_PHONE).build());
        assertThat(rsp.getAuthorized()).isTrue();
        mockUser("someUid", "otherPassport", "otherLogin");
        assertThat(rsp.getAuthorized()).isTrue();
    }

    @Test
    public void testCreateOrderAsLoggedOutThanLoginRetainsAccess() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        mockAnonymousUser("someUid", "someKey");
        TOrderInfo orderInfo =
                clientStub.createOrder(buildTestCreateOrderReq("someUid", null, null).build()).getNewOrder();
        TAuthorizeForOrderRsp rsp =
                clientStub.authorizeForOrder(TAuthorizeForOrderReq.newBuilder().setOrderId(orderInfo.getOrderId()).build());
        assertThat(rsp.getAuthorized()).isTrue();
        mockUser("someUid", "somePassport", "someLogin");
        assertThat(rsp.getAuthorized()).isTrue();
    }

    @Test
    public void testGetOrderInfoFailsOnEmptyId() {
        mockDefaultUserLoggedOut();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();

        TGetOrderInfoReq req = TGetOrderInfoReq.newBuilder().build();

        assertThatThrownBy(() -> clientStub.getOrderInfo(req))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.INVALID_ARGUMENT);
                });
    }

    @Test
    public void testGetOrderInfoNonExisting() {
        mockDefaultUserLoggedOut();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();

        TGetOrderInfoReq req = TGetOrderInfoReq.newBuilder().setOrderId("1-2-3-4-5").build();

        assertThatThrownBy(() -> clientStub.getOrderInfo(req))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.NOT_FOUND);
                });

        TGetOrderInfoReq req2 = TGetOrderInfoReq.newBuilder().setPrettyId("YA-0000-0001-0002").build();

        assertThatThrownBy(() -> clientStub.getOrderInfo(req2))
                .isInstanceOf(StatusRuntimeException.class)
                .satisfies(t -> {
                    StatusRuntimeException e = (StatusRuntimeException) t;
                    assertThat(e.getStatus().getCode()).isEqualTo(Status.Code.NOT_FOUND);
                });
    }

    @Test
    public void testReserve() {
        mockDefaultUser();
        ArgumentCaptor<UUID> workflowIdCaptor = ArgumentCaptor.forClass(UUID.class);
        ArgumentCaptor<Message> messageArgumentCaptor = ArgumentCaptor.forClass(Message.class);

        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();

        TCreateOrderReq createOrderReq = buildTestCreateOrderReq().build();
        TCreateOrderRsp createOrderRsp = clientStub.createOrder(createOrderReq);

        assertThat(createOrderRsp.getNewOrder().getOrderId()).isNotNull();

        TReserveRsp reserveRsp = clientStub.reserve(
                TReserveReq.newBuilder()
                        .setOrderId(createOrderRsp.getNewOrder().getOrderId())
                        .build());

        verify(workflowMessageSender).scheduleEvent(workflowIdCaptor.capture(), messageArgumentCaptor.capture());

        assertThat(workflowIdCaptor.getValue()).isNotNull();
        assertThat(messageArgumentCaptor.getValue()).isInstanceOf(TStartReservation.class);
    }

    @Test
    public void testReserveRetries() {
        mockDefaultUser();
        var clientStub = createServerAndBlockingStub();
        String orderId = clientStub.createOrder(buildTestCreateOrderReq().build()).getNewOrder().getOrderId();

        TReserveReq reqA = TReserveReq.newBuilder().setOrderId(orderId).setCallId("A").build();
        TReserveReq reqB = TReserveReq.newBuilder().setOrderId(orderId).setCallId("B").build();

        clientStub.reserve(reqA);

        // another reservation attempt should fail
        assertThatThrownBy(() -> clientStub.reserve(reqB))
                .isInstanceOf(StatusRuntimeException.class)
                .hasMessageContaining("Unable to start reservation: some action in progress");

        // but the retries of the original one should still work
        clientStub.reserve(reqA);
    }

    @Test
    public void testListWithPagination() {
        String ownerYandexUid = UUID.randomUUID().toString();
        String ownerPassportId = UUID.randomUUID().toString();
        mockUser(ownerYandexUid, ownerPassportId, "test");
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        List<String> orderIds = new ArrayList<>(12);
        for (int i = 0; i < 12; i++) {
            LocalDateTime servicedAt;
            if (i < 10) {
                servicedAt = LocalDateTime.now().plusDays(10);
            } else {
                servicedAt = LocalDateTime.now().minusDays(10);
            }
            TCreateOrderReq.Builder builder = buildTestCreateOrderReq(servicedAt);
            builder.getOwnerBuilder().setYandexUid(ownerYandexUid);
            builder.getOwnerBuilder().setPassportId(ownerPassportId);
            builder.getOwnerBuilder().setLogin("test");
            TCreateOrderReq createOrderReq = builder.build();
            TCreateOrderRsp createOrderRsp = clientStub.createOrder(createOrderReq);
            orderIds.add(createOrderRsp.getNewOrder().getOrderId());
        }
        TListOrdersRsp respPage1 = clientStub.listOrders(TListOrdersReq.newBuilder()
                .addTypes(EDisplayOrderType.DT_HOTEL)
                .setPage(TOffsetPage.newBuilder().setOffset(0).setLimit(10).build())
                .build());
        TListOrdersRsp respPage2 = clientStub.listOrders(TListOrdersReq.newBuilder()
                .addTypes(EDisplayOrderType.DT_HOTEL)
                .setPage(TOffsetPage.newBuilder().setOffset(10).setLimit(10).build())
                .build());
        assertThat(respPage1.getOrderInfoCount()).isEqualTo(10);
        assertThat(respPage2.getOrderInfoCount()).isEqualTo(2);
        assertThat(respPage1.getOrderInfoList())
                .extracting(TOrderInfo::getOrderId).containsAll(orderIds.subList(0, 10));
        assertThat(respPage2.getOrderInfoList()).extracting(TOrderInfo::getOrderId).containsAll(orderIds.subList(11,
                12));
    }

    @Test
    public void testListNonOwnedOrdersProducesEmptyList() {
        mockDefaultUser();
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub clientStub = createServerAndBlockingStub();
        for (int i = 0; i < 12; i++) {
            TCreateOrderReq createOrderReq = buildTestCreateOrderReq().build();
            clientStub.createOrder(createOrderReq);
        }
        mockUser("somethingElse", "someOtherKey");
        TListOrdersRsp res = clientStub.listOrders(TListOrdersReq.newBuilder()
                .addTypes(EDisplayOrderType.DT_HOTEL)
                .setPage(TOffsetPage.newBuilder().setOffset(0).setLimit(10).build())
                .build());
        assertThat(res.getOrderInfoCount()).isEqualTo(0);

        mockAnonymousUser(DEFAULT_TEST_YANDEX_UID, DEFAULT_SESSION_KEY);
        res = clientStub.listOrders(TListOrdersReq.newBuilder()
                .addTypes(EDisplayOrderType.DT_HOTEL)
                .setPage(TOffsetPage.newBuilder().setOffset(0).setLimit(10).build())
                .build());
        assertThat(res.getOrderInfoCount()).isEqualTo(0);
        mockUser("someNewUid", DEFAULT_TEST_PASSPORT_ID);
        res = clientStub.listOrders(TListOrdersReq.newBuilder()
                .addTypes(EDisplayOrderType.DT_HOTEL)
                .setPage(TOffsetPage.newBuilder().setOffset(0).setLimit(10).build())
                .build());
        assertThat(res.getOrderInfoCount()).isGreaterThan(0);
    }

    private TCreateOrderReq.Builder buildTestCreateOrderReq() {
        return buildTestCreateOrderReq(DEFAULT_TEST_YANDEX_UID, DEFAULT_TEST_PASSPORT_ID, DEFAULT_TEST_LOGIN);
    }

    private TCreateOrderReq.Builder buildTestCreateOrderReq(LocalDateTime servicedAt) {
        return buildTestCreateOrderReq(DEFAULT_TEST_YANDEX_UID, DEFAULT_TEST_PASSPORT_ID, DEFAULT_TEST_LOGIN,
                servicedAt);
    }


    private TCreateOrderReq.Builder buildTestCreateOrderReq(String ownerUid, String ownerPassport, String ownerLogin) {
        return buildTestCreateOrderReq(ownerUid, ownerPassport, ownerLogin, LocalDateTime.now().plusDays(1));
    }

    private TCreateOrderReq.Builder buildTestCreateOrderReq(String ownerUid, String ownerPassport, String ownerLogin,
                                                            LocalDateTime servicedAt) {
        HotelItinerary itinerary = new ExpediaHotelItinerary();
        Map<String, String> orderLabelParams = new HashMap<>();
        itinerary.setGeneratedAtInstant(Instant.now());
        itinerary.setRefundRules(RefundRules.builder().build());
        itinerary.setOrderDetails(OrderDetails.builder().checkinDate(servicedAt.toLocalDate()).build());
        itinerary.setFiscalPrice(Money.of(1000, ProtoCurrencyUnit.RUB));
        String deduplicationKey = UUID.randomUUID().toString();

        return TCreateOrderReq.newBuilder()
                .setDeduplicationKey(deduplicationKey)
                .setOrderType(EOrderType.OT_HOTEL_EXPEDIA)
                .setCurrency(ECurrency.C_RUB)
                .setOwner(
                        TUserInfo.newBuilder()
                                .setYandexUid(ownerUid)
                                .setPassportId(Strings.nullToEmpty(ownerPassport))
                                .setLogin(Strings.nullToEmpty(ownerLogin))
                                .setEmail(DEFAULT_TEST_EMAIL)
                                .setPhone(DEFAULT_TEST_PHONE)
                )
                .setLabelParams(ProtoUtils.toTJson(orderLabelParams))
                .addCreateServices(
                        TCreateServiceReq.newBuilder()
                                .setServiceType(EServiceType.PT_EXPEDIA_HOTEL)
                                .setSourcePayload(ProtoUtils.toTJson(itinerary))
                );
    }

    protected OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub createServerAndBlockingStub() {
        return IntegrationUtils.createServerAndBlockingStub(cleanupRule, ordersService);
    }
}
