package ru.yandex.travel.orders.configurations;

import java.io.IOException;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;
import lombok.Data;

import ru.yandex.travel.testing.local.LocalTestUtils;

/**
 * The tokens from the s3-idm api: https://wiki.yandex-team.ru/mds/s3-api/authorization/#sozdanieaccesskey
 * <p/>
 * To get one for order app buckets access use the command:
 * <pre>{@code curl -XPOST -H"Authorization: OAuth `cat  ~/.aws/s3_token`" \
 * "https://s3-idm.mds.yandex.net/credentials/create-access-key" --data "service_id=2833" \
 * --data "role=admin" > ~/.aws/access_data}</pre>
 */
@Data
@JsonNaming(PropertyNamingStrategy.UpperCamelCaseStrategy.class)
public class S3IdmAccessData {
    private String accessKeyId;
    private Integer serviceId;
    private Long userId;
    private String role;
    private String issueDate;
    private String accessSecretKey;

    // for manual local tests only
    public static S3IdmAccessData loadLocalToken() {
        String secretData = LocalTestUtils.readLocalToken(".aws/access_data");
        return fromJson(secretData);
    }

    static S3IdmAccessData fromJson(String data) {
        try {
            return new ObjectMapper().readValue(data, S3IdmAccessData.class);
        } catch (IOException e) {
            throw new RuntimeException("Failed to parse the data: " + data, e);
        }
    }
}
