package ru.yandex.travel.orders.grpc;

import java.util.List;
import java.util.Set;

import io.grpc.StatusRuntimeException;
import org.jetbrains.annotations.NotNull;
import org.junit.Test;
import org.springframework.boot.test.mock.mockito.MockBean;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.ReflectionEnhancedAbstractGrpcTest;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.cpa.CpaOrderSnapshotsServiceGrpc;
import ru.yandex.travel.orders.cpa.TBoyOrderSnapshot;
import ru.yandex.travel.orders.cpa.TBoyOrdersReq;
import ru.yandex.travel.orders.cpa.TBoyOrdersRsp;
import ru.yandex.travel.orders.cpa.TListSnapshotsReqV2;
import ru.yandex.travel.orders.cpa.TListSnapshotsRspV2;
import ru.yandex.travel.orders.cpa.TOrderSnapshot;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.repository.AuthorizedUserRepository;
import ru.yandex.travel.orders.repository.cpa.CpaAeroflotOrderRepository;
import ru.yandex.travel.orders.repository.cpa.CpaHotelOrderRepository;
import ru.yandex.travel.orders.repository.cpa.CpaOrderRepository;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.orders.test.TestOrderFactory.createAeroflotOrder;
import static ru.yandex.travel.orders.test.TestOrderFactory.createGenericSuburbanOrder;
import static ru.yandex.travel.orders.test.TestOrderFactory.createGenericTrainOrder;
import static ru.yandex.travel.orders.test.TestOrderFactory.createHotelOrder;

public class CpaOrderSnapshotServiceTest
        extends
        ReflectionEnhancedAbstractGrpcTest<CpaOrderSnapshotService,
                CpaOrderSnapshotsServiceGrpc.CpaOrderSnapshotsServiceBlockingStub> {
    @MockBean
    private CpaOrderRepository cpaOrderRepository;
    @MockBean
    private CpaHotelOrderRepository cpaHotelOrderRepository;
    @MockBean
    private CpaAeroflotOrderRepository cpaAeroflotOrderRepository;
    @MockBean
    private AuthorizedUserRepository authorizedUserRepository;

    @Test(expected = StatusRuntimeException.class)
    public void errorOccursForUnsupportedOrderTypeSnapshots() {
        getSnapshotsFor(EOrderType.OT_UNKNOWN, EServiceType.PT_UNKNOWN);
    }

    @Test
    public void getBoyOrdersForGenericTrainOrder() {
        GenericOrder order1 = createGenericTrainOrder();

        when(cpaOrderRepository.findOrdersByUpdatedAt(any(), any(), any(), any()))
                .thenReturn(List.of(order1));

        TBoyOrdersRsp response = client.getBoyOrders(TBoyOrdersReq.newBuilder()
                .setUpdatedAtFrom(ProtoUtils.timestamp())
                .setMaxSnapshots(10)
                .build());

        assertFalse(response.getHasMore());
        assertEquals(1, response.getOrderSnapshotsCount());
        TBoyOrderSnapshot snapshot1 = response.getOrderSnapshots(0);
        assertEquals(order1.getDisplayType(), snapshot1.getDisplayType());
        assertEquals(order1.getId().toString(), snapshot1.getTravelOrderId());
    }

    @Test
    public void getBoyOrdersWithNoData() {
        when(cpaOrderRepository.findOrdersByUpdatedAt(any(), any(), any(), any()))
                .thenReturn(List.of());

        TBoyOrdersRsp response = client.getBoyOrders(TBoyOrdersReq.newBuilder()
                .setUpdatedAtFrom(ProtoUtils.timestamp())
                .setMaxSnapshots(10)
                .build());

        assertFalse(response.getHasMore());
        assertEquals(0, response.getOrderSnapshotsCount());
        verify(cpaOrderRepository).findOrdersByUpdatedAt(any(), any(),
                eq(Set.of(EDisplayOrderType.DT_TRAIN, EDisplayOrderType.DT_SUBURBAN, EDisplayOrderType.DT_BUS)),
                any()
        );
    }

    @Test
    public void getBoyOrdersForGenericSuburbanOrder() {
        GenericOrder order1 = createGenericSuburbanOrder();

        when(cpaOrderRepository.findOrdersByUpdatedAt(any(), any(), any(), any()))
                .thenReturn(List.of(order1));

        TBoyOrdersRsp response = client.getBoyOrders(TBoyOrdersReq.newBuilder()
                .setUpdatedAtFrom(ProtoUtils.timestamp())
                .setMaxSnapshots(10)
                .build());

        assertFalse(response.getHasMore());
        assertEquals(1, response.getOrderSnapshotsCount());
        TBoyOrderSnapshot snapshot1 = response.getOrderSnapshots(0);
        assertEquals(order1.getDisplayType(), snapshot1.getDisplayType());
        assertEquals(order1.getId().toString(), snapshot1.getTravelOrderId());
    }

    @Test
    public void testGetSnapshotsForHotels() {
        HotelOrder order1 = createHotelOrder();

        when(cpaHotelOrderRepository.findOrdersByItemTypeAndUpdatedAtAndStateIn(
                any(), any(),
                any(), any(), any()
        )).thenReturn(List.of(order1));

        TListSnapshotsRspV2 resp = getSnapshotsFor(EOrderType.OT_HOTEL_EXPEDIA, EServiceType.PT_EXPEDIA_HOTEL);

        TOrderSnapshot snapshot1 = theOnlySnapshotinResponse(resp);
        assertEquals(snapshot1.getTravelOrderId(), order1.getId().toString());

        HotelOrderItem orderItem = (HotelOrderItem) order1.getOrderItems().get(0);
        Integer orderUserPlusBalance = orderItem.getHotelItinerary().getUserInfo().getPlusBalance();
        Integer snapshotUserPlusBalance = snapshot1.getYandexPlusCpaInfo().getUserBalance().getValue();
        assertEquals(snapshotUserPlusBalance, orderUserPlusBalance);
    }

    @NotNull
    private TOrderSnapshot theOnlySnapshotinResponse(TListSnapshotsRspV2 resp) {
        assertFalse(resp.getHasMore());
        assertEquals(1, resp.getOrderSnapshotsCount());
        return resp.getOrderSnapshots(0);
    }

    @Test
    public void testGetSnapshotsForAvia() {
        var order1 = createAeroflotOrder();

        when(cpaAeroflotOrderRepository.findByUpdatedAtAfterAndUpdatedAtBeforeAndStateInOrderByUpdatedAtAsc(
                any(),
                any(),
                any(),
                any()))
                .thenReturn(List.of(order1));

        TListSnapshotsRspV2 resp = getSnapshotsFor(EOrderType.OT_AVIA_AEROFLOT, EServiceType.PT_FLIGHT);

        assertFalse(resp.getHasMore());
    }

    @Test
    public void testGetSnapshotsForTrain() {
        var order1 = createGenericTrainOrder();

        when(cpaOrderRepository.findOrdersByUpdatedAt(any(), any(), any(), any()))
                .thenReturn(List.of(order1));
        when(authorizedUserRepository.findByIdOrderIdInAndRole(any(), any()))
                .thenReturn(List.of());

        TListSnapshotsRspV2 resp = getSnapshotsFor(EOrderType.OT_TRAIN, EServiceType.PT_TRAIN);

        TOrderSnapshot snapshot1 = theOnlySnapshotinResponse(resp);
        assertEquals(snapshot1.getTravelOrderId(), order1.getId().toString());
    }

    @Test
    public void testGetSnapshotsForSuburban() {
        var order1 = createGenericSuburbanOrder();

        when(cpaOrderRepository.findOrdersByUpdatedAt(any(), any(), any(), any()))
                .thenReturn(List.of(order1));

        TListSnapshotsRspV2 resp = getSnapshotsFor(EOrderType.OT_GENERIC, EServiceType.PT_SUBURBAN);

        TOrderSnapshot snapshot1 = theOnlySnapshotinResponse(resp);
        assertEquals(snapshot1.getTravelOrderId(), order1.getId().toString());
    }

    private TListSnapshotsRspV2 getSnapshotsFor(EOrderType orderType, EServiceType serviceType) {
        var builder = TListSnapshotsReqV2.newBuilder()
                .setUpdatedAtFrom(ProtoUtils.timestamp())
                .setOrderType(orderType)
                .setMaxSnapshots(10);

        if (serviceType != null) {
            builder.setServiceType(serviceType);
        }

        return client.getSnapshotsV2(builder.build());
    }

}
