package ru.yandex.travel.orders.grpc;

import java.time.LocalDate;
import java.util.UUID;

import org.junit.Before;
import org.junit.Test;
import org.springframework.boot.test.mock.mockito.MockBean;

import ru.yandex.travel.orders.ReflectionEnhancedAbstractGrpcTest;
import ru.yandex.travel.orders.commons.proto.EPromoCodeNominalType;
import ru.yandex.travel.orders.entities.promo.DiscountApplicationConfig;
import ru.yandex.travel.orders.entities.promo.PromoAction;
import ru.yandex.travel.orders.entities.promo.SimplePromoCodeGenerationConfig;
import ru.yandex.travel.orders.entities.promo.ValidTillGenerationType;
import ru.yandex.travel.orders.repository.promo.PromoActionRepository;
import ru.yandex.travel.orders.services.promo.proto.PromoCodesOperatorManagementInterfaceV1Grpc;
import ru.yandex.travel.orders.services.promo.proto.TGetPromoActionReq;
import ru.yandex.travel.orders.services.promo.proto.TGetPromoActionResp;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;

public class PromoCodesOperatorManagementServiceTest
        extends ReflectionEnhancedAbstractGrpcTest<PromoCodesOperatorManagementService,
                PromoCodesOperatorManagementInterfaceV1Grpc.PromoCodesOperatorManagementInterfaceV1BlockingStub> {

    @MockBean
    private PromoActionRepository repo;

    private PromoAction action;

    @Before
    public void setUp() {
        action = new PromoAction();
        action.setName("Test " + UUID.randomUUID());

        when(repo.getOne(any())).thenReturn(action);
    }

    @Test
    public void getPromoActionDetailsReturnsTheActionFromRepository() {
        TGetPromoActionResp proto = service.getPromoActionDetails(UUID.randomUUID());

        assertThat(proto.getActionName(), equalTo(action.getName()));
    }

    @Test
    public void getPromoActionDetailWithRealSetup() {
        DiscountApplicationConfig discountApplicationConfig = new DiscountApplicationConfig();
        discountApplicationConfig.setMaxConfirmedHotelOrders(1);
        action.setDiscountApplicationConfig(discountApplicationConfig);

        SimplePromoCodeGenerationConfig generationConfig = new SimplePromoCodeGenerationConfig();
        generationConfig.setPrefix("YA");
        generationConfig.setSuffix("SUB");
        generationConfig.setNominal(10.0);
        generationConfig.setFixedDate(LocalDate.now());
        generationConfig.setNominalType(EPromoCodeNominalType.NT_PERCENT);
        generationConfig.setMaxActivations(1);
        generationConfig.setMaxUsagePerUser(1);
        generationConfig.setValidTillGenerationType(ValidTillGenerationType.FIXED_DATE);
        action.setPromoCodeGenerationConfig(generationConfig);

        TGetPromoActionResp proto = service.getPromoActionDetails(UUID.randomUUID());

        assertThat(proto.getActionName(), equalTo(action.getName()));
    }

    @Test
    public void getPromoActionDetailsRequestsRepositoryIfCacheDoesNotExist() {
        var uuid = UUID.randomUUID();
        TGetPromoActionResp proto = client.getPromoActionDetails(
                TGetPromoActionReq.newBuilder()
                        .setPromoActionId(uuid.toString())
                        .build()
        );

        assertThat(proto.getActionName(), equalTo(action.getName()));
        verify(repo, times(1)).getOne(uuid);
        verifyNoMoreInteractions(repo);
    }

    @Test
    public void getPromoActionDetailsRequestsUsesCache() {
        var uuid = UUID.randomUUID();

        assertNotNull(client.getPromoActionDetails(
                TGetPromoActionReq.newBuilder()
                        .setPromoActionId(uuid.toString())
                        .build()
                )
        );
        assertNotNull(client.getPromoActionDetails(
                TGetPromoActionReq.newBuilder()
                        .setPromoActionId(uuid.toString())
                        .build()
                )
        );

        verify(repo, times(1)).getOne(uuid);
        verifyNoMoreInteractions(repo);
    }
}
