package ru.yandex.travel.orders.integration;

import java.time.Duration;
import java.util.function.BooleanSupplier;
import java.util.function.Predicate;

import io.grpc.BindableService;
import io.grpc.testing.GrpcCleanupRule;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.admin.proto.OrdersAdminInterfaceV1Grpc;
import ru.yandex.travel.orders.admin.proto.OrdersAdminInterfaceV1Grpc.OrdersAdminInterfaceV1BlockingStub;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub;
import ru.yandex.travel.orders.proto.OrderNoAuthInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.OrderNoAuthInterfaceV1Grpc.OrderNoAuthInterfaceV1BlockingStub;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.takeout.proto.TFindJobReq;
import ru.yandex.travel.orders.takeout.proto.TFindJobRsp;
import ru.yandex.travel.orders.takeout.proto.TakeoutOrdersInterfaceV1Grpc.TakeoutOrdersInterfaceV1BlockingStub;
import ru.yandex.travel.testing.TestUtils;

import static org.assertj.core.api.Assertions.assertThat;

@Slf4j
public class IntegrationUtils {
    // legacy method name, actually means "createOrderServiceBlockingGrpcClient" but there are too many usages to rename
    public static OrderInterfaceV1BlockingStub createServerAndBlockingStub(GrpcCleanupRule cleanupRule,
                                                                           BindableService service) {
        TestGrpcContext context = TestGrpcContext.createTestServer(cleanupRule, service);
        return OrderInterfaceV1Grpc.newBlockingStub(context.createChannel());
    }

    public static OrderNoAuthInterfaceV1BlockingStub createOrderNoAuthServiceAndStub(GrpcCleanupRule cleanupRule,
                                                                                     BindableService service) {
        TestGrpcContext context = TestGrpcContext.createTestServer(cleanupRule, service);
        return OrderNoAuthInterfaceV1Grpc.newBlockingStub(context.createChannel());
    }

    public static OrdersAdminInterfaceV1BlockingStub createAdminServerAndBlockingStub(GrpcCleanupRule cleanupRule,
                                                                                      BindableService service) {
        TestGrpcContext context = TestGrpcContext.createTestServer(cleanupRule, service);
        return OrdersAdminInterfaceV1Grpc.newBlockingStub(context.createChannel());
    }

    public static TGetOrderInfoRsp waitForPredicateOrTimeout(OrderInterfaceV1BlockingStub client, String orderId,
                                                             Predicate<TGetOrderInfoRsp> predicate,
                                                             String description) {
        return waitForPredicateOrTimeout(client, orderId, predicate, TestUtils.DEFAULT_WAIT_TIMEOUT, description);
    }

    @SuppressWarnings("BusyWait")
    public static TGetOrderInfoRsp waitForPredicateOrTimeout(OrderInterfaceV1BlockingStub client, String orderId,
                                                             Predicate<TGetOrderInfoRsp> predicate, Duration timeout,
                                                             String description) {
        TGetOrderInfoRsp rsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        long startWaitingTime = System.currentTimeMillis();
        while ((System.currentTimeMillis() - startWaitingTime) < timeout.toMillis() && !predicate.test(rsp)) {
            try {
                Thread.sleep(200);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt(); // preserving the status
                throw new RuntimeException(e);
            }
            rsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        }
        log.info("waitForPredicateOrTimeout has completed in {} ms with the {} result, check: {}",
                System.currentTimeMillis() - startWaitingTime, predicate.test(rsp) ? "OK" : "MISMATCH", description);
        assertThat(rsp)
                // preventing the huge trimmed rsp output mask description
                .withFailMessage(description + "; rsp " + rsp)
                .matches(predicate, description);
        return rsp;
    }

    @SuppressWarnings("BusyWait")
    public static TFindJobRsp waitForPredicateOrTimeout(TakeoutOrdersInterfaceV1BlockingStub client,
                                                        TFindJobReq req,
                                                        Predicate<TFindJobRsp> predicate, Duration timeout,
                                                        String description) throws InterruptedException {
        TFindJobRsp rsp = client.findJob(req);
        long startWaitingTime = System.currentTimeMillis();
        while ((System.currentTimeMillis() - startWaitingTime) < timeout.toMillis() && !predicate.test(rsp)) {
            Thread.sleep(200);
            rsp = client.findJob(req);
        }
        log.info("waitForPredicateOrTimeout(takeout) has completed in {} ms with the {} result, check: {}",
                System.currentTimeMillis() - startWaitingTime, predicate.test(rsp) ? "OK" : "MISMATCH", description);
        assertThat(rsp)
                // preventing the huge trimmed rsp output mask description
                .withFailMessage(description + "; rsp " + rsp)
                .matches(predicate, description);
        return rsp;
    }

    public static void waitForPredicateOrTimeout(BooleanSupplier predicate, Duration timeout, String description) {
        waitForPredicateOrTimeout(predicate, timeout, Duration.ofMillis(200), description);
    }

    public static void waitForPredicateOrTimeout(BooleanSupplier predicate, Duration timeout, Duration retryDelay,
                                                 String description) {
        TestUtils.waitForState(description, timeout, retryDelay, predicate);
    }
}
