package ru.yandex.travel.orders.integration.aeroflot;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import lombok.extern.slf4j.Slf4j;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqData;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqParser;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqRawData;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqReader;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustBindingToken;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceAdapter;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceProvider;
import ru.yandex.travel.workflow.WorkflowEventQueue;
import ru.yandex.travel.workflow.WorkflowProcessService;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.orders.integration.IntegrationUtils.createServerAndBlockingStub;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.SESSION_KEY;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.YANDEX_UID;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.aeroflotOrderCreateResultSuccessWith3ds;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.createOrderRequest;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.mockTrustCalls;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.parsePayload;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "aeroflot-mq.enabled=true",
                "aeroflot-mq.repeat-interval=100ms",
                "quartz.enabled=true",
                "aeroflot-workflow.invoice-confirmation-timeout=0ms",
                "single-node.auto-start=true"
        }
)
@ActiveProfiles("test")
@DirtiesContext
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@Slf4j
@SuppressWarnings({"ResultOfMethodCallIgnored"})
public class AeroflotOrder3dsConfirmationTimedOutTest {

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();

    @Autowired
    private OrdersService ordersService;

    private final UserCredentialsBuilder userCredentialsBuilder = new UserCredentialsBuilder();

    @Autowired
    private TrustClient aeroflotTrustClient;

    @MockBean
    private AeroflotServiceProvider aeroflotServiceProvider;

    @MockBean
    private AeroflotServiceAdapter aeroflotProviderAdapter;

    //    @MockBean
//    private AviaApiProxy aviaApiProxy;
//
    @MockBean
    private AeroflotMqReader aeroflotMqReader;

    @MockBean
    private AeroflotMqParser aeroflotMqParser;

    @Autowired
    private WorkflowProcessService workflowProcessService;

    @Autowired
    private WorkflowEventQueue workflowEventQueue;

    @Autowired
    private TransactionTemplate txTemplate;

    private Context context;

    @Before
    public void init() {
        UserCredentials credentials = userCredentialsBuilder.build(SESSION_KEY, YANDEX_UID, null, null, null, "127.0" +
                ".0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();

        Mockito.clearInvocations(aeroflotTrustClient);

        when(aeroflotMqReader.readOrders()).thenReturn(Collections.emptyList());
        when(aeroflotServiceProvider.getAeroflotServiceForProfile(any())).thenReturn(aeroflotProviderAdapter);
    }

    @After
    public void destroy() {
        Context.current().detach(context);
    }

    @Test
    public void testOrder3dsConfirmationTimedOut() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
        when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                .thenReturn(aeroflotOrderCreateResultSuccessWith3ds("pnr_3ds_to"));
        when(aeroflotTrustClient.getBasketStatus(any(), any())).thenReturn(TrustBasketStatusResponse.builder()
                .paymentStatus(PaymentStatusEnum.CLEARED)
                .bindingToken(TrustBindingToken.builder().expiration(LocalDateTime.now()).value("tokenized_card").build())
                .build());
        when(aeroflotProviderAdapter.getOrderStatus(any(), any())).thenReturn(TestHelpers.incompletePaymentResult());

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order item must be in IS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_TIMED_OUT);
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
    }

    @Test
    public void testOrder3dsConfirmationTimedOutButExternallyConfirmed() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        String pnr = "pnr_ERR_EXT_CONF";
        mockTrustCalls(aeroflotTrustClient);
        when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
        when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                .thenReturn(aeroflotOrderCreateResultSuccessWith3ds(pnr));
        when(aeroflotTrustClient.getBasketStatus(any(), any())).thenReturn(TrustBasketStatusResponse.builder()
                .paymentStatus(PaymentStatusEnum.CLEARED)
                .bindingToken(TrustBindingToken.builder().expiration(LocalDateTime.now()).value("tokenized_card").build())
                .build());
        when(aeroflotProviderAdapter.getOrderStatus(any(), any())).thenReturn(TestHelpers.incompletePaymentResult());

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order item must be in IS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_TIMED_OUT);
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);

        when(aeroflotMqReader.readOrders())
                // emulating one Aeroflot MQ message
                .thenReturn(List.of(AeroflotMqRawData.builder().data("sync_msg_1").build()))
                .thenReturn(Collections.emptyList());
        when(aeroflotMqParser.parseMessage(eq("sync_msg_1")))
                .thenReturn(AeroflotMqData.builder()
                        .bookingDate(ZonedDateTime.parse("2019-05-22T10:42:30+00:00"))
                        .pnr(pnr)
                        .tickets(List.of("ticket_123"))
                        .passengers(List.of(AeroflotMqData.PassengerRef.builder().lastName("TDSRESTOROV").build()))
                        .build());

        orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be restored and confirmed");
        assertThat(orderInfoRsp.getResult().getService(0).getServiceInfo().getAeroflotItemState() == EAeroflotItemState.IS_CONFIRMED);
        assertThat(orderInfoRsp.getResult().getInvoice(0).getAeroflotInvoiceState() == EAeroflotInvoiceState.IS_CONFIRMED);
        assertThat(parsePayload(orderInfoRsp.getResult()).getTickets()).containsEntry("1", List.of("ticket_123"));
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient aeroflotTrustClient() {
            return mock(TrustClient.class);
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient aeroflotTrustClient) {
            return paymentProfile -> aeroflotTrustClient;
        }
    }
}
