package ru.yandex.travel.orders.integration.aeroflot;

import java.time.Duration;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.dao.ConcurrencyFailureException;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotPaymentException;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotCategoryOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderCreateResult;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderStatus;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotPriceDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTotalOffer;
import ru.yandex.avia.booking.partners.gateways.model.availability.PriceChangedException;
import ru.yandex.avia.booking.partners.gateways.model.availability.VariantNotAvailableException;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureException;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureReason;
import ru.yandex.avia.booking.partners.gateways.model.payment.PaymentFailureReason;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonApiClient;
import ru.yandex.travel.commons.http.apiclient.HttpApiException;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TServiceInfo;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.repository.AeroflotOrderItemRepository;
import ru.yandex.travel.orders.services.avia.AviaApiProxy;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqData;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqParser;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqRawData;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqReader;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.promo.aeroflotplus.AeroflotPlusPromoService;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCardTokenized;
import ru.yandex.travel.orders.workflows.order.aeroflot.AeroflotWorkflowService;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.configuration.AeroflotWorkflowProperties;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers.AeroflotOrderItemHandlersHelper;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers.AeroflotOrderItemWaitTokenizationStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotOfferPriceChangedException;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotService;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceProvider;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.testing.mockito.ThreadSafeMock;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowEventRetryStrategy;
import ru.yandex.travel.workflow.WorkflowProcessService;
import ru.yandex.travel.workflow.exceptions.RetryableException;
import ru.yandex.travel.workflow.exceptions.TooManyRequestsRetryableException;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.orders.integration.IntegrationUtils.createServerAndBlockingStub;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.SESSION_KEY;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.YANDEX_UID;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.aeroflotOrderCreateResultSuccess;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.aeroflotOrderCreateResultSuccessWith3ds;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.createOrderRequest;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.mockTrustCalls;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.parsePayload;
import static ru.yandex.travel.testing.mockito.ThreadSafeMockBuilder.newThreadSafeMockBuilder;
import static ru.yandex.travel.testing.spring.SpringUtils.unwrapAopProxy;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=10ms",
                "aeroflot-mq.enabled=true",
                "aeroflot-mq.repeat-interval=10ms",
                "quartz.enabled=true",
                "aeroflot-workflow.invoice-trust-refresh-timeout=10ms",
                "aeroflot-workflow.invoice-confirmation-refresh-timeout=10ms",
                "aeroflot-workflow.invoice-awaiting-tokenization-refresh-rate=10ms",
                "aeroflot-workflow.invoice-awaiting-confirmation-refresh-rate=10ms",
                "single-node.auto-start=true"
        }
)
@ActiveProfiles("test")
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@DirtiesContext
@Slf4j
@SuppressWarnings({"ResultOfMethodCallIgnored"})
public class AeroflotOrderIntegrationTest {

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();

    @Rule
    public TestName name = new TestName();

    private static final AtomicInteger activeTests = new AtomicInteger(0);

    @Autowired
    private OrdersService ordersService;

    @Autowired
    private TrustClient aeroflotTrustClient;

    @Autowired
    private ThreadSafeMock<AeroflotService> aeroflotServiceSafeMock;

    @SpyBean
    private AeroflotWorkflowProperties properties;

    @MockBean
    private AviaApiProxy aviaApiProxy;

    @MockBean
    private AeroflotMqReader aeroflotMqReader;

    @MockBean
    private AeroflotMqParser aeroflotMqParser;

    @Autowired
    private WorkflowProcessService workflowProcessService;

    @SpyBean
    private WorkflowEventRetryStrategy retryStrategy;

    @SpyBean
    private AeroflotOrderItemWaitTokenizationStateHandler oiWaitTokenizationStateHandler;

    @MockBean
    private AeroflotOrderItemHandlersHelper orderItemHandlersWfHelper;

    @MockBean
    private AviaTicketDaemonApiClient aviaTdApiClient;

    @MockBean
    private StarTrekService starTrekService;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private AeroflotOrderItemRepository aeroflotOrderItemRepository;

    @MockBean
    private AeroflotWorkflowService aeroflotWorkflowService;

    @MockBean
    private AeroflotPlusPromoService aeroflotPlusPromoService;

    private Context context;

    @Before
    public void init() {
        log.info("Starting the {} test, activeTests={}", name.getMethodName(), activeTests.incrementAndGet());
        if (activeTests.get() > 1) {
            throw new IllegalStateException("The integration tests shouldn't run in parallel");
        }
        UserCredentials credentials = new UserCredentialsBuilder()
                .build(SESSION_KEY, YANDEX_UID, null, null, null, "127.0.0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();

        Mockito.clearInvocations(aeroflotTrustClient);

        // for some reason the default automatic mocks reset doesn't work on this proxy (@SpyBean(reset = AFTER))
        Mockito.reset(unwrapAopProxy(properties));

        when(aeroflotMqReader.readOrders()).thenReturn(Collections.emptyList());

        // we have to use a new thread safe instance every time we try to override any instance's method behaviour
        // (Mockito's ongoing stubbing chain fails if any other object invocation
        // is concurrently made from another thread, e.g. checkPaid from our task processor)
        aeroflotServiceSafeMock.resetToDefaultMocks();
    }

    @After
    public void destroy() {
        log.info("Finishing the {} test, activeTests={}", name.getMethodName(), activeTests.decrementAndGet());
        Context.current().detach(context);
    }

    @Test
    public void testHappyPath() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        String pnr = "pnr_HP";
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenReturn(aeroflotOrderCreateResultSuccessWith3ds(null));
            when(aeroflotProviderAdapter.getOrderStatus(any(), any())).thenReturn(
                    TestHelpers.incompletePaymentResult(),
                    // delayed PNR imitation
                    TestHelpers.successfulPaymentResult(null),
                    TestHelpers.successfulPaymentResult(null),
                    TestHelpers.successfulPaymentResult(pnr));
        });

        // create
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build(); // we'll
        // reuse it
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getAeroflotOrderState()).isEqualTo(EAeroflotOrderState.OS_NEW);

        // reservation
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state"
        );

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CONFIRMED,
                Duration.ofSeconds(15), "Order must be in OS_CONFIRMED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_CONFIRMED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState()).isEqualTo(EAeroflotInvoiceState.IS_CONFIRMED);
        assertThat(parsePayload(finalOrder).getBookingRef().getPnr()).isEqualTo(pnr);

        when(aeroflotMqReader.readOrders())
                // emulating one Aeroflot MQ message
                .thenReturn(List.of(AeroflotMqRawData.builder().data("sync_msg_1").build()))
                .thenReturn(Collections.emptyList());
        when(aeroflotMqParser.parseMessage(eq("sync_msg_1")))
                .thenReturn(AeroflotMqData.builder()
                        .bookingDate(ZonedDateTime.parse("2019-05-22T10:42:30+00:00"))
                        .pnr(pnr)
                        .tickets(List.of("ticket_123"))
                        .passengers(List.of(AeroflotMqData.PassengerRef.builder().lastName("HPRESTOROV").build()))
                        .build());

        orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> parsePayload(rsp.getResult()).getTickets() != null,
                Duration.ofSeconds(10), "Order item must receive ticket numbers from the sync event");
        assertThat(parsePayload(orderInfoRsp.getResult()).getTickets().get("1")).isEqualTo(List.of("ticket_123"));

        // checking actual http calls
        verify(aeroflotServiceSafeMock.getCurrentMocksHolder(), times(1))
                .createOrderAndStartPayment(any(), any(), any(), any());
        verify(aeroflotServiceSafeMock.getCurrentMocksHolder(), times(4))
                .getOrderStatus(eq(UUID.fromString(orderId)), any());
        verify(aeroflotMqParser, times(1)).parseMessage(any());
        verify(aeroflotWorkflowService, times(1))
                .issueHotelPromoCodeOnConfirmation(argThat(o -> o.getId().toString().equals(orderId)));
        verify(aeroflotPlusPromoService, times(1))
                .registerConfirmedOrder(argThat(o -> o.getId().toString().equals(orderId)));
    }

    @Test
    public void testHappyPathNo3ds() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            AeroflotOrderCreateResult createdAndPaidResult = aeroflotOrderCreateResultSuccessWith3ds("pnr_no_3ds");
            createdAndPaidResult.setStatusCode(AeroflotOrderStatus.PAID_TICKETED);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenReturn(createdAndPaidResult);
        });

        // create
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build(); // we'll
        // reuse it
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getAeroflotOrderState()).isEqualTo(EAeroflotOrderState.OS_NEW);

        // reservation
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state"
        );

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CONFIRMED,
                Duration.ofSeconds(15), "Order must be in OS_CONFIRMED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_CONFIRMED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState()).isEqualTo(EAeroflotInvoiceState.IS_CONFIRMED);

        // checking actual http calls
        verify(aeroflotServiceSafeMock.getCurrentMocksHolder(), times(1))
                .createOrderAndStartPayment(any(), any(), any(), any());
        // no extra status checks should happen without the 3ds confirmation stage
        verify(aeroflotServiceSafeMock.getCurrentMocksHolder(), times(0))
                .getOrderStatus(eq(UUID.fromString(orderId)), any());
    }

    @Test
    public void testHappyPathNo3dsDelayedPnr() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            AeroflotOrderCreateResult paidNoPnrResult = aeroflotOrderCreateResultSuccess(null);
            AeroflotOrderCreateResult paidWithPnrResult = aeroflotOrderCreateResultSuccess("delayed_pnr");
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenReturn(paidNoPnrResult);
            when(aeroflotProviderAdapter.getOrderStatus(any(), any()))
                    .thenReturn(paidNoPnrResult, paidNoPnrResult, paidWithPnrResult);
        });

        // create
        String orderId = client.createOrder(createOrderRequest()).getNewOrder().getOrderId();

        // reservation
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "The order must be in the OS_WAIT_CARD_TOKENIZED state"
        );

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CONFIRMED,
                Duration.ofSeconds(15), "The order must be in the OS_CONFIRMED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_CONFIRMED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState()).isEqualTo(EAeroflotInvoiceState.IS_CONFIRMED);
        assertThat(parsePayload(finalOrder).getBookingRef().getPnr()).isEqualTo("delayed_pnr");

        // checking actual http calls
        verify(aeroflotServiceSafeMock.getCurrentMocksHolder(), times(1))
                .createOrderAndStartPayment(any(), any(), any(), any());
        // extra status checks until we get the pnr
        verify(aeroflotServiceSafeMock.getCurrentMocksHolder(), times(3))
                .getOrderStatus(eq(UUID.fromString(orderId)), any());
    }

    @Test
    public void testNotAvailableDuringAvailabilityCheck() {
        String testVariantId = "variant_testNotAvailableDuringAvailabilityCheck";
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter ->
                when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(false));
        doThrow(new RetryableException("api isn't available"))
                .doNothing().when(aviaApiProxy).refreshVariant(eq(testVariantId));

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest(testVariantId));
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(8),
                "Order must be in OS_CANCELLED state"
        );
        assertThat(orderInfoRsp.getResult().getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(parsePayload(orderInfoRsp.getResult()).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.NOT_AVAILABLE);
        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(0);

        verify(aviaApiProxy, times(2)).refreshVariant(eq(testVariantId));
        verify(aviaTdApiClient, times(1)).invalidateVariant(any(), any());
    }

    @Test
    public void testNotAvailableAtAllDuringAvailabilityCheck() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter ->
                when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(false));
        doThrow(new HttpApiException("not offers at all", 404, null)).doNothing().when(aviaApiProxy).refreshVariant(any());

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(5),
                "Order must be in OS_CANCELLED state"
        );
        assertThat(orderInfoRsp.getResult().getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(parsePayload(orderInfoRsp.getResult()).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.NOT_AVAILABLE);
        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(0);

        verify(aviaApiProxy, times(1)).refreshVariant(any());
        verify(aviaTdApiClient, times(1)).invalidateVariant(any(), any());
    }

    @Test
    public void testPriceChangedDuringAvailabilityCheck() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        AeroflotTotalOffer newOffer = AeroflotTotalOffer.builder()
                .totalPrice(Money.of(100, "RUB"))
                .categoryOffers(List.of(
                        AeroflotCategoryOffer.builder()
                                .travellerId("SH1")
                                .totalPrice(AeroflotPriceDetail.builder()
                                        .totalPrice(Money.of(95, "RUB"))
                                        .basePrice(Money.of(50, "RUB"))
                                        .taxes(Money.of(45, "RUB"))
                                        .build())
                                .build()
                ))
                .build();
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter ->
                when(aeroflotProviderAdapter.checkAvailability(any(), any(), any()))
                        .thenThrow(new AeroflotOfferPriceChangedException("Test price changed", newOffer)));

        TCreateOrderReq createOrderReq = createOrderRequest();

        // create
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        String orderId = resp.getNewOrder().getOrderId();
        AeroflotTotalOffer srcOffer = parsePayload(resp.getNewOrder()).getVariant().getOffer();
        assertThat(srcOffer.getTotalPrice()).isNotEqualTo(Money.of(100, "RUB"));
        AeroflotPriceDetail srcCatOffer = srcOffer.getCategoryOfferFor("SH1").getTotalPrice();
        assertThat(srcCatOffer.getTotalPrice()).isNotEqualTo(Money.of(95, "RUB"));
        assertThat(srcCatOffer.getBasePrice()).isNotEqualTo(Money.of(50, "RUB"));
        assertThat(srcCatOffer.getTaxes()).isNotEqualTo(Money.of(45, "RUB"));

        // reservation
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(5), "Order must be in OS_WAIT_CARD_TOKENIZED state"
        );
        assertThat(orderInfoRsp.getResult().getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_WAIT_TOKENIZATION);
        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(0);

        AeroflotTotalOffer updOffer = parsePayload(orderInfoRsp.getResult()).getVariant().getOffer();
        assertThat(updOffer.getTotalPrice()).isEqualTo(Money.of(100, "RUB"));
        AeroflotPriceDetail updCatOffer = updOffer.getCategoryOfferFor("SH1").getTotalPrice();
        assertThat(updCatOffer.getTotalPrice()).isEqualTo(Money.of(95, "RUB"));
        assertThat(updCatOffer.getBasePrice()).isEqualTo(Money.of(50, "RUB"));
        assertThat(updCatOffer.getTaxes()).isEqualTo(Money.of(45, "RUB"));
        verify(aviaTdApiClient, times(1)).invalidateVariant(any(), any());
    }

    @Test
    public void testAvailabilityCheckWithRetries() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter ->
                when(aeroflotProviderAdapter.checkAvailability(any(), any(), any()))
                        .thenThrow(new RetryableException("503"))
                        .thenReturn(true));

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(10), "Order must be in OS_WAIT_CARD_TOKENIZED state");
        assertThat(orderInfoRsp.getResult().getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_WAIT_TOKENIZATION);
        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(0);

        verify(aeroflotServiceSafeMock.getCurrentMocksHolder(), times(2))
                .checkAvailability(any(), any(), any());
    }

    @Test
    public void testAvailabilityCheckWithUnexpectedError() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter ->
                when(aeroflotProviderAdapter.checkAvailability(any(), any(), any()))
                        .thenThrow(new RuntimeException("10^10 requests per day error")));

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getWorkflowState() == EWorkflowState.WS_CRASHED, Duration.ofSeconds(5),
                "Order workflow must be crashed"
        );
        assertThat(orderInfoRsp.getResult().getAeroflotOrderState()).isEqualTo(EAeroflotOrderState.OS_CHECK_AVAILABILITY);
        assertThat(orderInfoRsp.getResult().getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_CHECK_AVAILABILITY);
        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(0);
    }

    @Test
    public void testAvailabilityCheckWithTooManyRequests() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter ->
                when(aeroflotProviderAdapter.checkAvailability(any(), any(), any()))
                        .thenThrow(new TooManyRequestsRetryableException(null)));

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        TestUtils.sleep(Duration.ofSeconds(3));
        TGetOrderInfoRsp orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(orderInfoRsp.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
        assertThat(orderInfoRsp.getResult().getAeroflotOrderState()).isEqualTo(EAeroflotOrderState.OS_CHECK_AVAILABILITY);
        assertThat(orderInfoRsp.getResult().getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_CHECK_AVAILABILITY);
        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(0);

        // there should be retries but they should happen after long delays (5 and more seconds)
        verify(aeroflotServiceSafeMock.getCurrentMocksHolder(), times(1))
                .checkAvailability(any(), any(), any());
    }

    @Test
    public void testNotAvailableDuringCreateOrder() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenThrow(new VariantNotAvailableException("not available"));
        });

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest("variant_testNotAvailableDuringCreateOrder"));
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state"
        );

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(parsePayload(orderInfoRsp.getResult()).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.NOT_AVAILABLE);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);

        verify(aviaApiProxy, times(1)).refreshVariant(eq("variant_testNotAvailableDuringCreateOrder"));
        verify(aviaTdApiClient, times(1)).invalidateVariant(any(), any());
    }

    @Test
    public void testPriceChangedDuringCreateOrder() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenThrow(new PriceChangedException("price mismatch"));
        });

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        TServiceInfo service = finalOrder.getService(0).getServiceInfo();
        assertThat(parsePayload(finalOrder).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.PRICE_CHANGED);
        assertThat(service.getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);
        verify(aviaTdApiClient, times(1)).invalidateVariant(any(), any());
    }

    @Test
    public void testPaymentFailed() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            AeroflotOrderRef testOrderRef = AeroflotOrderRef.builder()
                    .pnr("ROKEYJ").pnrDate("PNR_date_2019-03-21").orderId("SU55500ROKEYJ-...").build();
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenThrow(new AeroflotPaymentException("payment rejected", PaymentFailureReason.PAYMENT_REJECTED,
                            testOrderRef));
        });

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(parsePayload(finalOrder).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.PAYMENT_FAILED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);

        transactionTemplate.execute(status -> {
            // the PNR should be stored properly
            AeroflotOrderItem item =
                    aeroflotOrderItemRepository.getOne(UUID.fromString(finalOrder.getService(0).getServiceId()));
            assertThat(item.getAviaPnr()).isEqualTo("ROKEYJ/PNR_date_2019-03-21");
            return null;
        });

        // successful payment via aeroflot website
        when(aeroflotMqReader.readOrders())
                // emulating one Aeroflot MQ message
                .thenReturn(List.of(AeroflotMqRawData.builder().data("sync_msg_1").build()))
                .thenReturn(Collections.emptyList());
        when(aeroflotMqParser.parseMessage(eq("sync_msg_1")))
                .thenReturn(AeroflotMqData.builder()
                        .bookingDate(ZonedDateTime.parse("2019-03-21T10:42:30+00:00"))
                        .pnr("ROKEYJ")
                        .tickets(List.of("ticket_ext_paid"))
                        .passengers(List.of(AeroflotMqData.PassengerRef.builder().lastName("RESTOROV").build()))
                        .build());

        // should be fully restored (support tickets are disabled for this case)
        orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be restored after it receives a confirmation via MQ");
        TOrderInfo restoredOrder = orderInfoRsp.getResult();
        assertThat(parsePayload(restoredOrder).getTickets().get("1")).isEqualTo(List.of("ticket_ext_paid"));
        assertThat(restoredOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CONFIRMED);
        assertThat(restoredOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CONFIRMED);

        //verify(starTrekService, times(1))
        //        .createIssueForAeroflotCancelledOrderPaid(any(), eq("RESTOROV"), any());
    }

    @Test
    public void testPaymentFailed_3ds() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenReturn(aeroflotOrderCreateResultSuccessWith3ds(null));
            when(aeroflotProviderAdapter.getOrderStatus(any(), any())).thenReturn(
                    TestHelpers.paymentResult(AeroflotOrderStatus.PAYMENT_FAILED));
        });
        // we have to imitate the expiration this way as the short circuit 3ds failure isn't implemented yet,
        // see notes in AeroflotInvoiceRefreshService.refreshInvoiceWaitingConfirmation
        // todo(tlg-13): use the browser redirect signal here instead
        when(properties.getInvoiceConfirmationTimeout()).thenReturn(Duration.ofMillis(100));

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(parsePayload(finalOrder).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.PAYMENT_FAILED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);
    }

    @Test
    public void testBrokenUserInput() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenThrow(new BookingFailureException(BookingFailureReason.INVALID_CONTACT, "malformed email"));
        });

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(parsePayload(finalOrder).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.INVALID_CONTACT);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);
    }

    @Test
    public void testUnhandledErrorDuringCreateOrder() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenThrow(new RuntimeException("Incorrect or outdated token"));
        });

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED ");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getWorkflowState() == EWorkflowState.WS_CRASHED, Duration.ofSeconds(10),
                "Order workflow must be crashed"
        );
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_WAIT_TOKENIZATION);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_WAIT_ORDER_CREATED);
    }

    @Test
    public void testSystemCrashDuringCreateOrder() {
        // we'll test the call-at-most-once functionality here
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        AtomicInteger handlerCalls = new AtomicInteger(0);
        AtomicInteger externalCalls = new AtomicInteger(0);
        doAnswer(call -> {
            if (call.getArgument(0) instanceof TAeroflotOrderItemCardTokenized) {
                handlerCalls.incrementAndGet();
            }
            return call.callRealMethod();
        }).when(oiWaitTokenizationStateHandler).handleEvent(any(), any());
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenAnswer(call -> {
                        externalCalls.incrementAndGet();
                        // we need to emulate repeated processing here without using a RetryableException
                        throw new ConcurrencyFailureException("Pseudo system crash");
                    });
        });
        when(retryStrategy.getWaitDuration(anyInt(), any())).thenReturn(Duration.ofMillis(100));

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(10), "Order state must be OS_WAIT_CARD_TOKENIZED");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getWorkflowState() == EWorkflowState.WS_CRASHED,
                Duration.ofSeconds(15), "Order workflow state must be WS_CRASHED");

        assertThat(handlerCalls.get()).isEqualTo(2);
        assertThat(externalCalls.get()).isEqualTo(1);
    }

    @Test
    public void testNetworkUnavailabilityDuringCreateOrder() {
        // we'll test retries over the call-at-most-once functionality here
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        AtomicInteger handlerCalls = new AtomicInteger(0);
        AtomicInteger externalCalls = new AtomicInteger(0);
        doAnswer(call -> {
            if (call.getArgument(0) instanceof TAeroflotOrderItemCardTokenized) {
                handlerCalls.incrementAndGet();
            }
            return call.callRealMethod();
        }).when(oiWaitTokenizationStateHandler).handleEvent(any(), any());
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter -> {
            when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true);
            when(aeroflotProviderAdapter.createOrderAndStartPayment(any(), any(), any(), any()))
                    .thenAnswer(call -> {
                        if (externalCalls.incrementAndGet() < 3) {
                            throw new RetryableException("Network is unavailable");
                        }
                        return aeroflotOrderCreateResultSuccessWith3ds("PNR_no_network_on_OCRQ");
                    });
        });
        when(retryStrategy.getWaitDuration(anyInt(), any())).thenReturn(Duration.ofMillis(100));

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(10), "Order state must be OS_WAIT_CARD_TOKENIZED");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                // we need to wait the last WF state transition out of the all 3 order-related entities,
                // which is the invoice transition (order item (partner service) -> order notified -> invoice notified)
                // todo(tlg-13): it's better to make the order have the actual stable state:
                //      order item (partner service) -> invoices notified -> order notified
                rsp -> rsp.getResult().getInvoice(0).getAeroflotInvoiceState() == EAeroflotInvoiceState.IS_WAIT_CONFIRMATION,
                Duration.ofSeconds(15), "Invoice must be in the IS_WAIT_CONFIRMATION state");
        // OrderCreateRQ has been successfully executed, waiting for 3ds confirmation
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getAeroflotOrderState()).isEqualTo(EAeroflotOrderState.OS_WAIT_ORDER_PAID);
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_WAIT_CONFIRMATION);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState()).isEqualTo(EAeroflotInvoiceState.IS_WAIT_CONFIRMATION);

        assertThat(handlerCalls.get()).isEqualTo(3);
        assertThat(externalCalls.get()).isEqualTo(3);
    }

    @Test
    public void testOrderTrustTokenizationFailed() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter ->
                when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true));
        when(aeroflotTrustClient.getBasketStatus(any(), any())).thenReturn(TrustBasketStatusResponse.builder()
                .paymentStatus(PaymentStatusEnum.CANCELED)
                .build());

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order workflow must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        when(properties.getInvoiceConfirmationTimeout()).thenReturn(Duration.ZERO);
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(5), "Order item must be in IS_CANCELLED state"
        );
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        verify(starTrekService, times(1)).createIssueForAeroflotFailedTokenization(any(), any(), any(), any());
    }

    @Test
    public void testOrderTrustTokenizationFailedWithOkReason() {
        OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule, ordersService);
        mockTrustCalls(aeroflotTrustClient);
        aeroflotServiceSafeMock.initNewMocks(aeroflotProviderAdapter ->
                when(aeroflotProviderAdapter.checkAvailability(any(), any(), any())).thenReturn(true));
        when(aeroflotTrustClient.getBasketStatus(any(), any())).thenReturn(TrustBasketStatusResponse.builder()
                .paymentStatus(PaymentStatusEnum.CANCELED)
                .paymentRespCode("user_cancelled")
                .build());

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order workflow must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        when(properties.getInvoiceConfirmationTimeout()).thenReturn(Duration.ZERO);
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(5), "Order item must be in IS_CANCELLED state"
        );
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        verify(starTrekService, times(0)).createIssueForAeroflotFailedTokenization(any(), any(), any(), any());
    }

    private static TOrderInfo getOrder(OrderInterfaceV1BlockingStub client, String orderId) {
        return client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build()).getResult();
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient aeroflotTrustClient() {
            return mock(TrustClient.class);
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient aeroflotTrustClient) {
            return paymentProfile -> aeroflotTrustClient;
        }

        @Bean
        @Primary
        public <T extends AeroflotService & ThreadSafeMock<AeroflotService>> T aeroflotServiceProxy() {
            return newThreadSafeMockBuilder(AeroflotService.class)
                    .withDefaultInitializer(aeroflotService ->
                            when(aeroflotService.getOrderStatus(any(), any()))
                                    .thenReturn(TestHelpers.incompletePaymentResult()))
                    .build();
        }

        @Bean
        @Primary
        public AeroflotServiceProvider aeroflotServiceProvider() {
            return (orderItem) -> aeroflotServiceProxy();
        }
    }
}
