package ru.yandex.travel.orders.integration.aeroflot;

import java.time.Clock;
import java.time.Duration;
import java.util.Collections;
import java.util.concurrent.atomic.AtomicInteger;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotProviderProperties;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotPriceDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTotalOffer;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureReason;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonApiClient;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.commons.proto.EAviaCheckAvailabilityOutcome;
import ru.yandex.travel.orders.commons.proto.EAviaConfirmationOutcome;
import ru.yandex.travel.orders.commons.proto.EAviaMqEventOutcome;
import ru.yandex.travel.orders.commons.proto.EAviaTokenizationOutcome;
import ru.yandex.travel.orders.commons.proto.TAviaPaymentTestContext;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TServiceInfo;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.repository.TrustInvoiceRepository;
import ru.yandex.travel.orders.repository.mock.MockAeroflotOrderRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustAccountRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustBasketRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustRefundRepository;
import ru.yandex.travel.orders.services.avia.AviaApiProxy;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqReader;
import ru.yandex.travel.orders.services.mock.MockDBTrustClient;
import ru.yandex.travel.orders.services.mock.TrustDBMockProperties;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.configuration.AeroflotWorkflowProperties;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.handlers.AeroflotOrderItemHandlersHelper;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotService;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceProvider;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.MockAeroflotServiceAdapter;
import ru.yandex.travel.workflow.WorkflowMessageSender;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.orders.integration.IntegrationUtils.createServerAndBlockingStub;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.SESSION_KEY;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.YANDEX_UID;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.createOrderRequest;
import static ru.yandex.travel.orders.integration.aeroflot.TestHelpers.parsePayload;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "aeroflot-mq.enabled=true",
                "aeroflot-mq.repeat-interval=100ms",
                "quartz.enabled=true",
                "aeroflot-workflow.invoice-trust-refresh-timeout=100ms",
                "aeroflot-workflow.invoice-confirmation-refresh-timeout=100ms",
                "single-node.auto-start=true",
                "trust-db-mock.enabled=true",
                "mock-aeroflot.enabled=true"
        }
)
@ActiveProfiles("test")
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@DirtiesContext
@Slf4j
@SuppressWarnings({"ResultOfMethodCallIgnored"})
public class MockAeroflotOrderIntegrationTest {

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();
    @Rule
    public TestName name = new TestName();

    private static final AtomicInteger activeTests = new AtomicInteger(0);

    @Autowired
    private OrdersService ordersService;
    @SpyBean
    private AeroflotWorkflowProperties properties;

    @MockBean
    private AviaApiProxy aviaApiProxy;
    @MockBean
    private AeroflotMqReader aeroflotMqReader;

    @MockBean
    private AeroflotOrderItemHandlersHelper orderItemHandlersWfHelper;
    @MockBean
    private AviaTicketDaemonApiClient aviaTdApiClient;

    @MockBean
    private StarTrekService starTrekService;

    private Context context;

    @Before
    public void init() {
        log.info("Starting the {} test, activeTests={}", name.getMethodName(), activeTests.incrementAndGet());
        if (activeTests.get() > 1) {
            throw new IllegalStateException("The integration tests shouldn't run in parallel");
        }
        UserCredentials credentials = new UserCredentialsBuilder()
                .build(SESSION_KEY, YANDEX_UID, null, null, null, "127.0.0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();

        when(aeroflotMqReader.readOrders()).thenReturn(Collections.emptyList());
    }

    @After
    public void destroy() {
        log.info("Finishing the {} test, activeTests={}", name.getMethodName(), activeTests.decrementAndGet());
        Context.current().detach(context);
    }

    @Test
    public void testHappyPath() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule,
                ordersService);

        TAviaTestContext testContext = TAviaTestContext.newBuilder()
                .setCheckAvailabilityOutcome(EAviaCheckAvailabilityOutcome.CAO_SUCCESS)
                .setConfirmationOutcome(EAviaConfirmationOutcome.CO_SUCCESS)
                .setMqEventOutcome(EAviaMqEventOutcome.MEO_SUCCESS)
                .build();
        TAviaPaymentTestContext paymentTestContext = TAviaPaymentTestContext.newBuilder()
                .setTokenizationOutcome(EAviaTokenizationOutcome.TO_SUCCESS)
                .build();
        // create
        TCreateOrderRsp resp = client.createOrder(createOrderRequest(testContext, paymentTestContext));
        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build(); // we'll
        // reuse it
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getAeroflotOrderState()).isEqualTo(EAeroflotOrderState.OS_NEW);

        // reservation
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state"
        );

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CONFIRMED,
                Duration.ofSeconds(15), "Order must be in OS_CONFIRMED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_CONFIRMED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState()).isEqualTo(EAeroflotInvoiceState.IS_CONFIRMED);

        orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> parsePayload(rsp.getResult()).getTickets() != null,
                Duration.ofSeconds(15), "Order item must receive ticket numbers from the sync event");
        assertThat(parsePayload(orderInfoRsp.getResult()).getTickets().get("1")).isNotEmpty();
    }

    @Test
    public void testPriceChangedDuringAvailabilityCheck() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule,
                ordersService);

        TAviaTestContext testContext = TAviaTestContext.newBuilder()
                .setCheckAvailabilityOutcome(EAviaCheckAvailabilityOutcome.CAO_PRICE_CHANGED)
                .setConfirmationOutcome(EAviaConfirmationOutcome.CO_SUCCESS)
                .build();
        TAviaPaymentTestContext paymentTestContext = TAviaPaymentTestContext.newBuilder()
                .setTokenizationOutcome(EAviaTokenizationOutcome.TO_SUCCESS)
                .build();

        TCreateOrderReq createOrderReq = createOrderRequest(testContext, paymentTestContext);

        // create
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        String orderId = resp.getNewOrder().getOrderId();
        AeroflotTotalOffer srcOffer = parsePayload(resp.getNewOrder()).getVariant().getOffer();
        assertThat(srcOffer.getTotalPrice()).isNotEqualTo(Money.of(100, "RUB"));
        AeroflotPriceDetail srcCatOffer = srcOffer.getCategoryOfferFor("SH1").getTotalPrice();
        assertThat(srcCatOffer.getTotalPrice()).isNotEqualTo(Money.of(95, "RUB"));
        assertThat(srcCatOffer.getBasePrice()).isNotEqualTo(Money.of(50, "RUB"));
        assertThat(srcCatOffer.getTaxes()).isNotEqualTo(Money.of(45, "RUB"));

        // reservation
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(5), "Order must be in OS_WAIT_CARD_TOKENIZED state"
        );
        assertThat(orderInfoRsp.getResult().getService(0).getServiceInfo().getAeroflotItemState()).isEqualTo(EAeroflotItemState.IS_WAIT_TOKENIZATION);
        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(0);

        verify(aviaTdApiClient, times(1)).invalidateVariant(any(), any());
    }

    @Test
    public void testNotAvailableDuringCreateOrder() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule,
                ordersService);
        TAviaTestContext testContext = TAviaTestContext.newBuilder()
                .setCheckAvailabilityOutcome(EAviaCheckAvailabilityOutcome.CAO_SUCCESS)
                .setConfirmationOutcome(EAviaConfirmationOutcome.CO_VARIANT_NOT_AVIALABLE)
                .build();
        TAviaPaymentTestContext paymentTestContext = TAviaPaymentTestContext.newBuilder()
                .setTokenizationOutcome(EAviaTokenizationOutcome.TO_SUCCESS)
                .build();

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest(testContext, paymentTestContext));
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state"
        );

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(parsePayload(orderInfoRsp.getResult()).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.NOT_AVAILABLE);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);

        verify(aviaApiProxy, times(1)).refreshVariant(any());
        verify(aviaTdApiClient, times(1)).invalidateVariant(any(), any());
    }

    @Test
    public void testPriceChangedDuringCreateOrder() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule,
                ordersService);
        TAviaTestContext testContext = TAviaTestContext.newBuilder()
                .setCheckAvailabilityOutcome(EAviaCheckAvailabilityOutcome.CAO_SUCCESS)
                .setConfirmationOutcome(EAviaConfirmationOutcome.CO_PRICE_CHANGED)
                .build();
        TAviaPaymentTestContext paymentTestContext = TAviaPaymentTestContext.newBuilder()
                .setTokenizationOutcome(EAviaTokenizationOutcome.TO_SUCCESS)
                .build();

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest(testContext, paymentTestContext));
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        TServiceInfo service = finalOrder.getService(0).getServiceInfo();
        assertThat(parsePayload(finalOrder).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.PRICE_CHANGED);
        assertThat(service.getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);
        verify(aviaTdApiClient, times(1)).invalidateVariant(any(), any());
    }

    @Test
    public void testPaymentFailed() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule,
                ordersService);
        TAviaTestContext testContext = TAviaTestContext.newBuilder()
                .setCheckAvailabilityOutcome(EAviaCheckAvailabilityOutcome.CAO_SUCCESS)
                .setConfirmationOutcome(EAviaConfirmationOutcome.CO_PAYMENT_FAILED)
                .build();
        TAviaPaymentTestContext paymentTestContext = TAviaPaymentTestContext.newBuilder()
                .setTokenizationOutcome(EAviaTokenizationOutcome.TO_SUCCESS)
                .build();

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest(testContext, paymentTestContext));
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(parsePayload(finalOrder).getBookingFailureReason())
                .isEqualTo(BookingFailureReason.PAYMENT_FAILED);
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);

    }

    @Test
    public void testOrderTrustTokenizationFailed() {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client = createServerAndBlockingStub(cleanupRule,
                ordersService);
        TAviaTestContext testContext = TAviaTestContext.newBuilder()
                .setCheckAvailabilityOutcome(EAviaCheckAvailabilityOutcome.CAO_SUCCESS)
                .setConfirmationOutcome(EAviaConfirmationOutcome.CO_SUCCESS)
                .build();
        TAviaPaymentTestContext paymentTestContext = TAviaPaymentTestContext.newBuilder()
                .setTokenizationOutcome(EAviaTokenizationOutcome.TO_FAILURE)
                .build();

        // create & reserve
        TCreateOrderRsp resp = client.createOrder(createOrderRequest(testContext, paymentTestContext));
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED,
                Duration.ofSeconds(3), "Order workflow must be in OS_WAIT_CARD_TOKENIZED state");

        // payment
        when(properties.getInvoiceConfirmationTimeout()).thenReturn(Duration.ZERO);
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_AVIA_AEROFLOT)
                .setOrderId(orderId).setReturnUrl("some_return_url").build());
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getAeroflotOrderState() == EAeroflotOrderState.OS_CANCELLED,
                Duration.ofSeconds(5), "Order item must be in IS_CANCELLED state"
        );
        TOrderInfo finalOrder = orderInfoRsp.getResult();
        assertThat(finalOrder.getInvoice(0).getAeroflotInvoiceState())
                .isEqualTo(EAeroflotInvoiceState.IS_CANCELLED);
        assertThat(finalOrder.getService(0).getServiceInfo().getAeroflotItemState())
                .isEqualTo(EAeroflotItemState.IS_CANCELLED);
        assertThat(
                ProtoUtils.fromTJson(
                        finalOrder.getService(0).getServiceInfo().getPayload(),
                        AeroflotServicePayload.class
                ).getBookingFailureReason()
        ).isEqualTo(BookingFailureReason.TOKENIZATION_FAILED);
        verify(starTrekService, times(1)).createIssueForAeroflotFailedTokenization(any(), any(), any(), any());
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient aeroflotTrustClient(@Autowired MockTrustBasketRepository mockTrustBasketRepository,
                                               @Autowired MockTrustRefundRepository mockTrustRefundRepository,
                                               @Autowired MockTrustAccountRepository mockTrustAccountRepository,
                                               @Autowired TrustInvoiceRepository trustInvoiceRepository,
                                               @Autowired WorkflowMessageSender workflowMessageSender,
                                               @Autowired TrustDBMockProperties trustDBMockProperties,
                                               @Autowired Clock clock) {
            return new MockDBTrustClient(
                    mockTrustBasketRepository, mockTrustRefundRepository,
                    mockTrustAccountRepository, trustInvoiceRepository,
                    workflowMessageSender,
                    clock,
                    trustDBMockProperties
            );
        }

        @Bean
        @Primary
        public TrustClientProvider trustClientProvider(@Autowired TrustClient aeroflotTrustClient) {
            return paymentProfile -> aeroflotTrustClient;
        }

        @Bean
        @Primary
        public AeroflotService aeroflotServiceAdapter(@Autowired MockAeroflotOrderRepository mockAeroflotOrderRepository,
                                                      @Autowired AeroflotProviderProperties aeroflotProviderProperties,
                                                      @Autowired TransactionTemplate transactionTemplate) {
            return new MockAeroflotServiceAdapter(mockAeroflotOrderRepository, aeroflotProviderProperties,
                    transactionTemplate);
        }

        @Bean
        @Primary
        public AeroflotServiceProvider aeroflotServiceProvider(@Autowired AeroflotService aeroflotServiceAdapter) {
            return orderItem -> aeroflotServiceAdapter;
        }
    }
}
