package ru.yandex.travel.orders.integration.bus;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.bus.model.BusDocumentType;
import ru.yandex.travel.bus.model.BusGenderType;
import ru.yandex.travel.bus.model.BusLegalEntity;
import ru.yandex.travel.bus.model.BusOrderStatus;
import ru.yandex.travel.bus.model.BusPointInfo;
import ru.yandex.travel.bus.model.BusRegisterType;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusRide;
import ru.yandex.travel.bus.model.BusTicketStatus;
import ru.yandex.travel.bus.model.BusTicketType;
import ru.yandex.travel.bus.model.BusesPassenger;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.commons.proto.EBusBookOutcome;
import ru.yandex.travel.orders.commons.proto.EBusConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.EBusRefundInfoOutcome;
import ru.yandex.travel.orders.commons.proto.EBusRefundOutcome;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EPaymentOutcome;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.TBusTestContext;
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.ERefundPartState;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TCalculateRefundRsp;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TRefundPartInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.services.UrlShortenerService;
import ru.yandex.travel.orders.services.YaSmsService;
import ru.yandex.travel.orders.services.cloud.s3.S3Service;
import ru.yandex.travel.orders.services.pdfgenerator.PdfGeneratorService;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;


@SuppressWarnings("ResultOfMethodCallIgnored")
@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "quartz.enabled=true",
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "trust-hotels.clearing-refresh-timeout=1s",
                "trust-hotels.payment-refresh-timeout=1s",
                "single-node.auto-start=true",
                "bus.check-expiration-task-period=100ms",
        }
)
//@TestExecutionListeners(listeners = TruncateDatabaseTestExecutionListener.class, mergeMode =
//        TestExecutionListeners.MergeMode.MERGE_WITH_DEFAULTS)
@DirtiesContext
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@ActiveProfiles("test")
public class BusOrderFlowTests {

    private static final String SESSION_KEY = "qwerty";
    private static final String YANDEX_UID = "1234567890";
    private final UserCredentialsBuilder userCredentialsBuilder = new UserCredentialsBuilder();
    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();
    @Autowired
    private OrdersService ordersService;
    private Context context;

    private OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client;

    @MockBean
    private UrlShortenerService urlShortenerService;

    @MockBean
    private PdfGeneratorService pdfGeneratorService;

    @MockBean
    private YaSmsService smsSenderService;

    @MockBean
    private S3Service s3Service;

    @Before
    public void setUpCredentialsContext() {
        UserCredentials credentials = userCredentialsBuilder.build(SESSION_KEY, YANDEX_UID, null, null, null,
                "127.0.0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();
        client = IntegrationUtils.createServerAndBlockingStub(cleanupRule, ordersService);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("short-url");
        when(pdfGeneratorService.downloadDocumentAsBytesSync(any())).thenReturn(new byte[0]);
    }

    @After
    public void tearDownCredentialsContext() {
        Context.current().detach(context);
    }

    @Test
    public void testOrderExpired() {
        TCreateOrderReq createOrderReq = createOrderRequest();
        var builder = createOrderReq.toBuilder();
        builder.getCreateServicesBuilder(0).getBusTestContextBuilder().setExpireAfterSeconds(2);
        createOrderReq = builder.build();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        getOrderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        getOrderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_CANCELLED);
    }

    @Test
    public void testOrderReserved() {
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        newOrderToConfirmedFlow(orderId);
    }

    private TGetOrderInfoRsp newOrderToConfirmedFlow(String orderId) {
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        TGetOrderInfoRsp getOrderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build());

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");
        assertThat(getBusReservation(confirmedOrderResponse).getOrder().getStatus()).isEqualTo(BusOrderStatus.SOLD);
        return confirmedOrderResponse;
    }

    @Test
    public void testOrderRefund() {
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoRsp orderInfo = newOrderToConfirmedFlow(orderId);

        List<TRefundPartInfo> refundParts = orderInfo.getResult().getRefundPartsList();
        assertThat(refundParts.stream().filter(x -> x.getType() == ERefundPartType.RPT_ORDER).count()).isEqualTo(1);
        assertThat(refundParts.stream().filter(x -> x.getType() == ERefundPartType.RPT_SERVICE).count()).isEqualTo(1);
        assertThat(refundParts.stream().filter(x -> x.getType() == ERefundPartType.RPT_SERVICE_PART).count()).isEqualTo(1);
        TRefundPartInfo serviceRefundPart =
                refundParts.stream().filter(x -> x.getType() == ERefundPartType.RPT_SERVICE).findFirst().orElseThrow();
        assertThat(serviceRefundPart.getState()).isEqualTo(ERefundPartState.RPS_ENABLED);

        TCalculateRefundRsp calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder()
                .setOrderId(orderId)
                .addContext(serviceRefundPart.getContext())
                .build());
        assertThat(calculateRefundRsp.getRefundAmount().getAmount()).isGreaterThan(0);

        client.startRefund(TStartRefundReq.newBuilder()
                .setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken())
                .build());
        orderInfo = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED,
                Duration.ofSeconds(10), "Order must be in OS_REFUNDED state");
        assertThat(getBusReservation(orderInfo).getOrder().getTickets().get(0).getStatus()).isEqualTo(BusTicketStatus.RETURNED);
        refundParts = orderInfo.getResult().getRefundPartsList();
        assertThat(refundParts.stream().allMatch(x -> x.getState() == ERefundPartState.RPS_REFUNDED)).isTrue();
    }

    private TCreateOrderReq createOrderRequest() {
        var payload = new BusReservation();
        var ride = new BusRide();
        payload.setRide(ride);
        ride.setRideId("rest-ride-id");
        ride.setSupplierId(80);
        ride.setSupplier(new BusLegalEntity());
        ride.getSupplier().setRegisterType(BusRegisterType.COMPANY);
        ride.getSupplier().setRegisterNumber("12345678901");
        ride.getSupplier().setTaxationNumber("123456789012");
        ride.getSupplier().setLegalName("ООО \"Люкс-тур\"");
        ride.setPointFrom(new BusPointInfo());
        ride.getPointFrom().setSupplierDescription("МСК");
        ride.setPointTo(new BusPointInfo());
        ride.getPointTo().setSupplierDescription("СПБ");
        ride.setTitlePointFrom(new BusPointInfo());
        ride.getTitlePointFrom().setPointKey("c213");
        ride.getTitlePointFrom().setTitle("Москва");
        ride.getTitlePointFrom().setTimezone("Europe/Moscow");
        ride.setTitlePointTo(new BusPointInfo());
        ride.getTitlePointTo().setPointKey("c2");
        ride.getTitlePointTo().setPointKey("Санкт-Петербург");
        ride.getTitlePointTo().setTimezone("Europe/Moscow");
        ride.setDepartureTime(Instant.now().plus(1, ChronoUnit.DAYS));
        ride.setArrivalTime(Instant.now().plus(1, ChronoUnit.DAYS).plus(3, ChronoUnit.HOURS));
        payload.setRequestPassengers(new ArrayList<>());
        var p1 = new BusesPassenger();
        payload.getRequestPassengers().add(p1);
        p1.setBirthday(LocalDate.of(1990, 5, 8));
        p1.setDocumentType(BusDocumentType.RU_PASSPORT);
        p1.setDocumentTypePartnerId("1");
        p1.setCitizenship("RU");
        p1.setCitizenshipPartnerId("RU");
        p1.setGender(BusGenderType.MALE);
        p1.setGenderPartnerId("M");
        p1.setTicketTypePartnerId("0");
        p1.setTicketType(BusTicketType.FULL);
        p1.setFirstName("John");
        p1.setLastName("McClane");
        p1.setMiddleName("");
        p1.setDocumentNumber("3123123456");
        payload.setEmail("test@test.com");
        payload.setPhone("+79111111111");
        return TCreateOrderReq.newBuilder()
                .setOrderType(EOrderType.OT_GENERIC)
                .setDeduplicationKey(UUID.randomUUID().toString())
                .addCreateServices(
                        TCreateServiceReq.newBuilder()
                                .setServiceType(EServiceType.PT_BUS)
                                .setBusTestContext(TBusTestContext.newBuilder()
                                        .setBookOutcome(EBusBookOutcome.BBO_SUCCESS)
                                        .setConfirmOutcome(EBusConfirmOutcome.BCO_SUCCESS)
                                        .setRefundInfoOutcome(EBusRefundInfoOutcome.BRIO_SUCCESS)
                                        .setRefundOutcome(EBusRefundOutcome.BRO_SUCCESS)
                                        .build())
                                .setSourcePayload(ProtoUtils.toTJson(payload))
                )
                .setPaymentTestContext(TPaymentTestContext.newBuilder()
                        .setPaymentOutcome(EPaymentOutcome.PO_SUCCESS)
                        .build())
                .setCurrency(ECurrency.C_RUB)
                .setOwner(TUserInfo.newBuilder()
                        .setEmail("test@test.com")
                        .setPhone("+79111111111")
                        .setYandexUid(YANDEX_UID)
                )
                .build();
    }

    private BusReservation getBusReservation(TGetOrderInfoRsp response) {
        var payloadJson = response.getResult().getService(0).getServiceInfo().getPayload();
        return fromTJson(payloadJson, BusReservation.class);
    }
}

