package ru.yandex.travel.orders.integration.hotels;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import lombok.Builder;
import lombok.Data;
import org.javamoney.moneta.Money;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.runner.RunWith;
import org.mockito.stubbing.Answer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.MockBeans;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.EFiscalReceiptType;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.partners.expedia.ApiVersion;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.CancellationStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.HoldItineraryLinks;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryRoom;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ReservationResult;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ResumeReservationStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.RoomConfirmation;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.RoomLinks;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.RoomStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.Link;
import ru.yandex.travel.hotels.proto.EHotelConfirmationOutcome;
import ru.yandex.travel.hotels.proto.EHotelRefundOutcome;
import ru.yandex.travel.hotels.proto.EHotelReservationOutcome;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.orders.admin.proto.OrdersAdminInsecureInterfaceV1Grpc;
import ru.yandex.travel.orders.cache.HotelAgreementDictionary;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.commons.proto.EPromoCodeNominalType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.WellKnownAccount;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.promo.DiscountApplicationConfig;
import ru.yandex.travel.orders.entities.promo.PromoAction;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivation;
import ru.yandex.travel.orders.entities.promo.PromoCodeBehaviourOverride;
import ru.yandex.travel.orders.grpc.OrdersAdminInsecureService;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.grpc.PromoCodeUserService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.integration.TestGrpcContext;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.PromoCodesUserInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TCalculateRefundReq;
import ru.yandex.travel.orders.proto.TCalculateRefundRsp;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInvoiceInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TServiceInfo;
import ru.yandex.travel.orders.proto.TSetDeferredPaymentReq;
import ru.yandex.travel.orders.proto.TStartCancellationReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.promo.PromoActionRepository;
import ru.yandex.travel.orders.repository.promo.PromoCodeActivationRepository;
import ru.yandex.travel.orders.repository.promo.PromoCodeRepository;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.MailSenderService;
import ru.yandex.travel.orders.services.PromoServiceHelper;
import ru.yandex.travel.orders.services.cloud.s3.InMemoryS3Object;
import ru.yandex.travel.orders.services.cloud.s3.S3Service;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.pdfgenerator.PdfGeneratorService;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfStateResponse;
import ru.yandex.travel.orders.services.plus.YandexPlusPromoService;
import ru.yandex.travel.orders.services.promo.PromoCodeUnifier;
import ru.yandex.travel.orders.services.promo.taxi2020.Taxi2020PromoService;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.EBronevikItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentClear;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.testing.misc.TestResources;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMaintenanceService;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

/**
 * Class to be extended by integration tests testing the hotel booking flow.
 *
 * @implNote later it might be reasonable to separate the class further to separate different logical parts.
 */
@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "quartz.enabled=true",
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "trust-hotels.clearing-refresh-timeout=100ms",
                "trust-hotels.payment-refresh-timeout=100ms",
                "trust-hotels.schedule-clearing-rate=100ms",
                "single-node.auto-start=true",
                "hotel-workflow.check-expiration-task.enabled=true",
                "hotel-workflow.check-expiration-task.period=100ms",
                "financial-events.enabled=true",
                "mock-trust.enabled=true"
        }
)
@ActiveProfiles("test")
@MockBeans({
        @MockBean(HotelAgreementDictionary.class)
})
public abstract class AbstractHotelOrderFlowTest {
    public static final String PAYLOAD_EXPEDIA = "expedia";
    public static final String PAYLOAD_EXPEDIA_2119 = "expedia_checkin_2119";
    public static final String PAYLOAD_BNOVO = "bnovo";
    public static final String PAYLOAD_BNOVO_FULLY_REFUNDABLE = "bnovo_fully_refundable";

    protected static final String SESSION_KEY = "qwerty";
    protected static final String YANDEX_UID = "1234567890";
    protected static final String PASSPORT_ID = "100";
    protected static final String ITINERARY_ID = "123";
    protected static final String TOKEN = "token";
    protected static final String ROOM_ID = "166341";
    protected static final UserCredentialsBuilder userCredentialsBuilder = new UserCredentialsBuilder();
    protected static final Duration TIMEOUT = Duration.ofSeconds(30);

    protected static final Tuple2<EPromoCodeNominalType, BigDecimal> HUNDRED_RUBLES_PROMO =
            Tuple2.tuple(EPromoCodeNominalType.NT_VALUE, BigDecimal.valueOf(100));
    protected static final Tuple2<EPromoCodeNominalType, BigDecimal> THREE_HUNDRED_RUBLES_PROMO =
            Tuple2.tuple(EPromoCodeNominalType.NT_VALUE, BigDecimal.valueOf(300));
    protected static final Tuple2<EPromoCodeNominalType, BigDecimal> TEN_PERCENT_PROMO =
            Tuple2.tuple(EPromoCodeNominalType.NT_PERCENT, BigDecimal.valueOf(10));
    // unique for each test
    protected String uniqueTestIp;

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();
    @Autowired
    protected OrdersService ordersService;
    @Autowired
    protected OrdersAdminInsecureService adminInsecureService;
    @Autowired
    protected PromoCodeUserService promoCodeUserService;
    @Autowired
    protected WorkflowMessageSender workflowMessageSender;
    @MockBean
    protected ExpediaClient expediaClient;
    @Autowired
    protected TransactionTemplate transactionTemplate;
    @Autowired
    protected OrderRepository orderRepository;
    @Autowired
    protected WorkflowRepository workflowRepository;
    @Autowired
    protected WorkflowMaintenanceService workflowMaintenanceService;
    @Autowired
    protected AccountService accountService;

    @Autowired
    protected TrustClientProvider trustClientProvider;

    @MockBean
    protected PdfGeneratorService pdfGeneratorService;

    @MockBean
    protected StarTrekService starTrekService;

    @MockBean
    protected MailSenderService mailSenderService;

    @Autowired
    protected PromoActionRepository promoActionRepository;

    @Autowired
    protected PromoCodeRepository promoCodeRepository;

    @Autowired
    protected PromoCodeActivationRepository promoCodeActivationRepository;

    @MockBean
    protected Taxi2020PromoService taxi2020PromoService;

    @MockBean
    protected YandexPlusPromoService yandexPlusPromoService;

    @Autowired
    protected FinancialEventRepository financialEventRepository;

    @MockBean
    protected PromoServiceHelper promoServiceHelper;

    @MockBean
    private S3Service s3Service;

    protected Context context;

    protected OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client;

    protected PromoCodesUserInterfaceV1Grpc.PromoCodesUserInterfaceV1BlockingStub promoCodeClient;

    protected OrdersAdminInsecureInterfaceV1Grpc.OrdersAdminInsecureInterfaceV1BlockingStub adminClient;

    protected String promoCodeString;

    private static boolean isServiceCancelled(TServiceInfo serviceInfo) {
        switch (serviceInfo.getOneOfItemStatesCase()) {
            case DOLPHINITEMSTATE:
                return serviceInfo.getDolphinItemState() == EDolphinItemState.IS_CANCELLED;
            case EXPEDIAITEMSTATE:
                return serviceInfo.getExpediaItemState() == EExpediaItemState.IS_CANCELLED;
            case TRAVELLINEITEMSTATE:
                return serviceInfo.getTravellineItemState() == ETravellineItemState.IS_CANCELLED;
            case BNOVOITEMSTATE:
                return serviceInfo.getBNovoItemState() == EBNovoItemState.IS_CANCELLED;
            case BRONEVIKITEMSTATE:
                return serviceInfo.getBronevikItemState() == EBronevikItemState.IS_CANCELLED;
            default:
                throw new IllegalStateException();
        }
    }

    protected static CancellationDetails getCancellationDetails(TServiceInfo serviceInfo) {
        return ProtoUtils.fromTJson(serviceInfo.getPayload(), ExpediaHotelItinerary.class).getOrderCancellationDetails();
    }

    @Before
    public void setUpCredentialsContext() {
        uniqueTestIp = UUID.randomUUID().toString();
        UserCredentials credentials = userCredentialsBuilder.build(SESSION_KEY, YANDEX_UID, PASSPORT_ID, null, null,
                uniqueTestIp, false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();

        TestGrpcContext grpcContext = TestGrpcContext.createTestServer(cleanupRule,
                ordersService, promoCodeUserService, adminInsecureService);

        client = OrderInterfaceV1Grpc.newBlockingStub(grpcContext.createChannel());
        promoCodeClient = PromoCodesUserInterfaceV1Grpc.newBlockingStub(grpcContext.createChannel());
        adminClient = OrdersAdminInsecureInterfaceV1Grpc.newBlockingStub(grpcContext.createChannel());
        when(pdfGeneratorService.downloadDocumentAsBytesSync(any())).thenReturn(new byte[0]);
        when(pdfGeneratorService.getState(any())).thenReturn(
                new PdfStateResponse("https://pdf.document.link", Instant.now().plus(1, ChronoUnit.MINUTES)));
        when(promoServiceHelper.determinePromosForOffers(any(), any(), any(), any(), any()))
                .thenReturn(CompletableFuture.completedFuture(Collections.emptyList()));
        when(promoServiceHelper.validateAppliedPromoCampaigns(any(Order.class), any(), any()))
                .thenReturn(CompletableFuture.completedFuture(null));
        when(pdfGeneratorService.downloadDocumentAsBytesSync(any())).thenReturn(new byte[0]);
        when(s3Service.readObject(any())).thenReturn(new InMemoryS3Object("tst", "test", "test", new byte[0]));
    }

    @After
    public void tearDownCredentialsContext() {
        Context.current().detach(context);
    }


    @After
    public void stopRunningOrderWorkflows() {
        transactionTemplate.execute(i1 -> {
            orderRepository.findAll().stream()
                    .filter(Objects::nonNull)
                    .filter(order -> order.getWorkflow() != null)
                    .filter(order -> order.getWorkflow().getId() != null)
                    .filter(order -> order.getWorkflow().getState().equals(EWorkflowState.WS_RUNNING))
                    .forEach(order -> {
                        transactionTemplate.execute(i2 ->
                                workflowMaintenanceService.stopRunningWorkflow(order.getWorkflow().getId()));
                        transactionTemplate.execute(i2 -> {
                            workflowMaintenanceService.stopSupervisedRunningWorkflows(order.getWorkflow().getId());
                            return null;
                        });
                    });
            return null;
        });
    }

    protected TGetOrderInfoRsp waitForOrderState(String orderId, EHotelOrderState state) {
        return IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getHotelOrderState() == state, TIMEOUT,
                "Order must be in " + state.name() + " state");
    }

    protected void confirmAndRefund(String orderId, PaymentsBehaviour paymentsBehaviour) {
        confirmAndRefund(orderId, paymentsBehaviour, PromoCodeCheckBehaviour.DO_NOT_CHECK);
    }

    protected void confirmAndRefund(String orderId, PaymentsBehaviour paymentsBehaviour,
                                    PromoCodeCheckBehaviour promoCodeCheckBehaviour) {
        confirm(orderId, promoCodeCheckBehaviour);
        if (paymentsBehaviour == PaymentsBehaviour.CLEAR) {
            clear(orderId);
        }
        refund(orderId, paymentsBehaviour);
    }

    protected void doRefundInner(String orderId) {
        Money oldTrustBalance;
        Money oldOrderAccountBalance;
        Money newTrustBalance;
        Money newOrderAccountBalance;
        BigDecimal trustBalanceDelta;
        BigDecimal orderBalanceDelta;


        oldTrustBalance = getTrustBalance();
        oldOrderAccountBalance = getOrderBalance(orderId);
        TGetOrderInfoRsp refundedResponse = refundConfirmedOrderWithoutRaces(orderId);
        newTrustBalance = getTrustBalance();
        newOrderAccountBalance = getOrderBalance(orderId);
        assertThat(refundedResponse.getResult().getServiceCount()).isEqualTo(1);
        trustBalanceDelta = newTrustBalance.subtract(oldTrustBalance).getNumberStripped();
        orderBalanceDelta = oldOrderAccountBalance.subtract(newOrderAccountBalance).getNumberStripped();
        assertThat(trustBalanceDelta).isEqualTo(orderBalanceDelta);
    }

    protected void refund(String orderId, PaymentsBehaviour paymentsBehaviour) {
        refund(orderId, paymentsBehaviour, ETrustInvoiceState.IS_CLEARED);
    }

    protected void refund(String orderId, PaymentsBehaviour paymentsBehaviour,
                          ETrustInvoiceState expectedTrustInvoiceState) {
        doRefundInner(orderId);

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                r -> r.getResult().getInvoiceList().size() > 0 &&
                        r.getResult().getInvoice(0).getTrustInvoiceState() == expectedTrustInvoiceState,
                Duration.ofSeconds(15),
                "Invoice must be in " + expectedTrustInvoiceState + " state");

        // in case of 'clearPayments=false' we un-hold some money and clear the rest, no actual refund happens
        boolean clearPayments = paymentsBehaviour == PaymentsBehaviour.CLEAR;
        EFiscalReceiptType expectedRefundFrt = clearPayments ? EFiscalReceiptType.FRT_REFUND :
                EFiscalReceiptType.FRT_CLEAR;
        String expectedRefundReceiptUrlPart = clearPayments ? "refund.fiscal.receipt" : "fiscal.clearing.receipt";
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getInvoice(0).getFiscalReceiptList().stream()
                        .filter(fr -> fr.getType() == expectedRefundFrt)
                        .filter(fr -> !Strings.isNullOrEmpty(fr.getUrl()))
                        .filter(fr -> fr.getUrl().contains(expectedRefundReceiptUrlPart))
                        .count() == 1,
                Duration.ofSeconds(15), String.format("We should have got one refund receipt of the %s type",
                        expectedRefundFrt));

        transactionTemplate.execute(tStatus -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            Invoice invoice = order.getCurrentInvoice();
            assertThat(invoice).isNotNull();
            assertThat(invoice.getInvoiceItems()).hasSize(1);
            return null;
        });

        waitForOrderState(orderId, EHotelOrderState.OS_REFUNDED);
    }

    protected void refundWithZeroPayment(String orderId) {
        doRefundInner(orderId);

        waitForOrderState(orderId, EHotelOrderState.OS_REFUNDED);
    }

    protected void clear(String orderId) {
        TGetOrderInfoRsp rsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        UUID invoiceWorkflowId = UUID.fromString(rsp.getResult().getInvoice(0).getWorkflowId());
        transactionTemplate.execute(tStatus -> {
            workflowMessageSender.scheduleEvent(invoiceWorkflowId, TPaymentClear.newBuilder().build());
            return null;
        });

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                r -> r.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_CLEARED,
                Duration.ofSeconds(15), "Invoice must be in IS_CLEARED state");
    }

    protected void startAndAuthorizePaymentAndCheck(String orderId, PromoCodeCheckBehaviour promoCodeCheckBehaviour) {
        Money oldTrustBalance = getTrustBalance();
        Money oldOrderAccountBalance = getOrderBalance(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = startAndAuthorizePayment(orderId, promoCodeCheckBehaviour);

        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(1);
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getCurrentInvoice().getFiscalReceiptList().stream()
                        .filter(fr -> fr.getType() == EFiscalReceiptType.FRT_ACQUIRE)
                        .filter(fr -> !Strings.isNullOrEmpty(fr.getUrl()))
                        .filter(fr -> fr.getUrl().contains("fiscal.receipt"))
                        .count() == 1,
                Duration.ofSeconds(15), "We should have got one fiscal receipt of the ACQUIRE type");
        verify(taxi2020PromoService).registerConfirmedOrder(argThat((HotelOrder order) ->
                order.getId().toString().equals(orderId)));

        Money newTrustBalance = getTrustBalance();
        Money newOrderAccountBalance = getOrderBalance(orderId);
        BigDecimal trustBalanceDelta = newTrustBalance.subtract(oldTrustBalance).getNumberStripped();
        BigDecimal orderBalanceDelta = oldOrderAccountBalance.subtract(newOrderAccountBalance).getNumberStripped();
        assertThat(trustBalanceDelta).isEqualTo(orderBalanceDelta);
    }

    protected void confirmWithoutPayment(String orderId) {
        reserveOrderWithoutPayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForOrderState(orderId, EHotelOrderState.OS_CONFIRMED);
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
    }

    /**
     * The difference from {@link #confirm} is that after start payment we don't need authorizePayment.
     * The order should be set to use either deferred or postpay
     */
    protected void confirmWithZeroPayment(String orderId) {
        reserveOrder(orderId);

        startPayment(orderId, true);

        TGetOrderInfoRsp confirmedOrderResponse = waitForOrderState(orderId, EHotelOrderState.OS_CONFIRMED);
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
    }

    protected void confirm(String orderId, PromoCodeCheckBehaviour promoCodeCheckBehaviour) {
        if (promoCodeCheckBehaviour == PromoCodeCheckBehaviour.CHECK_APPLIED) {
            TGetOrderInfoRsp originalOrder =
                    client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
            Money discountAmount = ProtoUtils.fromTPrice(originalOrder.getResult().getPriceInfo().getDiscountAmount());
            assertThat(discountAmount).isEqualTo(Money.zero(ProtoCurrencyUnit.RUB));
        }
        reserveOrder(orderId);

        startAndAuthorizePaymentAndCheck(orderId, promoCodeCheckBehaviour);
    }

    protected void confirmFailed(String orderId) {
        reserveOrder(orderId);

        startPayment(orderId);
        authorizePayment(orderId);

        waitForOrderState(orderId, EHotelOrderState.OS_CANCELLED);

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId, rsp -> {
            assertThat(rsp.getResult().getInvoiceCount()).isEqualTo(1);
            assertThat(rsp.getResult().getInvoice(0).getOneOfInvoiceStatesCase()).isEqualTo(TOrderInvoiceInfo.OneOfInvoiceStatesCase.TRUSTINVOICESTATE);
            return rsp.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_CANCELLED;
        }, TIMEOUT, "Multiple assertions for order must be valid");
        Money currentOrderBalance = getOrderBalance(orderId);
        assertThat(currentOrderBalance.getNumberStripped()).isEqualTo(BigDecimal.ZERO);
    }

    protected void cancelOrder(String orderId) {
        client.startCancellation(TStartCancellationReq.newBuilder().setOrderId(orderId).build());
        waitForOrderState(orderId, EHotelOrderState.OS_CANCELLED);
    }

    protected void reserveAndCancel(String orderId, PromoCodeCheckBehaviour promoCodeCheckBehaviour) {
        reserveOrder(orderId);
        cancelOrder(orderId);

        TGetOrderInfoRsp reservedOrder = IntegrationUtils.waitForPredicateOrTimeout(client, orderId, rsp -> {
            assertThat(rsp.getResult().getServiceCount()).isEqualTo(1);
            return isServiceCancelled(rsp.getResult().getService(0).getServiceInfo());
        }, TIMEOUT, "Multiple assertions for order must be valid");
        switch (promoCodeCheckBehaviour) {
            case CHECK_APPLIED:
                Money discountAmount =
                        ProtoUtils.fromTPrice(reservedOrder.getResult().getPriceInfo().getDiscountAmount());
                assertThat(discountAmount).isGreaterThan(Money.zero(ProtoCurrencyUnit.RUB));
                break;
            case CHECK_NOT_APPLICABLE:
                boolean allNotApplicable =
                        reservedOrder.getResult().getPriceInfo().getPromoCodeApplicationResultsList().stream()
                                .allMatch(p -> p.getType() == EPromoCodeApplicationResultType.ART_NOT_APPLICABLE);
                assertThat(allNotApplicable).isTrue().withFailMessage("All promocodes must be NOT_APPLICABLE");
                break;
            case CHECK_ALREADY_APPLIED:
                boolean allAlreadyApplied =
                        reservedOrder.getResult().getPriceInfo().getPromoCodeApplicationResultsList().stream()
                                .allMatch(p -> p.getType() == EPromoCodeApplicationResultType.ART_ALREADY_APPLIED);
                assertThat(allAlreadyApplied).isTrue().withFailMessage("All promocodes must be ALREADY_APPLIED");
                break;
        }
        Money currentOrderBalance = getOrderBalance(orderId);
        assertThat(currentOrderBalance.getNumberStripped()).isEqualTo(BigDecimal.ZERO);
    }

    protected void failOnReservation(String orderId, CancellationDetails.Reason reason) {
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                infoRsp -> {
                    var cancellationDetails = getCancellationDetails(
                            infoRsp.getResult().getService(0).getServiceInfo());
                    return infoRsp.getResult().getHotelOrderState() == EHotelOrderState.OS_CANCELLED &&
                            cancellationDetails != null &&
                            cancellationDetails.getReason() == reason;
                },
                TIMEOUT, "Order must be in OS_CANCELLED state " +
                        "and the service cancellation reason must be " + reason.toString());
    }

    protected void failOnReservation(String orderId) {
        failOnReservation(orderId, CancellationDetails.Reason.SOLD_OUT);
    }

    protected TGetOrderInfoRsp startAndAuthorizePayment(String orderId) {
        return startAndAuthorizePayment(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
    }

    protected TGetOrderInfoRsp startAndAuthorizePayment(String orderId,
                                                        PromoCodeCheckBehaviour promoCodeCheckBehaviour) {
        TGetOrderInfoRsp reservedOrder = startPayment(orderId);
        if (promoCodeCheckBehaviour == PromoCodeCheckBehaviour.CHECK_APPLIED) {
            Money discountAmount = ProtoUtils.fromTPrice(reservedOrder.getResult().getPriceInfo().getDiscountAmount());
            assertThat(discountAmount).isGreaterThan(Money.zero(ProtoCurrencyUnit.RUB));
        }
        authorizePayment(orderId);
        return waitForOrderState(orderId, EHotelOrderState.OS_CONFIRMED);
    }

    protected TGetOrderInfoRsp startPayment(String orderId) {
        return startPayment(orderId, false);
    }

    protected TGetOrderInfoRsp startPayment(String orderId, boolean deferred) {
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST).setSource("desktop")
                .setUseNewInvoiceModel(true)
                .setOrderId(orderId).setReturnUrl("some_return_url").build()); // safely ignoring response, as we
        return IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp2 -> deferred ||
                        (rsp2.getResult().getInvoiceList().size() > 0 &&
                                rsp2.getResult().getInvoice(0).getTrustInvoiceState()
                                        == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT),
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
    }

    protected TGetOrderInfoRsp reserveOrder(String orderId) {
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        return waitForOrderState(orderId, EHotelOrderState.OS_WAITING_PAYMENT);
    }

    /**
     * For orders that do not need payment and go reserve -> confirm (e.g postpay)
     */
    protected void reserveOrderWithoutPayment(String orderId) {
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
    }

    /**
     * The confirmed order is updated even after reaching the OS_CONFIRMED state.
     * E.g. in the ConfirmedStateHandler.handleVoucherCreated(TVoucherCreated) event handler.
     * Because of it calling the grpc 'refund' api leads to ObjectOptimisticLockingFailureException sometimes.
     * To work the problem around this method tries to wait until the last known update is executed.
     * In other words it artificially serializes all such modifications.
     * <p>
     * todo(tlg-13,mbobrov): think about better ways of handling optimistic locking exceptions in tests
     */
    protected TGetOrderInfoRsp refundConfirmedOrderWithoutRaces(String orderIdStr) {
        UUID orderId = UUID.fromString(orderIdStr);
        TestUtils.waitForState("Order gets updated by WPS for the last time", () ->
                transactionTemplate.execute(cb -> orderRepository.getOne(orderId).getDocumentUrl()) != null);

        return refundOrder(orderIdStr);
    }

    protected TGetOrderInfoRsp refundOrder(String orderId) {
        TCalculateRefundRsp calculateRefundRsp =
                client.calculateRefund(TCalculateRefundReq.newBuilder().setOrderId(orderId).build());
        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId).setRefundToken(calculateRefundRsp.getRefundToken()).build());

        return waitForOrderState(orderId, EHotelOrderState.OS_REFUNDED);
    }

    protected void reserveThenExpire(String orderId) {
        reserveOrder(orderId);

        expireOrderItem(orderId);

        waitForOrderState(orderId, EHotelOrderState.OS_CANCELLED);
    }

    protected THotelTestContext getAllSuccessfulContext() {
        return THotelTestContext.newBuilder()
                .setReservationOutcome(EHotelReservationOutcome.RO_SUCCESS)
                .setConfirmationOutcome(EHotelConfirmationOutcome.CO_SUCCESS)
                .setRefundOutcome(EHotelRefundOutcome.RF_SUCCESS)
                .build();
    }

    protected THotelTestContext getContextFailedOnConfirmation() {
        return THotelTestContext.newBuilder()
                .setReservationOutcome(EHotelReservationOutcome.RO_SUCCESS)
                .setConfirmationOutcome(EHotelConfirmationOutcome.CO_NOT_FOUND)
                .setRefundOutcome(EHotelRefundOutcome.RF_SUCCESS)
                .build();
    }

    protected THotelTestContext getContextSoldOut() {
        return THotelTestContext.newBuilder()
                .setReservationOutcome(EHotelReservationOutcome.RO_SOLD_OUT)
                .build();
    }

    protected void authorizePayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            Invoice invoice = order.getCurrentInvoice();
            assertThat(invoice).isNotNull();
            String purchaseToken = invoice.getPurchaseToken();

            MockTrustClient mockTrustClient =
                    (MockTrustClient) trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile());
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    protected void failPayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            Invoice invoice = order.getCurrentInvoice();
            assertThat(invoice).isNotNull();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient =
                    (MockTrustClient) trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile());
            mockTrustClient.paymentNotAuthorized(purchaseToken);
            return null;
        });
    }

    protected void failPaymentAndRetry(String orderId) {
        reserveOrder(orderId);

        startPayment(orderId);

        failPayment(orderId);

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED, TIMEOUT
                , "Order must have failed payment state");

        startAndAuthorizePayment(orderId);
    }

    protected void expireOrderItem(String orderId) {
        transactionTemplate.execute(ignored -> {
            var order = orderRepository.getOne(UUID.fromString(orderId));
            order.getOrderItems().get(0).setExpiresAt(Instant.now().minusSeconds(60 * 60));
            return null;
        });
    }

    protected void initializeExpediaMockForGetHeldItineraryByAffiliateId(ApiVersion version) {
        when(expediaClient.getItineraryByAffiliateIdSync(any(), any(), any(), any(), any()))
                .thenReturn(Itinerary.builder()
                        .links(HoldItineraryLinks.builder()
                                .resume(Link.builder().href(String.format("/%s/itineraries/%s?token=%s",
                                        version.getValue(), ITINERARY_ID, TOKEN)).build())
                                .cancel(Link.builder().href(String.format("/%s/itineraries/%s?token=%s",
                                        version.getValue(), ITINERARY_ID, TOKEN)).build())
                                .retrieve(Link.builder().href(String.format("/%s/itineraries/%s?token=%s",
                                        version.getValue(), ITINERARY_ID, TOKEN)).build())
                                .build())
                        .build());
    }

    protected void initializeExpediaMockForGetConfirmedItineraryByAffiliateId(ApiVersion apiVersion) {
        when(expediaClient.getItineraryByAffiliateIdSync(any(), any(), any(), any(), any()))
                .thenAnswer(
                        (Answer<Itinerary>) invocation ->
                                Itinerary.builder()
                                        .itineraryId(invocation.getArgument(0))
                                        .rooms(List.of(ItineraryRoom.builder()
                                                .confirmationId(RoomConfirmation.builder()
                                                        .property("hotelConfirmationId")
                                                        .expedia("partnerConfirmationId")
                                                        .build())
                                                .id(ROOM_ID)
                                                .status(RoomStatus.BOOKED)
                                                .links(RoomLinks.builder()
                                                        .cancel(Link.builder()
                                                                .href(String.format("/%s/itineraries/%s/rooms/%s" +
                                                                                "?token=%s",
                                                                        apiVersion.getValue(), ITINERARY_ID, ROOM_ID,
                                                                        TOKEN))
                                                                .build())
                                                        .build())
                                                .build()))
                                        .build()
                );
    }

    protected void initializeExpediaMocksForConfirmedAndRefunded(ApiVersion apiVersion) {
        ReservationResult expediaReservationResult = createExpediaReservationResult();
        when(expediaClient.usingApi(any())).thenReturn(expediaClient);
        when(expediaClient.getApiVersion()).thenReturn(apiVersion);
        when(expediaClient.reserveItinerarySync(any(), any(), any(), any(), any())).thenReturn(
                expediaReservationResult
        );
        when(expediaClient.resumeItinerarySync(any(), any(), any(), any(), any()))
                .thenReturn(ResumeReservationStatus.SUCCESS);

        when(expediaClient.cancelConfirmedItinerarySync(any(), any(), any(), any(), any(), any()))
                .thenReturn(CancellationStatus.SUCCESS);


        when(expediaClient.getItinerarySync(any(), any(), any(), any(), any()))
                .thenReturn(Itinerary.builder()
                        .links(HoldItineraryLinks.builder()
                                .resume(Link.builder().href(String.format("/%s/itineraries/%s?token=%s",
                                        apiVersion.getValue(), ITINERARY_ID, TOKEN)).build())
                                .cancel(Link.builder().href(String.format("/%s/itineraries/%s?token=%s",
                                        apiVersion.getValue(), ITINERARY_ID, TOKEN)).build())
                                .retrieve(Link.builder().href(String.format("/%s/itineraries/%s?token=%s",
                                        apiVersion.getValue(), ITINERARY_ID, TOKEN)).build())
                                .build())
                        .build())
                .thenAnswer(
                        (Answer<Itinerary>) invocation ->
                                Itinerary.builder()
                                        .itineraryId(invocation.getArgument(0))
                                        .rooms(List.of(ItineraryRoom.builder()
                                                .confirmationId(RoomConfirmation.builder()
                                                        .property("hotelConfirmationId")
                                                        .expedia("partnerConfirmationId")
                                                        .build())
                                                .id(ROOM_ID)
                                                .status(RoomStatus.BOOKED)
                                                .links(RoomLinks.builder()
                                                        .cancel(Link.builder()
                                                                .href(String.format("/%s/itineraries/%s/rooms/%s" +
                                                                                "?token=%s",
                                                                        apiVersion.getValue(), ITINERARY_ID, ROOM_ID,
                                                                        TOKEN))
                                                                .build())
                                                        .build())
                                                .build()))
                                        .build()
                );
    }

    protected void initializeExpediaMocksForOrderConfirmFailed(ApiVersion apiVersion) {
        when(expediaClient.usingApi(any())).thenReturn(expediaClient);
        when(expediaClient.getApiVersion()).thenReturn(apiVersion);
        ReservationResult expediaReservationResult = createExpediaReservationResult();
        when(expediaClient.reserveItinerarySync(any(), any(), any(), any(), any())).thenReturn(
                expediaReservationResult
        );
        when(expediaClient.getItinerarySync(any(), any(), any(), any(), any()))
                .thenReturn(null);
        when(expediaClient.resumeItinerarySync(any(), any(), any(), any(), any()))
                .thenReturn(ResumeReservationStatus.NOT_FOUND);
    }


    protected void initializeExpediaMocksForOrderReservedPaymentFailed(ApiVersion apiVersion) {
        when(expediaClient.usingApi(any())).thenReturn(expediaClient);
        when(expediaClient.getApiVersion()).thenReturn(apiVersion);
        ReservationResult expediaReservationResult = createExpediaReservationResult();
        when(expediaClient.reserveItinerarySync(any(), any(), any(), any(), any())).thenReturn(
                expediaReservationResult
        );
        when(expediaClient.cancelItinerarySync(any(), any(), any(), any(), any()))
                .thenReturn(CancellationStatus.SUCCESS);
    }


    protected void initializeExpediaMocksForOrderReservedAndCancelled(ApiVersion apiVersion) {
        when(expediaClient.usingApi(any())).thenReturn(expediaClient);
        when(expediaClient.getApiVersion()).thenReturn(apiVersion);
        ReservationResult expediaReservationResult = createExpediaReservationResult();
        when(expediaClient.reserveItinerarySync(any(), any(), any(), any(), any())).thenReturn(
                expediaReservationResult
        );
        when(expediaClient.cancelItinerarySync(any(), any(), any(), any(), any()))
                .thenReturn(CancellationStatus.SUCCESS);
    }

    protected Money getOrderBalance(String orderId) {
        return transactionTemplate.execute(tStatus -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            UUID accountId = order.getAccount().getId();
            return accountService.getAccountBalance(accountId);
        });
    }

    protected Money getTrustBalance() {
        return transactionTemplate.execute(ignored -> accountService.getAccountBalance(WellKnownAccount.TRUST.getUuid()));
    }


    protected ReservationResult createExpediaReservationResult() {
        return ReservationResult.builder()
                .itineraryId(ITINERARY_ID)
                .links(HoldItineraryLinks.builder()
                        .resume(Link.builder()
                                .href("/2.4/itineraries/" + ITINERARY_ID + "?token=" + TOKEN)
                                .build())
                        .build())
                .build();
    }

    @Data
    @Builder
    public static class PromoParams {
        private Tuple2<EPromoCodeNominalType, BigDecimal> promo = HUNDRED_RUBLES_PROMO;
        private PromoCodeBehaviourOverride promoCodeBehaviourOverride;
    }

    @Data
    @Builder
    public static class CreateOrderParams {
        public EServiceType serviceType;
        public String payloadName;
        public THotelTestContext testContext;
        public String promoCode;
        public boolean useDeferred;
        public boolean usePostPay;
        public boolean useAutopay;
        public Function<String, String> payloadCustomizer;
        public TPaymentTestContext paymentTestContext;
        public PromoParams promo;

        public static CreateOrderParams from(EServiceType serviceType, String payloadName,
                                             THotelTestContext testContext,
                                             String promoCode, boolean useDeferred, boolean useAutopay,
                                             Function<String, String> payloadCustomizer,
                                             TPaymentTestContext paymentTestContext, PromoParams promo) {
            return CreateOrderParams.builder()
                    .serviceType(serviceType)
                    .payloadName(payloadName)
                    .testContext(testContext)
                    .promoCode(promoCode)
                    .useDeferred(useDeferred)
                    .useAutopay(useAutopay)
                    .payloadCustomizer(payloadCustomizer)
                    .paymentTestContext(paymentTestContext)
                    .promo(promo)
                    .build();
        }
    }

    protected String createOrder(CreateOrderParams params) {
        Preconditions.checkState(params.getPayloadName() != null, "Payload name is required");
        Preconditions.checkState(params.getServiceType() != null, "Service type is required");

        if (params.getTestContext() == null) {
            params.setTestContext(getAllSuccessfulContext());
        }

        if (params.getPromo() != null) {
            var promo = params.getPromo();
            if (promo.getPromo() == null) {
                promo.setPromo(HUNDRED_RUBLES_PROMO);
            }

            createSuccessPromoCode(promo.getPromo(), promo.getPromoCodeBehaviourOverride());
            params.setPromoCode(promoCodeString);
        }

        var resourceName = String.format("integration/hotels/%s.json", params.getPayloadName());
        String payload = TestResources.readResource(resourceName);
        if (params.getPayloadCustomizer() != null) {
            payload = params.getPayloadCustomizer().apply(payload);
        }
        var builder = TCreateOrderReq.newBuilder()
                .setOrderType(EOrderType.OT_HOTEL_EXPEDIA)
                .setDeduplicationKey(UUID.randomUUID().toString())
                .addCreateServices(
                        TCreateServiceReq.newBuilder()
                                .setServiceType(params.getServiceType())
                                .setHotelTestContext(params.getTestContext())
                                .setSourcePayload(TJson.newBuilder().setValue(payload))
                )
                .setCurrency(ECurrency.C_RUB)
                .setOwner(TUserInfo.newBuilder()
                        .setEmail("test@test.com")
                        .setPhone("+79111111111")
                        .setPassportId(PASSPORT_ID)
                        .setYandexUid(YANDEX_UID)
                        .setIp(uniqueTestIp)
                );
        if (params.getPromoCode() != null) {
            builder.addPromoCodeStrings(params.getPromoCode());
        }
        if (params.getPaymentTestContext() != null) {
            builder.setPaymentTestContext(params.getPaymentTestContext());
        }
        builder.setUseDeferredPayment(params.isUseDeferred());
        builder.setUsePostPay(params.isUsePostPay());
        var resp = client.createOrder(builder.build());
        String orderId = resp.getNewOrder().getOrderId();
        assertOrderState(orderId, EHotelOrderState.OS_NEW);

        return orderId;
    }

    protected void assertOrderState(String orderId, EHotelOrderState state) {
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getHotelOrderState()).isEqualTo(state);
    }

    protected <T extends HotelItinerary> Function<String, String> payloadCustomizer(Class<T> payloadType,
                                                                                    Consumer<T> customizer) {
        return payloadJson -> {
            T itinerary = ProtoUtils.fromTJson(TJson.newBuilder().setValue(payloadJson).build(), payloadType);
            customizer.accept(itinerary);
            return ProtoUtils.toTJson(itinerary).getValue();
        };
    }

    protected TrustUserInfo uniqueTestTrustUserInfo() {
        return new TrustUserInfo(PASSPORT_ID, uniqueTestIp);
    }

    protected List<FinancialEvent> getFinancialEventsOfOrder(String orderId) {
        return transactionTemplate.execute(txStatus ->
                financialEventRepository.findAll().stream()
                        .filter(e -> e.getOrder() != null && e.getOrder().getId().toString().equals(orderId))
                        .collect(Collectors.toList()));
    }

    protected PromoCode createSuccessPromoCode(Tuple2<EPromoCodeNominalType, BigDecimal> promo) {
        return createSuccessPromoCode(promo, null);
    }

    protected PromoCode createSuccessPromoCode(Tuple2<EPromoCodeNominalType, BigDecimal> promo,
                                               PromoCodeBehaviourOverride behaviourOverride) {
        return createSuccessPromoCode(promo, behaviourOverride, null);
    }

    protected PromoCode createSuccessPromoCode(Tuple2<EPromoCodeNominalType, BigDecimal> promo,
                                               PromoCodeBehaviourOverride behaviourOverride,
                                               BigDecimal budget) {
        promoCodeString = PromoCodeUnifier.unifyCode("PROMO_CODE_HOFT_" + System.nanoTime());
        return transactionTemplate.execute(ignored -> {
            PromoAction promoAction = new PromoAction();
            promoAction.setId(UUID.randomUUID());
            promoAction.setName("SUCCESS ACTION");
            promoAction.setDiscountApplicationConfig(new DiscountApplicationConfig());
            if (budget != null) {
                promoAction.setInitialBudget(budget);
                promoAction.setRemainingBudget(budget);
            }
            promoAction = promoActionRepository.save(promoAction);

            PromoCode promoCode = new PromoCode();
            promoCode.setId(UUID.randomUUID());
            promoCode.setCode(promoCodeString);
            promoCode.setNominalType(promo.get1());
            promoCode.setNominal(promo.get2());
            promoCode.setPromoAction(promoAction);
            promoCode.setBehaviourOverride(behaviourOverride);

            promoCodeRepository.save(promoCode);
            return promoCode;
        });
    }

    protected void setDeferredPaymentEnabled(String orderId) {
        client.setDeferredPayment(TSetDeferredPaymentReq.newBuilder()
                .setEnabled(true)
                .setOrderId(orderId)
                .setCallId(UUID.randomUUID().toString())
                .build());
    }

    protected void checkPromoCodeActivated(int times) {
        transactionTemplate.execute(ignored -> {
            PromoCodeActivation activation =
                    promoCodeActivationRepository.lookupActivationByCodeAndPassportId(
                            PASSPORT_ID, promoCodeString
                    );
            assertThat(activation.getTimesUsed()).isEqualTo(times);
            return null;
        });
    }

    protected enum PaymentsBehaviour {
        DO_NOT_CLEAR, CLEAR
    }

    protected enum PromoCodeCheckBehaviour {
        DO_NOT_CHECK, CHECK_APPLIED, CHECK_NOT_APPLICABLE, CHECK_ALREADY_APPLIED
    }

}
