package ru.yandex.travel.orders.integration.hotels

import lombok.SneakyThrows
import org.assertj.core.api.Assertions
import org.mockito.ArgumentMatcher
import org.mockito.ArgumentMatchers.any
import org.mockito.Mockito.`when`
import org.mockito.kotlin.argThat
import org.springframework.transaction.support.TransactionTemplate

import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient
import ru.yandex.travel.hotels.common.partners.bronevik.model.FaultCode.INTERNAL_ERROR
import ru.yandex.travel.hotels.common.partners.bronevik.model.FaultCode.INVALID_OFFER_CODE
import ru.yandex.travel.hotels.common.partners.bronevik.model.FaultCode.SOLD_OUT
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus.AWAITING_CANCELLATION
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus.AWAITING_CONFIRMATION
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus.CANCELLED_WITH_PENALTY
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus.CONFIRMED
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus.NOT_CONFIRMED
import ru.yandex.travel.hotels.common.partners.bronevik.utils.MockUtils

class BronevikMockHelper internal constructor(
    private val bronevikClient: BronevikClient,
    private val partnerConfigRepository: BillingPartnerConfigRepository,
    private val transactionTemplate: TransactionTemplate
) {
    @SneakyThrows
    fun initializeMockForOrderConfirmed() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val createOrderResponse = MockUtils.mockCreateOrderResponse(28117.15f, CONFIRMED)
        val getOrderResponse = MockUtils.mockGetOrderResponse(28117.15f, CONFIRMED)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any())).thenReturn(createOrderResponse)
        `when`(bronevikClient.getOrderSync(any(), any())).thenReturn(getOrderResponse)
    }

    @SneakyThrows
    fun initializeMockForOrderWithChildrenConfirmed() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val createOrderResponse = MockUtils.mockCreateOrderResponse(28117.15f, CONFIRMED)
        val getOrderResponse = MockUtils.mockGetOrderResponse(28117.15f, CONFIRMED)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), argThat( ArgumentMatcher {
            Assertions.assertThat(it.size).isEqualTo(2)
            Assertions.assertThat(it.filter { child -> child.age == 1 }.size).isEqualTo(1)
            Assertions.assertThat(it.filter { child -> child.age == 1 }[0].count).isEqualTo(2)
            Assertions.assertThat(it.filter { child -> child.age == 10 }.size).isEqualTo(1)
            Assertions.assertThat(it.filter { child -> child.age == 10 }[0].count).isEqualTo(1)
            return@ArgumentMatcher true
        }))).thenReturn(createOrderResponse)
        `when`(bronevikClient.getOrderSync(any(), any())).thenReturn(getOrderResponse)
    }

    @SneakyThrows
    fun initializeMockForPriceMismatchInCheckingPriceState() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28300.00f)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
    }

    @SneakyThrows
    fun initializeMockForSoldOutInCheckingPriceState() {
        mockBillingConfig()

        val getHotelOfferPricingFault = MockUtils.mockGetHotelOfferPricingFault(SOLD_OUT.value)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenThrow(getHotelOfferPricingFault)
    }

    @SneakyThrows
    fun initializeMockForPriceMismatchInConfirmingState() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val getHotelOfferPricingResponseAfterPayment = MockUtils.mockGetHotelOfferPricingResponse(28300.00f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
            .thenReturn(getHotelOfferPricingResponseAfterPayment)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
    }

    @SneakyThrows
    fun initializeMockForBadRequestExceptionInCheckingPriceState() {
        mockBillingConfig()

        val getHotelOfferPricingFault = MockUtils.mockGetHotelOfferPricingFault(INVALID_OFFER_CODE.value)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenThrow(getHotelOfferPricingFault)
    }

    @SneakyThrows
    fun initializeMockForSoldOutInConfirmingState() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val getHotelOfferPricingFault = MockUtils.mockGetHotelOfferPricingFault(SOLD_OUT.value)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
            .thenThrow(getHotelOfferPricingFault)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
    }

    @SneakyThrows
    fun initializeMockForBadRequestExceptionInConfirmingState() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val getHotelOfferPricingFault = MockUtils.mockGetHotelOfferPricingFault(INVALID_OFFER_CODE.value)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
            .thenThrow(getHotelOfferPricingFault)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
    }

    @SneakyThrows
    fun initializeMockForOrderConfirmedAndRefund(cancelStatus: OrderStatus) {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val createOrderResponse = MockUtils.mockCreateOrderResponse(28117.15f, CONFIRMED)
        val getOrderResponse = MockUtils.mockGetOrderResponse(28117.15f, CONFIRMED)
        val getOrderResponseAfterCancel = MockUtils.mockGetOrderResponse(28117.15f, cancelStatus)
        val cancelOrderResponse = MockUtils.mockCancelOrderResponse()

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any())).thenReturn(createOrderResponse)
        `when`(bronevikClient.getOrderSync(any(), any()))
            .thenReturn(getOrderResponse)
            .thenReturn(getOrderResponseAfterCancel)
        `when`(bronevikClient.cancelOrderSync(any(), any())).thenReturn(cancelOrderResponse)
    }

    @SneakyThrows
    fun initializeMockForSoldOutOnPartnerOrderCreation() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val createOrderFault = MockUtils.mockCreateOrderFault(SOLD_OUT.value)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any())).thenThrow(createOrderFault)
    }

    @SneakyThrows
    fun initializeMockForBadRequestExceptionOnPartnerOrderCreation() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val createOrderFault = MockUtils.mockCreateOrderFault(INVALID_OFFER_CODE.value)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any())).thenThrow(createOrderFault)
    }

    @SneakyThrows
    fun initializeMockForCancelWithAwaitingCancellation() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val createOrderResponse = MockUtils.mockCreateOrderResponse(28117.15f, CONFIRMED)
        val cancelOrderResponse = MockUtils.mockCancelOrderResponse()
        val firstGetOrderResponse = MockUtils.mockGetOrderResponse(28117.15f, AWAITING_CANCELLATION)
        val secondGetOrderResponse = MockUtils.mockGetOrderResponse(28117.15f, CANCELLED_WITH_PENALTY)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any()))
            .thenReturn(createOrderResponse)
        `when`(bronevikClient.cancelOrderSync(any(), any()))
            .thenReturn(cancelOrderResponse)
        `when`(bronevikClient.getOrderSync(any(), any()))
            .thenReturn(firstGetOrderResponse)
            .thenReturn(secondGetOrderResponse)
    }

    @SneakyThrows
    fun initializeMockForInternalExceptionOnCreation() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val createOrderFault = MockUtils.mockCreateOrderFault(INTERNAL_ERROR.value)
        val offerCode = "T1IyNDQjMTM2OTcjZG91YmxlIzE0NjkjMjAyMi0wNS0wMiMyMDIyLTA1LTA2IzIjMCMwIzEw"
        val firstSearchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val secondSearchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, offerCode)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any())).thenThrow(createOrderFault)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any()))
            .thenReturn(firstSearchOrdersResponse)
            .thenReturn(secondSearchOrdersResponse)
    }

    @SneakyThrows
    fun initializeMockForOrderCreationWithNotConfirmedStatus() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val createOrderResponse = MockUtils.mockCreateOrderResponse(28117.15f, NOT_CONFIRMED)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any()))
            .thenReturn(createOrderResponse)
    }

    @SneakyThrows
    fun initializeMockForOrderCreationWithConfirmingStatus() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val searchOrdersResponse = MockUtils.mockSearchOrderResponse(28117.15f, CONFIRMED, null)
        val createOrderResponse = MockUtils.mockCreateOrderResponse(28117.15f, AWAITING_CONFIRMATION)
        val cancelOrderResponse = MockUtils.mockCancelOrderResponse()
        val getOrderResponse = MockUtils.mockGetOrderResponse(28117.15f, AWAITING_CONFIRMATION)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.searchOrdersByReferenceIdSync(any(), any())).thenReturn(searchOrdersResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any())).thenReturn(createOrderResponse)
        `when`(bronevikClient.cancelOrderSync(any(), any())).thenReturn(cancelOrderResponse)
        `when`(bronevikClient.getOrderSync(any(), any())).thenReturn(getOrderResponse)
    }

    @SneakyThrows
    fun initializeMockForOrderCreationWithPriceMismatch() {
        mockBillingConfig()

        val getHotelOfferPricingResponse = MockUtils.mockGetHotelOfferPricingResponse(28117.15f)
        val createOrderResponse = MockUtils.mockCreateOrderResponse(30117.15f, CONFIRMED)
        val cancelOrderResponse = MockUtils.mockCancelOrderResponse()
        val getOrderResponse = MockUtils.mockGetOrderResponse(30117.15f, CONFIRMED)

        mockCommonMethods()
        `when`(bronevikClient.getHotelOfferPricingSync(any(), any(), any(), any(), any(), any()))
            .thenReturn(getHotelOfferPricingResponse)
        `when`(bronevikClient.createOrderSync(any(), any(), any(), any(), any(), any(), any()))
            .thenReturn(createOrderResponse)
        `when`(bronevikClient.cancelOrderSync(any(), any())).thenReturn(cancelOrderResponse)
        `when`(bronevikClient.getOrderSync(any(), any())).thenReturn(getOrderResponse)
    }

    private fun mockCommonMethods() {
        `when`(bronevikClient.withCallContext(any())).thenReturn(bronevikClient)
    }

    private fun mockBillingConfig() {
        transactionTemplate.execute<Any> {
            val billingConfig = BillingPartnerConfig.builder()
                .billingClientId(-1L)
                .agreementActive(true)
                .generateTransactions(false)
                .exportToYt(false)
                .synchronizeAgreement(false)
                .build()
            partnerConfigRepository.save(billingConfig)
            null
        }
    }
}
