package ru.yandex.travel.orders.integration.hotels

import org.assertj.core.api.Assertions
import org.junit.Before
import org.junit.Test
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.test.mock.mockito.MockBean
import org.springframework.boot.test.mock.mockito.SpyBean
import ru.yandex.travel.commons.proto.ProtoUtils
import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary
import ru.yandex.travel.hotels.common.orders.CancellationDetails.Reason.INVALID_INPUT
import ru.yandex.travel.hotels.common.orders.CancellationDetails.Reason.PRICE_CHANGED
import ru.yandex.travel.hotels.common.orders.CancellationDetails.Reason.SOLD_OUT
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus.CANCELLED_WITHOUT_PENALTY
import ru.yandex.travel.hotels.common.partners.bronevik.model.OrderStatus.CANCELLED_WITH_PENALTY
import ru.yandex.travel.hotels.proto.THotelTestContext
import ru.yandex.travel.orders.commons.proto.EServiceType
import ru.yandex.travel.orders.commons.proto.EServiceType.PT_BRONEVIK_HOTEL
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext
import ru.yandex.travel.orders.integration.hotels.AbstractHotelOrderFlowTest.PaymentsBehaviour.CLEAR
import ru.yandex.travel.orders.integration.hotels.AbstractHotelOrderFlowTest.PromoCodeCheckBehaviour.DO_NOT_CHECK
import ru.yandex.travel.orders.proto.TGetOrderInfoReq
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository
import ru.yandex.travel.orders.services.payments.TrustClient
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState.IS_REFUNDED
import java.util.function.Function

class BronevikOrderFlowTests : AbstractHotelOrderFlowTest() {
    private val BRONEVIK_PAYLOAD = "bronevik"
    private val BRONEVIK_CHILDREN_PAYLOAD = "bronevik_children"
    private val BRONEVIK_PAYLOAD_FULLY_REFUNDABLE = "bronevik_fully_refundable"

    @SpyBean(name = "mockTrustClient")
    private lateinit var trustClient: TrustClient

    @MockBean
    private lateinit var bronevikClient: BronevikClient

    @Autowired
    private lateinit var partnerConfigRepository: BillingPartnerConfigRepository

    private lateinit var bronevikMockHelper: BronevikMockHelper

    @Before
    fun setUpMockHelper() {
        bronevikMockHelper = BronevikMockHelper(bronevikClient, partnerConfigRepository, transactionTemplate)
    }

    @Test
    fun testOrderConfirmed() {
        bronevikMockHelper.initializeMockForOrderConfirmed()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirm(orderId, DO_NOT_CHECK)

        val order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())
        Assertions.assertThat(order.result.currentInvoice).isNotNull()
        Assertions.assertThat(order.result.getPayments(0).paidAmount.amount).isEqualTo(2811700)
        Assertions.assertThat(order.result.getPayments(0).totalAmount.amount).isEqualTo(2811700)
        Assertions.assertThat(order.result.zeroFirstPayment).isFalse()

        val itinerary = ProtoUtils.fromTJson(
            order.result.getService(0).serviceInfo.payload,
            BronevikHotelItinerary::class.java
        )
        Assertions.assertThat(itinerary.actualPrice!!.number.toDouble()).isEqualTo(28117.15)
        Assertions.assertThat(itinerary.orderId).isEqualTo(49)

        // TODO(mokosha): Добавить проверки, когда будут известны данные биллинга
       // val orderEvents = getFinancialEventsOfOrder(orderId)
       // Assertions.assertThat(orderEvents).isNotNull()
       // Assertions.assertThat(orderEvents.size).isEqualTo(1)
       // Assertions.assertThat(orderEvents[0].totalAmount).isEqualTo(Money.of(28117.15, "RUB"))
       // Assertions.assertThat(orderEvents[0].type).isEqualTo(PAYMENT)
    }

    @Test
    fun testOrderConfirmedWithChildren() {
        bronevikMockHelper.initializeMockForOrderWithChildrenConfirmed()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_CHILDREN_PAYLOAD
        ))
        confirm(orderId, DO_NOT_CHECK)

        val order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())
        Assertions.assertThat(order.result.currentInvoice).isNotNull()
        Assertions.assertThat(order.result.getPayments(0).paidAmount.amount).isEqualTo(2811700)
        Assertions.assertThat(order.result.getPayments(0).totalAmount.amount).isEqualTo(2811700)
        Assertions.assertThat(order.result.zeroFirstPayment).isFalse()

        val itinerary = ProtoUtils.fromTJson(
            order.result.getService(0).serviceInfo.payload,
            BronevikHotelItinerary::class.java
        )
        Assertions.assertThat(itinerary.actualPrice!!.number.toDouble()).isEqualTo(28117.15)
        Assertions.assertThat(itinerary.orderId).isEqualTo(49)
    }

    @Test
    fun testOrderRefundWithPenalty() {
        bronevikMockHelper.initializeMockForOrderConfirmedAndRefund(CANCELLED_WITH_PENALTY)
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmAndRefund(orderId, CLEAR)

        val orderMoney = getOrderBalance(orderId)
        Assertions.assertThat(orderMoney.number.toDouble()).isEqualTo(12468.00)

        // TODO(mokosha): Добавить проверки, когда будут известны данные биллинга
        // val orderEvents = getFinancialEventsOfOrder(orderId)
        // Assertions.assertThat(orderEvents).isNotNull()
        // val paymentEvents = orderEvents.filter { it.type == PAYMENT }
        // val refundEvents = orderEvents.filter { it.type == REFUND }
        // Assertions.assertThat(paymentEvents.size).isEqualTo(1)
        // Assertions.assertThat(refundEvents.size).isEqualTo(1)
        // Assertions.assertThat(paymentEvents[0].totalAmount).isEqualTo(Money.of(28117.15, "RUB"))
        // Assertions.assertThat(refundEvents[0].totalAmount).isEqualTo(Money.of(15649.15, "RUB"))
    }

    @Test
    fun testOrderRefundWithoutPenalty() {
        bronevikMockHelper.initializeMockForOrderConfirmedAndRefund(CANCELLED_WITHOUT_PENALTY)
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD_FULLY_REFUNDABLE
        ))

        confirm(orderId, DO_NOT_CHECK)
        clear(orderId)
        refund(orderId, CLEAR, IS_REFUNDED)

        val orderMoney = getOrderBalance(orderId)
        Assertions.assertThat(orderMoney.number.toDouble()).isEqualTo(0.0)

        // TODO(mokosha): Добавить проверки, когда будут известны данные биллинга
        // val orderEvents = getFinancialEventsOfOrder(orderId)
        // Assertions.assertThat(orderEvents).isNotNull()
        // val paymentEvents = orderEvents.filter { it.type == PAYMENT }
        // val refundEvents = orderEvents.filter { it.type == REFUND }
        // Assertions.assertThat(paymentEvents.size).isEqualTo(1)
        // Assertions.assertThat(refundEvents.size).isEqualTo(1)
        // Assertions.assertThat(paymentEvents[0].totalAmount).isEqualTo(Money.of(28117.15, "RUB"))
        // Assertions.assertThat(refundEvents[0].totalAmount).isEqualTo(Money.of(28117.15, "RUB"))
    }

    @Test
    fun testPriceMismatchOnReservation() {
        bronevikMockHelper.initializeMockForPriceMismatchInCheckingPriceState()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        failOnReservation(orderId, PRICE_CHANGED)
    }

    @Test
    fun testSoldOutOnReservation() {
        bronevikMockHelper.initializeMockForSoldOutInCheckingPriceState()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        failOnReservation(orderId, SOLD_OUT)
    }

    @Test
    fun testBadRequestExceptionOnReservation() {
        bronevikMockHelper.initializeMockForBadRequestExceptionInCheckingPriceState()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        failOnReservation(orderId, INVALID_INPUT)
    }

    @Test
    fun testReserveAndCancel() {
        bronevikMockHelper.initializeMockForOrderConfirmed()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        reserveAndCancel(orderId, DO_NOT_CHECK)
    }

    @Test
    fun testPriceMismatchAfterPayment() {
        bronevikMockHelper.initializeMockForPriceMismatchInConfirmingState()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmFailed(orderId)

        val order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())
        val cancellationDetails = getCancellationDetails(order.result.getService(0).serviceInfo)
        Assertions.assertThat(cancellationDetails).isNotNull()
        Assertions.assertThat(cancellationDetails.reason == PRICE_CHANGED).isTrue()
    }

    @Test
    fun testSoldOutAfterPayment() {
        bronevikMockHelper.initializeMockForSoldOutInConfirmingState()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmFailed(orderId)

        val order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())
        val cancellationDetails = getCancellationDetails(order.result.getService(0).serviceInfo)
        Assertions.assertThat(cancellationDetails).isNotNull()
        Assertions.assertThat(cancellationDetails.reason == SOLD_OUT).isTrue()
    }

    @Test
    fun testBadRequestExceptionAfterPayment() {
        bronevikMockHelper.initializeMockForBadRequestExceptionInConfirmingState()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmFailed(orderId)

        val order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())
        val cancellationDetails = getCancellationDetails(order.result.getService(0).serviceInfo)
        Assertions.assertThat(cancellationDetails).isNotNull()
        Assertions.assertThat(cancellationDetails.reason == INVALID_INPUT).isTrue()
    }

    @Test
    fun testSoldOutOnPartnerOrderCreation() {
        bronevikMockHelper.initializeMockForSoldOutOnPartnerOrderCreation()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmFailed(orderId)

        val order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())
        val cancellationDetails = getCancellationDetails(order.result.getService(0).serviceInfo)
        Assertions.assertThat(cancellationDetails).isNotNull()
        Assertions.assertThat(cancellationDetails.reason == SOLD_OUT).isTrue()
    }

    @Test
    fun testBadRequestExceptionOnPartnerOrderCreation() {
        bronevikMockHelper.initializeMockForBadRequestExceptionOnPartnerOrderCreation()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmFailed(orderId)

        val order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())
        val cancellationDetails = getCancellationDetails(order.result.getService(0).serviceInfo)
        Assertions.assertThat(cancellationDetails).isNotNull()
        Assertions.assertThat(cancellationDetails.reason == INVALID_INPUT).isTrue()
    }

    @Test
    fun testCancelWithAwaitingCancellationStatus() {
        bronevikMockHelper.initializeMockForCancelWithAwaitingCancellation()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmAndRefund(orderId, CLEAR)
    }

    @Test
    fun testFindOrderAfterInternalPartnerExceptionOnCreation() {
        bronevikMockHelper.initializeMockForInternalExceptionOnCreation()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirm(orderId, DO_NOT_CHECK)

        val order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build())
        val itinerary = ProtoUtils.fromTJson(
            order.result.getService(0).serviceInfo.payload,
            BronevikHotelItinerary::class.java
        )
        Assertions.assertThat(itinerary.orderId).isEqualTo(123)
    }

    @Test
    fun testCancelOrderWithNotConfirmedStatus() {
        bronevikMockHelper.initializeMockForOrderCreationWithNotConfirmedStatus()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmFailed(orderId)
    }

    @Test
    fun testCancelOrderWithConfirmingStatus() {
        bronevikMockHelper.initializeMockForOrderCreationWithConfirmingStatus()
        val orderId = createOrder(createOrderParams(
            serviceType = PT_BRONEVIK_HOTEL,
            payloadName = BRONEVIK_PAYLOAD
        ))

        confirmFailed(orderId)
    }

    private fun createOrderParams(serviceType: EServiceType, payloadName: String, testContext: THotelTestContext? = null,
        promoCode: String? = null, useDeferred: Boolean = false, useAutopay: Boolean = false,
        payloadCustomizer: Function<String, String>? = null, paymentTestContext: TPaymentTestContext? = null,
        promo: PromoParams? = null): CreateOrderParams {
        return CreateOrderParams.from(serviceType, payloadName, testContext, promoCode, useDeferred, useAutopay,
            payloadCustomizer, paymentTestContext, promo)
    }
}
