package ru.yandex.travel.orders.integration.hotels;

import java.time.Instant;
import java.time.LocalDate;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import io.grpc.StatusRuntimeException;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.data.repository.CrudRepository;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.AbstractGrpcTest;
import ru.yandex.travel.orders.admin.proto.TGetOrderReq;
import ru.yandex.travel.orders.admin.proto.TGetOrderRsp;
import ru.yandex.travel.orders.commons.proto.ESnippet;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.FiscalReceipt;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.SimpleTrustRefund;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.grpc.OrdersNoAuthService;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.OrderNoAuthInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TGenerateBusinessTripPdfReq;
import ru.yandex.travel.orders.proto.TGenerateBusinessTripPdfRsp;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.repository.AuthorizedUserRepository;
import ru.yandex.travel.orders.repository.FiscalReceiptRepository;
import ru.yandex.travel.orders.repository.InvoiceRepository;
import ru.yandex.travel.orders.repository.OrderAggregateStateRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TrustRefundRepository;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.services.payments.InvoicePaymentFlags;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentReceiptResponse;
import ru.yandex.travel.orders.services.pdfgenerator.PdfGeneratorService;
import ru.yandex.travel.orders.services.pdfgenerator.PdfNotFoundException;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfStateResponse;
import ru.yandex.travel.orders.test.TestOrderFactory;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherState;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherType;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.catchThrowableOfType;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.matches;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;


@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "single-node.auto-start=true",
                "voucher-workflow.check-created-period=100ms",
                "voucher-workflow.check-created-timeout=10s",
                "voucher-workflow.check-state-task.initial-start-delay=100ms",
                "voucher-workflow.check-state-task.schedule-rate=100ms",
                "documents.hotel-business-trip.min-days-after-checkout=3",
                "trust-db-mock.enabled=false",
                "order-aggregate-state-refresh.task-processor.initial-start-delay=100d"  // disable task
        }
)
@DirtiesContext(classMode = DirtiesContext.ClassMode.AFTER_EACH_TEST_METHOD)
@Slf4j
public class BusinessTripDocGenerationTest extends AbstractGrpcTest {
    private static final String PURCHASE_TOKEN = "a08bd56c723f104fd745f21bff0559b8";
    private static final Integer RECEIPT_NUMBER = 456;

    public OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub orcApi;
    public OrderNoAuthInterfaceV1Grpc.OrderNoAuthInterfaceV1BlockingStub orcNoAuthApi;

    @Autowired
    protected OrdersService ordersService;

    @Autowired
    protected OrdersNoAuthService ordersNoAuthService;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private AuthorizedUserRepository authUserRepository;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private WorkflowRepository workflowRepository;

    @Autowired
    private InvoiceRepository invoiceRepository;

    @Autowired
    private TrustRefundRepository trustRefundRepository;

    @Autowired
    private FiscalReceiptRepository fiscalReceiptRepository;

    @Autowired
    private VoucherRepository voucherRepository;

    @Autowired
    private OrderAggregateStateRepository orderAggregateStateRepository;

    @MockBean
    private PdfGeneratorService pdfGeneratorService;

    @MockBean
    private TrustClient trustClient;

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trustClient) {
            return paymentProfile -> trustClient;
        }
    }

    private void clearRepositories() {
        log.info("Clearing repositories");
        transactionTemplate.execute(
                s -> {
                    List.of(
                            trustRefundRepository,
                            orderAggregateStateRepository,
                            voucherRepository,
                            invoiceRepository,
                            orderRepository,
                            authUserRepository
                    ).forEach(CrudRepository::deleteAll);
                    return null;
                }
        );
    }

    @Before
    public void setUp() {
        clearRepositories();

        mockDefaultUserLoggedOut();
        orcApi = IntegrationUtils.createServerAndBlockingStub(cleanupRule, ordersService);
        orcNoAuthApi = IntegrationUtils.createOrderNoAuthServiceAndStub(cleanupRule, ordersNoAuthService);
    }

    @After
    public void tearDown() {
        clearRepositories();
    }

    @Test
    public void testHotelBusinessTripDoc() {
        var order = createHotelOrder();

        String docFileName = String.format("hotels/business_trip_%s.pdf", order.getPrettyId());
        String generatedUrl = "https://fakes3.mds.yandex.net/hotels/businesstrip42.pdf";

        AtomicBoolean voucherStateCreated = new AtomicBoolean(false);
        when(pdfGeneratorService.getState(any())).thenAnswer(rq -> {
            if (voucherStateCreated.get() && rq.getArgument(0, String.class).equals(docFileName)) {
                return new PdfStateResponse(generatedUrl, Instant.now());
            }
            throw new PdfNotFoundException("not found");
        });

        TGetOrderInfoRsp orderInfoResp = orcApi.getOrderInfo(
                TGetOrderInfoReq.newBuilder().setOrderId(order.getId().toString()).build());
        assertThat(orderInfoResp.getResult().getCanGenerateBusinessTripDoc()).isTrue();

        callGenerateBusinessTripPdf(order);

        TestUtils.waitForState("Voucher CREATING", () -> transactionTemplate.execute(
                s -> voucherRepository.findAllByOrderId(order.getId()).get(0).getState() == EVoucherState.VS_CREATING));
        verify(pdfGeneratorService, atLeastOnce()).generateHotelsBusinessTripDoc(argThat(
                rq -> rq.getOrderId().equals(order.getId().toString()) && rq.getFileName().equals(docFileName)));

        orderInfoResp = orcApi.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(order.getId().toString()).build());
        assertThat(orderInfoResp.getResult().getBusinessTripDocUrl()).isEqualTo("");

        // check right receipt data for generation
        when(trustClient.getReceipt(
                matches("^" + PURCHASE_TOKEN + "$"),
                any(TrustUserInfo.class)
        )).thenAnswer(rq ->
                TrustPaymentReceiptResponse.builder().documentIndex(RECEIPT_NUMBER).shiftNumber(42).build()
        );
        TGetOrderRsp getOrderResp = orcNoAuthApi.getOrder(
                TGetOrderReq.newBuilder()
                        .setOrderId(order.getId().toString())
                        .addSnippet(ESnippet.S_PRIVATE_INFO)
                        .addSnippet(ESnippet.S_RECEIPTS_DATA)
                        .build());
        assertThat(
                getOrderResp.getOrderInfo().getInvoice(0).getFiscalReceipt(0).getDocumentIndex()
        ).isEqualTo(RECEIPT_NUMBER);

        voucherStateCreated.set(true);
        TestUtils.waitForState("Voucher CREATED", () -> transactionTemplate.execute(
                s -> {
                    var voucher = voucherRepository.findAllByOrderId(order.getId()).get(0);
                    return voucher.getState() == EVoucherState.VS_CREATED && voucher.getVoucherUrl().equals(generatedUrl);
                }));

        TestUtils.waitForState("businessTripDocUrl is set on order", () -> transactionTemplate.execute(
                s -> {
                    var foundOrder = orderRepository.getOne(order.getId());
                    return foundOrder != null && foundOrder.getBusinessTripDocUrl() != null &&
                            foundOrder.getBusinessTripDocUrl().equals(generatedUrl);
                }));

        orderInfoResp = orcApi.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(order.getId().toString()).build());
        assertThat(orderInfoResp.getResult().getBusinessTripDocUrl()).isEqualTo(generatedUrl);
    }

    @Test
    public void testGenerationFailBecauseOfTrust() {
        var order = createHotelOrder();
        String generatedUrl = "https://fakes3.mds.yandex.net/hotels/businesstrip42.pdf";
        String docFileName = String.format("hotels/business_trip_%s.pdf", order.getPrettyId());
        AtomicBoolean voucherStateCreated = new AtomicBoolean(false);
        when(pdfGeneratorService.getState(any())).thenAnswer(rq -> {
            if (voucherStateCreated.get() && rq.getArgument(0, String.class).equals(docFileName)) {
                return new PdfStateResponse(generatedUrl, Instant.now());
            }
            throw new PdfNotFoundException("not found");
        });
        when(trustClient.getReceipt(
                matches("^" + PURCHASE_TOKEN + "$"),
                any(TrustUserInfo.class)
        )).thenAnswer(rq -> {
                throw new HttpApiRetryableException("Trust failed somehow");
        });

        callGenerateBusinessTripPdf(order);

        TestUtils.waitForState("Voucher CREATING", () -> transactionTemplate.execute(
                s -> voucherRepository.findAllByOrderId(order.getId()).get(0).getState() == EVoucherState.VS_CREATING));

        TestUtils.waitForState("Voucher FAILED", () -> transactionTemplate.execute(
                s -> {
                    var voucher = voucherRepository.findAllByOrderId(order.getId()).get(0);
                    return voucher.getState() == EVoucherState.VS_FAILED;
                }));
        Voucher failedVocuher = voucherRepository.findAllByOrderId(order.getId()).get(0);

        // имеем сфейленный ваучер. Надо проверить, что создастся новый при повторном вызове
        when(trustClient.getReceipt(
                matches("^" + PURCHASE_TOKEN + "$"),
                any(TrustUserInfo.class)
        )).thenAnswer(rq ->
                TrustPaymentReceiptResponse.builder().documentIndex(RECEIPT_NUMBER).shiftNumber(42).build()
        );

        callGenerateBusinessTripPdf(order);

        TestUtils.waitForState("Voucher CREATING", () -> transactionTemplate.execute(
                s -> {
                    var vouchers = voucherRepository.findForOrderByTypesAndStates(
                            order.getId(), EVoucherType.VT_HOTELS_BUSINESS_TRIP, EnumSet.of(EVoucherState.VS_FAILED)
                    );
                    assertThat(vouchers.size()).isEqualTo(1);
                    assertThat(vouchers.get(0).getId()).isEqualTo(failedVocuher.getId());

                    vouchers = voucherRepository.findForOrderByTypesAndStates(
                            order.getId(), EVoucherType.VT_HOTELS_BUSINESS_TRIP, EnumSet.of(
                                    EVoucherState.VS_NEW, EVoucherState.VS_CREATING)
                    );
                    assertThat(vouchers.size()).isEqualTo(1);
                    assertThat(vouchers.get(0).getId()).isNotEqualTo(failedVocuher.getId());

                    return true;
                }));
        voucherStateCreated.set(true);

        TestUtils.waitForState("businessTripDocUrl is set on order", () -> transactionTemplate.execute(
                s -> {
                    var foundOrder = orderRepository.getOne(order.getId());
                    return foundOrder != null && foundOrder.getBusinessTripDocUrl() != null &&
                            foundOrder.getBusinessTripDocUrl().equals(generatedUrl);
                }));
    }

    @Test
    public void testTooEarlyToStartGeneration() {
        var order = createHotelOrder(-2L, false);
        TGetOrderInfoRsp orderInfoResp = orcApi.getOrderInfo(
                TGetOrderInfoReq.newBuilder().setOrderId(order.getId().toString()).build());
        assertThat(orderInfoResp.getResult().getCanGenerateBusinessTripDoc()).isFalse();

        StatusRuntimeException throwable = catchThrowableOfType(
                () -> callGenerateBusinessTripPdf(order), StatusRuntimeException.class);
        assertThat(throwable.getStatus().getDescription())
                .contains("FAILED_PRECONDITION").contains("days must be passed after checkout");
    }

    @Test
    public void testOrderRefunded() {
        var order = createHotelOrder(-3L, true);

        TGetOrderInfoRsp orderInfoResp = orcApi.getOrderInfo(
                TGetOrderInfoReq.newBuilder().setOrderId(order.getId().toString()).build());
        assertThat(orderInfoResp.getResult().getCanGenerateBusinessTripDoc()).isFalse();

        StatusRuntimeException throwable = catchThrowableOfType(
                () -> callGenerateBusinessTripPdf(order), StatusRuntimeException.class);
        assertThat(throwable.getStatus().getDescription())
                .contains("FAILED_PRECONDITION")
                .contains("order must not be refunded");
    }

    private TGenerateBusinessTripPdfRsp callGenerateBusinessTripPdf(HotelOrder order) throws StatusRuntimeException {
        return orcApi.generateBusinessTripPdf(
                TGenerateBusinessTripPdfReq.newBuilder().setOrderId(order.getId().toString()).build());
    }

    private HotelOrder createHotelOrder() {
        return createHotelOrder(-3L, false);
    }

    /**
     * @param checkoutDateShift - выезд из отеля через столько дней от текущей даты
     */
    private HotelOrder createHotelOrder(Long checkoutDateShift, boolean withRefund) {
        return transactionTemplate.execute(ignored -> {
            TravellineOrderItem orderItem = TestOrderFactory.createTravellineOrderItem();

            LocalDate checkinDate = LocalDate.now().plusDays(checkoutDateShift);
            orderItem.getItinerary().setOrderDetails(OrderDetails.builder()
                    .checkinDate(checkinDate.minusDays(2))
                    .checkoutDate(checkinDate)
                    .build());
            orderItem.getItinerary().setFiscalPrice(Money.of(1042, "RUB"));

            HotelOrder order = TestOrderFactory.createHotelOrder(orderItem);
            Workflow orderItemWorkflow = Workflow.createWorkflowForEntity(orderItem, order.getId());
            orderItemWorkflow.setSupervisorId(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid());
            workflowRepository.saveAndFlush(orderItemWorkflow);

            Workflow orderWorkflow = Workflow.createWorkflowForEntity(order);
            orderWorkflow.setSupervisorId(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid());
            workflowRepository.saveAndFlush(orderWorkflow);
            order = orderRepository.saveAndFlush(order);

            var authUser = AuthorizedUser.createGuest(
                    order.getId(), DEFAULT_SESSION_KEY, DEFAULT_TEST_YANDEX_UID, AuthorizedUser.OrderUserRole.OWNER);
            authUserRepository.saveAndFlush(authUser);

            TrustInvoice invoice = TrustInvoice.createInvoice(order, authUser, InvoicePaymentFlags.builder().build());
            invoice.setPurchaseToken(PURCHASE_TOKEN);
            invoiceRepository.saveAndFlush(invoice);
            Workflow workflow = Workflow.createWorkflowForEntity(invoice, order.getWorkflow().getId());
            workflowRepository.saveAndFlush(workflow);

            invoice.initAcquireFiscalReceipt();
            FiscalReceipt receipt = invoice.getFiscalReceipts().get(0);
            fiscalReceiptRepository.saveAndFlush(receipt);

            if (withRefund) {
                SimpleTrustRefund refund = new SimpleTrustRefund();
                refund.setInvoice(invoice);
                trustRefundRepository.saveAndFlush(refund);
            }

            return order;
        });
    }
}
