package ru.yandex.travel.orders.integration.hotels;

import java.util.UUID;

import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.annotation.DirtiesContext;

import ru.yandex.travel.hotels.proto.EHotelConfirmationOutcome;
import ru.yandex.travel.hotels.proto.EHotelRefundOutcome;
import ru.yandex.travel.hotels.proto.EHotelReservationOutcome;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig;
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository;
import ru.yandex.travel.orders.services.admin.OrderAdminOperationsService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinProperties;

@DirtiesContext(classMode = DirtiesContext.ClassMode.AFTER_CLASS)
public class DolphinManualFlowTest extends AbstractHotelOrderFlowTest {

    @Autowired
    private BillingPartnerConfigRepository billingPartnerConfigRepository;

    @Autowired
    private DolphinProperties properties;

    @Autowired
    private OrderAdminOperationsService orderAdminOperationsService;

    /**
     * Dolphin properties are overridden for test profile (not the case for expedia)
     */
    @Before
    public void prepareBillingPartnerConfig() {
        properties.setPollingRetryAttempts(0);

        Long billingClientId = properties.getFinancialData().getBillingClientId();
        if (billingPartnerConfigRepository.findById(billingClientId).isEmpty()) {
            transactionTemplate.execute(tStatus -> {
                BillingPartnerConfig entity = new BillingPartnerConfig();
                entity.setAgreementActive(true);
                entity.setBillingClientId(billingClientId);
                return billingPartnerConfigRepository.save(entity);
            });
        }
    }

    @Test
    public void testSimpleOrderCancel() {
        String orderId = createOrderAndTransferToManualProcessing(false);
        restoreDolphinOrder(orderId, true);
        waitForOrderState(orderId, EHotelOrderState.OS_CANCELLED);
    }

    private void restoreDolphinOrder(String orderId, boolean toCancel) {
        transactionTemplate.executeWithoutResult(tStatus -> {
            orderAdminOperationsService.restoreDolphinOrder(UUID.fromString(orderId), toCancel);
        });
    }

    @Test
    public void testOrderRestore() {
        String orderId = createOrderAndTransferToManualProcessing(false);
        restoreDolphinOrder(orderId, false);
        waitForOrderState(orderId, EHotelOrderState.OS_WAITING_CONFIRMATION);
    }

    @Test
    public void testSimpleOrderCancelDeferred() {
        String orderId = createOrderAndTransferToManualProcessing(true);
        restoreDolphinOrder(orderId, true);
        waitForOrderState(orderId, EHotelOrderState.OS_CANCELLED);
    }

    @Test
    public void testSimpleOrderRestoreDeferred() {
        String orderId = createOrderAndTransferToManualProcessing(true);
        restoreDolphinOrder(orderId, false);
        waitForOrderState(orderId, EHotelOrderState.OS_WAITING_CONFIRMATION);
    }

    private String createOrderAndTransferToManualProcessing(boolean deferred) {
        String orderId = createDolphinOrder(deferred);
        reserveOrder(orderId);
        startPayment(orderId, deferred);
        if (!deferred) {
            authorizePayment(orderId);
        }
        waitForOrderState(orderId, EHotelOrderState.OS_MANUAL_PROCESSING);
        return orderId;
    }

    private String createDolphinOrder(boolean deferred) {
        return createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_DOLPHIN_HOTEL)
                .payloadName(deferred ? "dolphin_fully_refundable" : "dolphin")
                .testContext(getTestContext())
                .useDeferred(deferred)
                .build());
    }

    private THotelTestContext getTestContext() {
        return THotelTestContext.newBuilder()
                .setReservationOutcome(EHotelReservationOutcome.RO_SUCCESS)
                .setConfirmationOutcome(EHotelConfirmationOutcome.CO_WAITLIST_THEN_OK)
                .setRefundOutcome(EHotelRefundOutcome.RF_SUCCESS)
                .build();
    }
}
