package ru.yandex.travel.orders.integration.hotels;

import java.util.List;

import org.javamoney.moneta.Money;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.orders.admin.proto.TAddExtraChargeReq;
import ru.yandex.travel.orders.admin.proto.TOrderId;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.promo.PromoAction;
import ru.yandex.travel.orders.entities.promo.PromoCodeBehaviourOverride;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static ru.yandex.travel.hotels.common.partners.expedia.ApiVersion.V2_4;
import static ru.yandex.travel.hotels.common.partners.expedia.ApiVersion.V3;
import static ru.yandex.travel.orders.entities.finances.FinancialEventType.PAYMENT;
import static ru.yandex.travel.orders.entities.finances.FinancialEventType.REFUND;

// todo(tlg-13,mbobrov) uncomment all the refund-related tests when they are supported + multiple invoices
@SuppressWarnings("ResultOfMethodCallIgnored")
public class GenericHotelOrderFlowTests extends AbstractHotelOrderFlowTest {

    @Test
    public void testExpediaOrderReservedAndCancelledWithPromoCode() {
        PromoAction promoAction = createSuccessPromoCode(HUNDRED_RUBLES_PROMO).getPromoAction();
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .promoCode(promoCodeString)
                .build());
        reserveAndCancel(orderId, PromoCodeCheckBehaviour.CHECK_APPLIED);

        // after reserve-cancel cycle the budget should remain the initial
        transactionTemplate.executeWithoutResult(ignored -> {
            assertThat(
                    promoActionRepository.getOne(promoAction.getId()).getRemainingBudget())
                    .isEqualTo(promoAction.getInitialBudget());
        });
    }

    @Test
    public void testExpediaOrderReservedAndCancelledWithOverrideBehaviourPromoCode() {
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .promo(PromoParams.builder()
                        .promo(HUNDRED_RUBLES_PROMO)
                        .promoCodeBehaviourOverride(PromoCodeBehaviourOverride.RESTRICT_ALREADY_APPLIED)
                        .build())
                .build());
        reserveAndCancel(orderId, PromoCodeCheckBehaviour.CHECK_ALREADY_APPLIED);
    }

    @Test
    public void testExpediaOrderConfirmedClearedRefundedWithPromoCode() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .promo(PromoParams.builder().build())
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.CLEAR, PromoCodeCheckBehaviour.DO_NOT_CHECK);
    }

    @Ignore
    @Test
    public void testBnovoOrderConfirmedClearedRefundedWithPromoCodeAndFinEvents() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .promo(PromoParams.builder()
                        .promo(THREE_HUNDRED_RUBLES_PROMO)
                        .build())
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.CLEAR, PromoCodeCheckBehaviour.CHECK_APPLIED);

        List<FinancialEvent> orderEvents = getFinancialEventsOfOrder(orderId);

        FinancialEvent paymentEvent = orderEvents.stream().filter(e -> e.getType() == PAYMENT).findAny().orElse(null);
        FinancialEvent refundEvent = orderEvents.stream().filter(e -> e.getType() == REFUND).findAny().orElse(null);
        assertThat(paymentEvent).isNotNull();
        assertThat(paymentEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
        assertThat(paymentEvent.getPromoCodePartnerAmount()).isEqualTo(Money.of(300, "RUB"));
        assertThat(refundEvent).isNotNull();
        assertThat(refundEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(refundEvent.getTotalAmount()).isEqualTo(Money.of(6000, "RUB"));
        assertThat(refundEvent.getPromoCodePartnerRefundAmount()).isEqualTo(Money.of(300, "RUB"));
        assertThat(refundEvent.getOriginalEvent()).isEqualTo(paymentEvent);
    }

    @Test
    public void testExpediaOrderConfirmedClearedRefunded() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.CLEAR, PromoCodeCheckBehaviour.DO_NOT_CHECK);
    }

    @Test
    public void testExpediaOrderConfirmedRefunded() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.DO_NOT_CLEAR);
    }

    @Test
    public void testExpediaOrderReservedAndCancelled() {
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        reserveAndCancel(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
    }

    @Test
    public void testExpediaOrderConfirmFailed() {
        initializeExpediaMocksForOrderConfirmFailed(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .testContext(getContextFailedOnConfirmation())
                .build());
        confirmFailed(orderId);
    }

    @Test
    public void testExpediaApiVersionChangeOnConfirm() {
        initializeExpediaMocksForConfirmedAndRefunded(V3);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        TGetOrderInfoRsp resp = reserveOrder(orderId);
        var payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        var itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V2_4);
        verify(expediaClient, never()).usingApi(V3);
        verify(expediaClient, times(1)).usingApi(V2_4);
        initializeExpediaMockForGetHeldItineraryByAffiliateId(V3);
        resp = startAndAuthorizePayment(orderId);
        payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V3);
    }

    @Ignore
    @Test
    public void testExpediaApiVersionChangeOnRefund() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        TGetOrderInfoRsp resp = reserveOrder(orderId);
        var payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        var itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V2_4);
        verify(expediaClient, never()).usingApi(V3);
        verify(expediaClient, times(1)).usingApi(V2_4);
        resp = startAndAuthorizePayment(orderId);
        payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V2_4);
        initializeExpediaMocksForConfirmedAndRefunded(V3);
        initializeExpediaMockForGetConfirmedItineraryByAffiliateId(V3);
        resp = refundConfirmedOrderWithoutRaces(orderId);
        payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V3);
    }

    @Test
    public void testExpediaOrderExpired() {
        initializeExpediaMocksForOrderReservedPaymentFailed(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        reserveThenExpire(orderId);
        verify(expediaClient).cancelItinerarySync(any(), any(), any(), any(), any());
    }

    @Test
    public void testBNovoOrderConfirmedClearedRefunded() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.CLEAR);
    }

    @Test
    public void testBNovoOrderConfirmedRefunded() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.DO_NOT_CLEAR);
    }

    @Test
    public void testBNovoOrderConfirmFailed() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .testContext(getContextFailedOnConfirmation())
                .build());
        confirmFailed(orderId);
    }

    @Test
    public void testBNovoSoldOutOnReservation() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .testContext(getContextSoldOut())
                .build());
        failOnReservation(orderId);
    }

    @Test
    public void testBNovoOrderExpired() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        reserveThenExpire(orderId);
    }

    @Ignore
    @Test
    public void testBNovoMultiInvoiceCreateConfirmAddRefund() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        var oldTrustBalance = getTrustBalance();
        var oldOrderAccountBalance = getOrderBalance(orderId);

        adminClient.addExtraCharge(TAddExtraChargeReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setExtraAmount(ProtoUtils.toTPrice(Money.of(100, ProtoCurrencyUnit.RUB)))
                .build());
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                // todo(tlg-13): implement extra payments
                //rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_WAITING_EXTRA_PAYMENT,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_UNKNOWN,
                TIMEOUT, "Order must be in OS_WAITING_EXTRA_PAYMENT state");
        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url
        TGetOrderInfoRsp reservedOrder = IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT, TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");
        var newTrustBalance = getTrustBalance();
        var newOrderAccountBalance = getOrderBalance(orderId);
        var trustBalanceDelta = newTrustBalance.subtract(oldTrustBalance).getNumberStripped().abs();
        var orderBalanceDelta = newOrderAccountBalance.subtract(oldOrderAccountBalance).getNumberStripped().abs();
        assertThat(trustBalanceDelta).isEqualByComparingTo("100");
        assertThat(orderBalanceDelta).isEqualByComparingTo("100");
        refund(orderId, PaymentsBehaviour.DO_NOT_CLEAR);
    }

}
