package ru.yandex.travel.orders.integration.hotels;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.proto.EFiscalReceiptType;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.hotels.administrator.export.proto.ContractInfo;
import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.orders.promo.WhiteLabelApplication;
import ru.yandex.travel.hotels.common.orders.promo.YandexPlusApplication;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;
import ru.yandex.travel.hotels.models.booking_flow.promo.WhiteLabelPromoCampaign;
import ru.yandex.travel.hotels.proto.EWhiteLabelEligibility;
import ru.yandex.travel.hotels.proto.TGetWhiteLabelPointsPropsRsp;
import ru.yandex.travel.hotels.proto.TWhiteLabelPointsLinguistics;
import ru.yandex.travel.orders.admin.proto.TAddExtraChargeReq;
import ru.yandex.travel.orders.admin.proto.TCalculateMoneyOnlyRefundReq;
import ru.yandex.travel.orders.admin.proto.TManualRefundMoneyOnlyReq;
import ru.yandex.travel.orders.admin.proto.TMoveHotelOrderToNewContractReq;
import ru.yandex.travel.orders.admin.proto.TOrderId;
import ru.yandex.travel.orders.admin.proto.TPatchPartnerCommissionReq;
import ru.yandex.travel.orders.cache.BalanceContractDictionary;
import ru.yandex.travel.orders.commons.proto.EPromoCodeNominalType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.entities.promo.PromoCodeBehaviourOverride;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TWhiteLabelPromoCampaignInfo;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundRequest;
import ru.yandex.travel.orders.services.payments.model.TrustResizeRequest;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.white_label.proto.EWhiteLabelPartnerId;
import ru.yandex.travel.white_label.proto.EWhiteLabelPointsType;
import ru.yandex.travel.workflow.EWorkflowState;

import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.hotels.common.partners.expedia.ApiVersion.V2_4;
import static ru.yandex.travel.hotels.common.partners.expedia.ApiVersion.V3;
import static ru.yandex.travel.orders.TestOrderObjects.moneyMarkup;
import static ru.yandex.travel.orders.entities.finances.FinancialEventType.PAYMENT;
import static ru.yandex.travel.orders.entities.finances.FinancialEventType.REFUND;

@SuppressWarnings({"ResultOfMethodCallIgnored", "SameParameterValue"})
@Slf4j
public class HotelOrderFlowTests extends AbstractHotelOrderFlowTest {

    @SpyBean(name = "mockTrustClient")
    private TrustClient trustClient;

    @MockBean
    private BalanceContractDictionary balanceContractDictionary;

    @Test
    public void testExpediaOrderReservedAndCancelledWithPromoCode() {
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .promo(PromoParams.builder().build())
                .build());
        reserveAndCancel(orderId, PromoCodeCheckBehaviour.CHECK_APPLIED);
    }

    @Test
    public void testExpediaOrderReservedAndCancelledWithPromoCodeV2() {
        createSuccessPromoCode(HUNDRED_RUBLES_PROMO);
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .promoCode(promoCodeString)
                .build());
        reserveAndCancel(orderId, PromoCodeCheckBehaviour.CHECK_APPLIED);
    }

    @Test
    public void testExpediaOrderReservedAndCancelledWithOverrideBehaviourPromoCode() {
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .promo(PromoParams.builder()
                        .promo(HUNDRED_RUBLES_PROMO)
                        .promoCodeBehaviourOverride(PromoCodeBehaviourOverride.RESTRICT_ALREADY_APPLIED)
                        .build())
                .build());
        reserveAndCancel(orderId, PromoCodeCheckBehaviour.CHECK_ALREADY_APPLIED);
    }

    @Test
    public void testExpediaOrderConfirmedClearedRefundedWithPromoCode() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .promo(PromoParams.builder().build())
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.CLEAR, PromoCodeCheckBehaviour.DO_NOT_CHECK);

        ArgumentCaptor<TrustCreateBasketRequest> reqCaptor = ArgumentCaptor.forClass(TrustCreateBasketRequest.class);
        verify(trustClient).createBasket(reqCaptor.capture(), eq(uniqueTestTrustUserInfo()), any());
        TrustCreateBasketRequest req = reqCaptor.getValue();
        assertThat(req.getOrders()).hasSize(1).first().satisfies(order ->
                assertThat(order.getPrice()).isEqualByComparingTo(BigDecimal.valueOf(28_017/*28117 - 100*/)));
        // yandex plus check
        assertThat(req.getPaymethodMarkup()).isNull();
        assertThat(req.getPassParams().getPayload()).isNull();
    }

    @Test
    public void testManualMoneyOnlyCalculate() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        clear(orderId);

        Money initialInvoiceAmount = getOrderBalance(orderId);

        var calcRsp = adminClient.calculateMoneyOnlyRefund(TCalculateMoneyOnlyRefundReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .build());

        assertThat(ProtoUtils.fromTPrice(calcRsp.getTotalAmount())).isEqualTo(initialInvoiceAmount);
        assertThat(ProtoUtils.fromTPrice(calcRsp.getRemainingAmount())).isEqualTo(initialInvoiceAmount);
    }

    @Test
    public void testManualMoneyOnlyCalculateWithFailedPayment() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());

        failPaymentAndRetry(orderId);

        Money initialInvoiceAmount = getOrderBalance(orderId);

        var calcRsp = adminClient.calculateMoneyOnlyRefund(TCalculateMoneyOnlyRefundReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .build());

        assertThat(ProtoUtils.fromTPrice(calcRsp.getTotalAmount())).isEqualTo(initialInvoiceAmount);
        assertThat(ProtoUtils.fromTPrice(calcRsp.getRemainingAmount())).isEqualTo(initialInvoiceAmount);
    }

    @Test
    public void testManualMoneyOnlyCalculateRefunded() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        clear(orderId);
        Money initialInvoiceAmount = getOrderBalance(orderId);
        refund(orderId, PaymentsBehaviour.CLEAR);

        var calcRsp = adminClient.calculateMoneyOnlyRefund(TCalculateMoneyOnlyRefundReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .build());

        assertThat(ProtoUtils.fromTPrice(calcRsp.getTotalAmount())).isEqualTo(initialInvoiceAmount);
        assertThat(ProtoUtils.fromTPrice(calcRsp.getRemainingAmount())).isEqualTo(Money.of(12468,
                ProtoCurrencyUnit.RUB));
    }

    @Test
    public void testManualMoneyOnlyRefund() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        clear(orderId);

        var initalOrderInfo = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        Money initialInvoiceAmount = ProtoUtils.fromTPrice(initalOrderInfo.getResult().getPayments(0).getPaidAmount());
        Money refundAmount = Money.of(3000, ProtoCurrencyUnit.RUB);
        Money newInvoiceAmount = initialInvoiceAmount.subtract(refundAmount);

        var oldOrderAccountBalance = getOrderBalance(orderId);

        adminClient.manualRefundMoneyOnly(TManualRefundMoneyOnlyReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setNewInvoiceAmount(ProtoUtils.toTPrice(newInvoiceAmount))
                .setRefundAmount(ProtoUtils.toTPrice(refundAmount))
                .setRefundUserMoney(true)
                .setGenerateFinEvents(true)
                .setMoneyRefundMode(EMoneyRefundMode.MRM_PROMO_MONEY_FIRST)
                .build());

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                r -> r.getResult().getHotelOrderState() == EHotelOrderState.OS_CONFIRMED &&
                        getOrderBalance(orderId).isLessThan(oldOrderAccountBalance),
                Duration.ofSeconds(15), "Order must be confirmed and must have a refund in trust");

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                r -> getFinancialEventsOfOrder(orderId).size() == 2,
                Duration.ofSeconds(15), "Order must have two finevents");

        var newOrderAccountBalance = getOrderBalance(orderId);
        assertThat(oldOrderAccountBalance.subtract(newOrderAccountBalance)).isEqualTo(refundAmount);
        assertThat(newOrderAccountBalance).isEqualTo(newInvoiceAmount);

        var orderEvents = getFinancialEventsOfOrder(orderId);
        assertThat(orderEvents).isNotNull();
        var paymentEvents =
                orderEvents.stream().filter(e -> e.getType() == PAYMENT).collect(Collectors.toUnmodifiableList());
        var refundEvents =
                orderEvents.stream().filter(e -> e.getType() == REFUND).collect(Collectors.toUnmodifiableList());
        assertThat(paymentEvents.size()).isEqualTo(1);
        assertThat(refundEvents.size()).isEqualTo(1);
        assertThat(paymentEvents.get(0)).isNotNull();
        assertThat(paymentEvents.get(0).getTotalAmount()).isEqualTo(initialInvoiceAmount);
        assertThat(refundEvents.get(0)).isNotNull();
        assertThat(refundEvents.get(0).getTotalAmount()).isEqualTo(refundAmount);
    }

    @Test
    public void testExtraChargeAfterManualMoneyOnlyRefund() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        clear(orderId);

        var initalOrderInfo = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        Money initialInvoiceAmount = ProtoUtils.fromTPrice(initalOrderInfo.getResult().getPayments(0).getPaidAmount());
        Money refundAmount = Money.of(3000, ProtoCurrencyUnit.RUB);
        Money newInvoiceAmount = initialInvoiceAmount.subtract(refundAmount);

        Money oldOrderAccountBalance = getOrderBalance(orderId);

        adminClient.manualRefundMoneyOnly(TManualRefundMoneyOnlyReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setNewInvoiceAmount(ProtoUtils.toTPrice(newInvoiceAmount))
                .setRefundAmount(ProtoUtils.toTPrice(refundAmount))
                .setRefundUserMoney(true)
                .setGenerateFinEvents(true)
                .setMoneyRefundMode(EMoneyRefundMode.MRM_PROMO_MONEY_FIRST)
                .build());

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                r -> r.getResult().getHotelOrderState() == EHotelOrderState.OS_CONFIRMED &&
                        getOrderBalance(orderId).isLessThan(oldOrderAccountBalance),
                Duration.ofSeconds(15), "Order must be confirmed and must have a refund in trust");

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                r -> getFinancialEventsOfOrder(orderId).size() == 2,
                Duration.ofSeconds(15), "Order must have two finevents");

        var newOrderAccountBalance = getOrderBalance(orderId);
        assertThat(oldOrderAccountBalance.subtract(newOrderAccountBalance)).isEqualTo(refundAmount);
        assertThat(newOrderAccountBalance).isEqualTo(newInvoiceAmount);

        var orderEvents = getFinancialEventsOfOrder(orderId);
        assertThat(orderEvents).isNotNull();
        var paymentEvents =
                orderEvents.stream().filter(e -> e.getType() == PAYMENT).collect(Collectors.toUnmodifiableList());
        var refundEvents =
                orderEvents.stream().filter(e -> e.getType() == REFUND).collect(Collectors.toUnmodifiableList());
        assertThat(paymentEvents.size()).isEqualTo(1);
        assertThat(refundEvents.size()).isEqualTo(1);
        assertThat(paymentEvents.get(0)).isNotNull();
        assertThat(paymentEvents.get(0).getTotalAmount()).isEqualTo(initialInvoiceAmount);
        assertThat(refundEvents.get(0)).isNotNull();
        assertThat(refundEvents.get(0).getTotalAmount()).isEqualTo(refundAmount);

        var oldTrustBalance = getTrustBalance();
        var secondOldOrderAccountBalance = getOrderBalance(orderId);

        adminClient.addExtraCharge(TAddExtraChargeReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setExtraAmount(ProtoUtils.toTPrice(Money.of(100, ProtoCurrencyUnit.RUB)))
                .build());
        waitForOrderState(orderId, EHotelOrderState.OS_WAITING_EXTRA_PAYMENT);
        startAndAuthorizePayment(orderId);
        var newTrustBalance = getTrustBalance();
        newOrderAccountBalance = getOrderBalance(orderId);
        var trustBalanceDelta = newTrustBalance.subtract(oldTrustBalance).getNumberStripped().abs();
        var orderBalanceDelta = newOrderAccountBalance.subtract(secondOldOrderAccountBalance).getNumberStripped().abs();
        assertThat(trustBalanceDelta).isEqualByComparingTo("100");
        assertThat(orderBalanceDelta).isEqualByComparingTo("100");
        waitForOrderState(orderId, EHotelOrderState.OS_CONFIRMED);
    }

    @Test
    public void testManualMoneyOnlyRefundDouble() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        clear(orderId);

        var initalOrderInfo = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        Money initialInvoiceAmount = ProtoUtils.fromTPrice(initalOrderInfo.getResult().getPayments(0).getPaidAmount());
        List<Money> refundAmounts = List.of(
                Money.of(3000, ProtoCurrencyUnit.RUB),
                Money.of(1700, ProtoCurrencyUnit.RUB));
        List<Money> newInvoiceAmounts = List.of(
                initialInvoiceAmount.subtract(Money.of(3000, ProtoCurrencyUnit.RUB)),
                initialInvoiceAmount.subtract(Money.of(4700, ProtoCurrencyUnit.RUB)));

        for (int i = 0; i < 2; i++) {
            Money refundAmount = refundAmounts.get(i);
            Money newInvoiceAmount = newInvoiceAmounts.get(i);

            var oldOrderAccountBalance = getOrderBalance(orderId);

            adminClient.manualRefundMoneyOnly(TManualRefundMoneyOnlyReq.newBuilder()
                    .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                    .setNewInvoiceAmount(ProtoUtils.toTPrice(newInvoiceAmount))
                    .setRefundAmount(ProtoUtils.toTPrice(refundAmount))
                    .setRefundUserMoney(true)
                    .setGenerateFinEvents(true)
                    .setMoneyRefundMode(EMoneyRefundMode.MRM_PROMO_MONEY_FIRST)
                    .build());

            IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                    r -> r.getResult().getHotelOrderState() == EHotelOrderState.OS_CONFIRMED &&
                            getOrderBalance(orderId).isLessThan(oldOrderAccountBalance),
                    Duration.ofSeconds(15), "Order must be confirmed and must have a refund in trust");

            var expectedFinEventCount = 2 + i;
            IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                    r -> getFinancialEventsOfOrder(orderId).size() == expectedFinEventCount,
                    Duration.ofSeconds(15), String.format("Order must have %s finevents", expectedFinEventCount));

            var newOrderAccountBalance = getOrderBalance(orderId);
            assertThat(oldOrderAccountBalance.subtract(newOrderAccountBalance)).isEqualTo(refundAmount);
            assertThat(newOrderAccountBalance).isEqualTo(newInvoiceAmount);
        }

        var orderEvents = getFinancialEventsOfOrder(orderId);
        assertThat(orderEvents).isNotNull();
        var paymentEvents =
                orderEvents.stream().filter(e -> e.getType() == PAYMENT).collect(Collectors.toUnmodifiableList());
        var refundEvents = orderEvents.stream().filter(e -> e.getType() == REFUND)
                .sorted(Comparator.comparing(FinancialEvent::getId)).collect(Collectors.toUnmodifiableList());
        assertThat(paymentEvents.size()).isEqualTo(1);
        assertThat(refundEvents.size()).isEqualTo(2);
        assertThat(paymentEvents.get(0)).isNotNull();
        assertThat(paymentEvents.get(0).getTotalAmount()).isEqualTo(initialInvoiceAmount);
        assertThat(refundEvents.get(0)).isNotNull();
        assertThat(refundEvents.get(0).getTotalAmount()).isEqualTo(refundAmounts.get(0));
        assertThat(refundEvents.get(1)).isNotNull();
        assertThat(refundEvents.get(1).getTotalAmount()).isEqualTo(refundAmounts.get(1));
    }

    @Test
    public void testExpediaOrderExpiredWhenPaymentStartedWithPromoCode() {
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .promo(PromoParams.builder().build())
                .build());
        reserveOrder(orderId);

        checkPromoCodeActivated(0);

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST).setSource("desktop")
                .setOrderId(orderId).setReturnUrl("some_return_url").build()); // safely ignoring response, as we
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        expireOrderItem(orderId);

        waitForOrderState(orderId, EHotelOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION);

        checkPromoCodeActivated(0);

        failPayment(orderId);

        waitForOrderState(orderId, EHotelOrderState.OS_CANCELLED);

        checkPromoCodeActivated(0);
    }

    @Test
    public void testBnovoOrderConfirmedClearedRefundedWithPromoCodeAndFinEvents() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .promo(PromoParams.builder()
                        .promo(THREE_HUNDRED_RUBLES_PROMO)
                        .build())
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.CLEAR, PromoCodeCheckBehaviour.CHECK_APPLIED);

        List<FinancialEvent> orderEvents = getFinancialEventsOfOrder(orderId);
        //noinspection ConstantConditions
        FinancialEvent paymentEvent = orderEvents.stream().filter(e -> e.getType() == PAYMENT).findAny().orElse(null);
        FinancialEvent refundEvent = orderEvents.stream().filter(e -> e.getType() == REFUND).findAny().orElse(null);
        assertThat(paymentEvent).isNotNull();
        assertThat(paymentEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
        assertThat(paymentEvent.getPromoCodePartnerAmount()).isEqualTo(Money.of(300, "RUB"));
        assertThat(refundEvent).isNotNull();
        assertThat(refundEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(refundEvent.getTotalAmount()).isEqualTo(Money.of(6000, "RUB"));
        assertThat(refundEvent.getPromoCodePartnerRefundAmount()).isEqualTo(Money.of(300, "RUB"));
        assertThat(refundEvent.getOriginalEvent()).isEqualTo(paymentEvent);
    }

    @Test
    public void testBnovoOrderWithPromoCodeAndFinEventsProportionalRefund() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .promo(PromoParams.builder()
                        .promo(Tuple2.tuple(EPromoCodeNominalType.NT_PERCENT, BigDecimal.valueOf(20)))
                        .build())
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        clear(orderId);

        performManualRefund(orderId, 3200, 6400);
        waitNFinancialEvents(orderId, 2);

        List<FinancialEvent> orderEvents = transactionTemplate.execute(txStatus ->
                financialEventRepository.findAll().stream()
                        .filter(e -> e.getOrder().getId().toString().equals(orderId))
                        .collect(Collectors.toList()));
        //noinspection ConstantConditions
        FinancialEvent paymentEvent = orderEvents.stream().filter(e -> e.getType() == PAYMENT).findAny().orElse(null);
        FinancialEvent refundEvent = orderEvents.stream().filter(e -> e.getType() == REFUND).findAny().orElse(null);
        assertThat(paymentEvent).isNotNull();
        assertThat(paymentEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
        assertThat(paymentEvent.getPromoCodePartnerAmount()).isEqualTo(Money.of(2400, "RUB"));
        assertThat(paymentEvent.getPartnerAmount()).isEqualTo(Money.of(8400, "RUB"));
        assertThat(paymentEvent.getFeeAmount()).isEqualTo(Money.of(1200, "RUB"));
        assertThat(refundEvent).isNotNull();
        assertThat(refundEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(refundEvent.getTotalAmount()).isEqualTo(Money.of(8000, "RUB"));
        assertThat(refundEvent.getPromoCodePartnerRefundAmount()).isEqualTo(Money.of(1600, "RUB"));
        assertThat(refundEvent.getPartnerRefundAmount()).isEqualTo(Money.of(5600, "RUB"));
        assertThat(refundEvent.getFeeRefundAmount()).isEqualTo(Money.of(800, "RUB"));
        assertThat(refundEvent.getOriginalEvent()).isEqualTo(paymentEvent);
    }

    @Test
    public void testBnovoOrderWithPromoCodeAndFinEventsProportionalRefundWithPlus() {
        var promocode = createSuccessPromoCode(Tuple2.tuple(EPromoCodeNominalType.NT_PERCENT, BigDecimal.valueOf(20))
                , null);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .promoCode(promocode.getCode())
                .payloadCustomizer(getBnovoPayloadCustomizerForYandexPlusWithdraw())
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        clear(orderId);

        performManualRefund(orderId, 3200, 6400);
        waitNFinancialEvents(orderId, 2);

        List<FinancialEvent> orderEvents = transactionTemplate.execute(txStatus ->
                financialEventRepository.findAll().stream()
                        .filter(e -> e.getOrder().getId().toString().equals(orderId))
                        .collect(Collectors.toList()));
        //noinspection ConstantConditions
        FinancialEvent paymentEvent = orderEvents.stream().filter(e -> e.getType() == PAYMENT).findAny().orElse(null);
        FinancialEvent refundEvent = orderEvents.stream().filter(e -> e.getType() == REFUND).findAny().orElse(null);
        assertThat(paymentEvent).isNotNull();
        assertThat(paymentEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
        assertThat(paymentEvent.getPromoCodePartnerAmount()).isEqualTo(Money.of(2400, "RUB"));
        assertThat(paymentEvent.getPartnerAmount()).isEqualTo(Money.of(6400, "RUB"));
        assertThat(paymentEvent.getPlusPartnerAmount()).isEqualTo(Money.of(2000, "RUB"));
        assertThat(paymentEvent.getFeeAmount()).isEqualTo(Money.of(1200, "RUB"));
        assertThat(refundEvent).isNotNull();
        assertThat(refundEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(refundEvent.getTotalAmount()).isEqualTo(Money.of(8000, "RUB"));
        assertThat(refundEvent.getPromoCodePartnerRefundAmount()).isEqualTo(Money.of(1600, "RUB"));
        assertThat(refundEvent.getPartnerRefundAmount()).isEqualTo(Money.of(3600, "RUB"));
        assertThat(refundEvent.getPlusPartnerRefundAmount()).isEqualTo(Money.of(2000, "RUB"));
        assertThat(refundEvent.getFeeRefundAmount()).isEqualTo(Money.of(800, "RUB"));
        assertThat(refundEvent.getOriginalEvent()).isEqualTo(paymentEvent);
    }

    private void performManualRefund(String orderId, Number newInvoiceAmount, Number refundAmount) {
        performManualRefund(orderId, newInvoiceAmount, refundAmount, true);
    }

    private void performManualRefund(String orderId, Number newInvoiceAmount, Number refundAmount,
                                     boolean refundUserMoney) {
        var oldOrderAccountBalance = getOrderBalance(orderId);

        adminClient.manualRefundMoneyOnly(TManualRefundMoneyOnlyReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setNewInvoiceAmount(ProtoUtils.toTPrice(Money.of(newInvoiceAmount, ProtoCurrencyUnit.RUB)))
                .setRefundAmount(ProtoUtils.toTPrice(Money.of(refundAmount, ProtoCurrencyUnit.RUB)))
                .setRefundUserMoney(refundUserMoney)
                .setGenerateFinEvents(true)
                .setMoneyRefundMode(EMoneyRefundMode.MRM_PROPORTIONAL)
                .build());

        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                r -> r.getResult().getHotelOrderState() == EHotelOrderState.OS_CONFIRMED &&
                        getOrderBalance(orderId).isLessThanOrEqualTo(oldOrderAccountBalance),
                Duration.ofSeconds(15), "Order must be confirmed and must have a refund in trust");
    }

    @Test
    public void testExpediaOrderConfirmedClearedRefunded() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.CLEAR, PromoCodeCheckBehaviour.DO_NOT_CHECK);
    }

    @Test
    public void testExpediaOrderConfirmedWithPaymentRetry() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());

        failPaymentAndRetry(orderId);
    }

    @Test
    public void testExpediaOrderConfirmedRefunded() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.DO_NOT_CLEAR);
    }

    @Test
    public void testExpediaOrderReservedAndCancelled() {
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        reserveAndCancel(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
    }

    @Test
    public void testExpediaOrderConfirmFailed() {
        initializeExpediaMocksForOrderConfirmFailed(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .testContext(getContextFailedOnConfirmation())
                .build());
        confirmFailed(orderId);
    }

    @Test
    public void testExpediaApiVersionChangeOnConfirm() {
        initializeExpediaMocksForConfirmedAndRefunded(V3);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        TGetOrderInfoRsp resp = reserveOrder(orderId);
        var payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        var itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V2_4);
        verify(expediaClient, never()).usingApi(V3);
        verify(expediaClient, times(1)).usingApi(V2_4);
        initializeExpediaMockForGetHeldItineraryByAffiliateId(V3);
        resp = startAndAuthorizePayment(orderId);
        payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V3);
    }


    @Test
    public void testExpediaApiVersionChangeOnRefund() {
        initializeExpediaMocksForConfirmedAndRefunded(V3);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        TGetOrderInfoRsp resp = reserveOrder(orderId);
        var payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        var itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V2_4);
        verify(expediaClient, never()).usingApi(V3);
        verify(expediaClient, times(1)).usingApi(V2_4);
        resp = startAndAuthorizePayment(orderId);
        payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V2_4);
        initializeExpediaMocksForConfirmedAndRefunded(V3);
        initializeExpediaMockForGetConfirmedItineraryByAffiliateId(V3);
        resp = refundConfirmedOrderWithoutRaces(orderId);
        payload = resp.getResult().getService(0).getServiceInfo().getPayload();
        itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
        assertThat(itinerary.getApiVersion()).isEqualTo(V3);
    }

    @Test
    public void testExpediaOrderExpired() {
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .build());
        reserveThenExpire(orderId);
        verify(expediaClient).cancelItinerarySync(any(), any(), any(), any(), any());
    }

    @Test
    public void testBNovoOrderConfirmedClearedRefunded() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.CLEAR);
    }

    @Test
    public void testBNovoOrderConfirmedRefunded() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirmAndRefund(orderId, PaymentsBehaviour.DO_NOT_CLEAR);
    }

    @Test
    public void testBNovoOrderConfirmFailed() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .testContext(getContextFailedOnConfirmation())
                .build());
        confirmFailed(orderId);
    }

    @Test
    public void testBNovoSoldOutOnReservation() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .testContext(getContextSoldOut())
                .build());
        failOnReservation(orderId);
    }

    @Test
    public void testBNovoOrderExpired() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        reserveThenExpire(orderId);
    }

    @Test
    public void testBNovoMultiInvoiceCreateConfirmAddRefund() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        var oldTrustBalance = getTrustBalance();
        var oldOrderAccountBalance = getOrderBalance(orderId);

        adminClient.addExtraCharge(TAddExtraChargeReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setExtraAmount(ProtoUtils.toTPrice(Money.of(100, ProtoCurrencyUnit.RUB)))
                .build());
        waitForOrderState(orderId, EHotelOrderState.OS_WAITING_EXTRA_PAYMENT);
        startAndAuthorizePayment(orderId);
        var newTrustBalance = getTrustBalance();
        var newOrderAccountBalance = getOrderBalance(orderId);
        var trustBalanceDelta = newTrustBalance.subtract(oldTrustBalance).getNumberStripped().abs();
        var orderBalanceDelta = newOrderAccountBalance.subtract(oldOrderAccountBalance).getNumberStripped().abs();
        assertThat(trustBalanceDelta).isEqualByComparingTo("100");
        assertThat(orderBalanceDelta).isEqualByComparingTo("100");
        refund(orderId, PaymentsBehaviour.DO_NOT_CLEAR);
    }

    @Test
    public void testYandexPlusWithdrawFlow() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .payloadCustomizer(getBnovoPayloadCustomizerForYandexPlusWithdraw())
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);

        ArgumentCaptor<TrustCreateBasketRequest> reqCaptor = ArgumentCaptor.forClass(TrustCreateBasketRequest.class);
        verify(trustClient).createBasket(reqCaptor.capture(), eq(uniqueTestTrustUserInfo()), any());
        TrustCreateBasketRequest req = reqCaptor.getValue();
        assertThat(req.getOrders()).hasSize(1).first().satisfies(order ->
                assertThat(order.getPrice()).isEqualByComparingTo(BigDecimal.valueOf(12_000)));
        assertThat(req.getPaymethodMarkup()).hasSize(1).allSatisfy((oid, markup) -> {
            assertThat(markup.getCard()).isEqualByComparingTo(BigDecimal.valueOf(10_000));
            assertThat(markup.getYandexAccount()).isEqualByComparingTo(BigDecimal.valueOf(2_000));
        });
        assertThat(req.getPassParams().getPayload()).isNotNull();
        assertThat(req.getPassParams().getPayload().getCashbackService()).isEqualTo("travel");

        // resize (partial unhold before clearing)
        refund(orderId, PaymentsBehaviour.DO_NOT_CLEAR);

        MoneyMarkup invoiceMarkup = transactionTemplate.execute(cb ->
                orderRepository.getOne(UUID.fromString(orderId))
                        .getInvoices().stream()
                        .flatMap(i -> i.getInvoiceItems().stream())
                        .map(InvoiceItem::getPriceMarkup)
                        .collect(CustomCollectors.exactlyOne()));
        // returned plus points first and the remaining part with actual card money
        assertThat(invoiceMarkup).isEqualTo(moneyMarkup(6_000, 0));

        ArgumentCaptor<TrustResizeRequest> resizeReqCaptor = ArgumentCaptor.forClass(TrustResizeRequest.class);
        verify(trustClient).resize(any(), any(), resizeReqCaptor.capture(), eq(uniqueTestTrustUserInfo()));
        TrustResizeRequest resizeReq = resizeReqCaptor.getValue();
        assertThat(resizeReq.getAmount()).isEqualByComparingTo(BigDecimal.valueOf(6_000));
        assertThat(resizeReq.getPaymethodMarkup()).hasSize(1).allSatisfy((oid, markup) -> {
            assertThat(markup.getCard()).isEqualByComparingTo(BigDecimal.valueOf(6_000));
            assertThat(markup.getYandexAccount()).isEqualByComparingTo(BigDecimal.valueOf(0));
        });
    }

    @Test
    public void testYandexPlusWithdrawFlow_refundAfterClearing() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .payloadCustomizer(getBnovoPayloadCustomizerForYandexPlusWithdraw())
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        // ^ the initial setup is identical to the test above

        // real refund
        clear(orderId);
        refund(orderId, PaymentsBehaviour.CLEAR);

        MoneyMarkup invoiceMarkup = transactionTemplate.execute(cb ->
                orderRepository.getOne(UUID.fromString(orderId))
                        .getInvoices().stream()
                        .flatMap(i -> i.getInvoiceItems().stream())
                        .map(InvoiceItem::getPriceMarkup)
                        .collect(CustomCollectors.exactlyOne()));
        // returned plus points first and the remaining part with actual card money
        assertThat(invoiceMarkup).isEqualTo(moneyMarkup(6_000, 0));

        ArgumentCaptor<TrustCreateRefundRequest> refundReqCaptor =
                ArgumentCaptor.forClass(TrustCreateRefundRequest.class);
        verify(trustClient).createRefund(refundReqCaptor.capture(), eq(uniqueTestTrustUserInfo()));
        TrustCreateRefundRequest refundReq = refundReqCaptor.getValue();
        assertThat(refundReq.getOrders()).hasSize(1).first().satisfies(order ->
                assertThat(order.getDeltaAmount()).isEqualByComparingTo(BigDecimal.valueOf(6_000)));
        assertThat(refundReq.getPaymethodMarkup()).hasSize(1).allSatisfy((oid, markup) -> {
            assertThat(markup.getCard()).isEqualByComparingTo(BigDecimal.valueOf(4_000));
            assertThat(markup.getYandexAccount()).isEqualByComparingTo(BigDecimal.valueOf(2_000));
        });
    }

    private Function<String, String> getBnovoPayloadCustomizerForYandexPlusWithdraw() {
        return payloadCustomizer(BNovoHotelItinerary.class, itinerary ->
                itinerary.setAppliedPromoCampaigns(AppliedPromoCampaigns.builder()
                        .yandexPlus(YandexPlusApplication.builder()
                                .mode(YandexPlusApplication.Mode.WITHDRAW)
                                .points(2000)
                                .build())
                        .build()));
    }

    @Test
    public void testCreateAndConfirmOrderWithDeferredPayments() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .useDeferred(true)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);
        assertThat(finEvents.size()).isEqualTo(0);

        TGetOrderInfoRsp order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(order.getResult().getCurrentInvoice()).isNotNull();
        assertThat(order.getResult().getPayments(0).getPaidAmount().getAmount()).isEqualTo(1246800);
        assertThat(order.getResult().getPayments(0).getTotalAmount().getAmount()).isEqualTo(1246800);
        assertThat(order.getResult().getZeroFirstPayment()).isFalse();
    }

    private CreateOrderParams.CreateOrderParamsBuilder postPayBNovoOrderParams() {
        return CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName("bnovo_postpay")
                .testContext(getAllSuccessfulContext())
                .useDeferred(false)
                .usePostPay(true);
    }

    @Test
    public void testPostPayOrderCreation() {
        String orderId = createOrder(postPayBNovoOrderParams().build());

        TGetOrderInfoRsp order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        Money price = Money.of(12000, ProtoCurrencyUnit.RUB); // see in bnovo_postpay.json
        assertThat(ProtoUtils.fromTPrice(order.getResult().getPriceInfo().getOriginalPrice())).isEqualTo(price);
        assertThat(ProtoUtils.fromTPrice(order.getResult().getPriceInfo().getPrice())).isEqualTo(price);

        HotelItinerary itinerary = ProtoUtils.fromTJson(order.getResult().getService(0).getServiceInfo().getPayload()
                , HotelItinerary.class);
        assertThat(itinerary.getFiscalPrice()).isEqualTo(price);
    }

    @Test
    public void testPostPayOrderConfirmation() {
        String orderId = createOrder(postPayBNovoOrderParams().build());
        confirmWithoutPayment(orderId);

        TGetOrderInfoRsp order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(order.getResult().getPaymentsCount()).isEqualTo(0);
        assertThat(order.getResult().getInvoiceCount()).isEqualTo(0);

        waitNFinancialEvents(orderId, 1);
        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);

        FinancialEvent event = finEvents.get(0);
        Money price = Money.of(12000, ProtoCurrencyUnit.RUB); // see in bnovo_postpay.json
        double rate = 0.1; // see in BillingPartnerService.addBNovoAgreementOrThrow
        assertThat(event.getTotalAmount()).isEqualTo(price.multiply(1 + rate));
        assertThat(event.getPostPayPartnerPayback()).isEqualTo(price.multiply(rate));
        assertThat(event.getPostPayUserAmount()).isEqualTo(price);
        assertThat(event.getPartnerAmount().isZero()).isTrue();
        assertThat(event.getFeeAmount().isZero()).isTrue();
    }

    @Test
    public void testPostPayOrderRefund() {
        String orderId = createOrder(postPayBNovoOrderParams().build());
        confirmWithoutPayment(orderId);

        refundOrder(orderId);

        TGetOrderInfoRsp order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(order.getResult().getPaymentsCount()).isZero();
        assertThat(order.getResult().getInvoiceCount()).isZero();

        waitNFinancialEvents(orderId, 2);
        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);

        FinancialEvent paymentEvent = finEvents.stream().filter(FinancialEvent::isPayment).findFirst().orElse(null);
        FinancialEvent refundEvent = finEvents.stream().filter(FinancialEvent::isRefund).findFirst().orElse(null);
        assertThat(paymentEvent).isNotNull();
        assertThat(refundEvent).isNotNull();
        assertThat(paymentEvent.getPostPayUserAmount()).isEqualTo(refundEvent.getPostPayUserRefund());
        assertThat(paymentEvent.getPostPayPartnerPayback()).isEqualTo(refundEvent.getPostPayPartnerRefund());
    }

    @Test
    public void testPostPayAndDeferredOrderCreationFails() {
        assertThatThrownBy(() -> createOrder(postPayBNovoOrderParams().useDeferred(true).build()))
                .hasMessageContaining("Cannot use deferred payment and post payment together");
    }

    @Test
    public void testPostPayOrderPartialRefund() {
        String orderId = createOrder(postPayBNovoOrderParams().build());
        confirmWithoutPayment(orderId);

        performManualRefund(orderId, 4000, 8000, false);

        waitNFinancialEvents(orderId, 2);
        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);

        FinancialEvent refundEvent = finEvents.stream().filter(FinancialEvent::isRefund).findFirst().orElse(null);
        assertThat(refundEvent).isNotNull();
        assertThat(refundEvent.getPostPayUserRefund()).isEqualTo(Money.of(8000, ProtoCurrencyUnit.RUB));
        assertThat(refundEvent.getPostPayPartnerRefund()).isEqualTo(Money.of(800, ProtoCurrencyUnit.RUB));
    }

    @Test
    public void testPostPayOrderExtraPayment() {
        String orderId = createOrder(postPayBNovoOrderParams().build());
        confirmWithoutPayment(orderId);

        adminClient.addExtraCharge(TAddExtraChargeReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setExtraAmount(ProtoUtils.toTPrice(Money.of(1000, ProtoCurrencyUnit.RUB)))
                .build());

        waitForOrderState(orderId, EHotelOrderState.OS_CONFIRMED);

        waitNFinancialEvents(orderId, 2);
        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);
        Money totalCost = finEvents.stream()
                .map(FinancialEvent::getTotalAmount)
                .reduce(Money.zero(ProtoCurrencyUnit.RUB), Money::add);
        Money reverseFee = finEvents.stream()
                .map(FinancialEvent::getPostPayPartnerPayback)
                .reduce(Money.zero(ProtoCurrencyUnit.RUB), Money::add);
        assertThat(totalCost).isEqualTo(Money.of(14300, ProtoCurrencyUnit.RUB));
        assertThat(reverseFee).isEqualTo(Money.of(1300, ProtoCurrencyUnit.RUB));
    }

    @Test
    public void testCreateAndConfirmOrderWithMultipleStartPaymentDeferred() {
        // если этот тест флапает, значит что-то не работает и нужно разбираться.
        // тут мы шлем startPayment много раз и в зависимости от порядка обработки событий
        // тест может иногда проходить, но если он иногда падает, значит заказы будут иногда падать
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA_2119)
                .useDeferred(true)
                .build());
        reserveOrder(orderId);

        startPayment(orderId);
        authorizePayment(orderId);
        TGetOrderInfoRsp confirmedOrderResponse = IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp2 -> {
                    if (rsp2.getResult().getHotelOrderState() == EHotelOrderState.OS_CONFIRMED) {
                        return true;
                    }
                    assertThat(rsp2.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
                    tryStartPayment(orderId);
                    return false;
                },
                TIMEOUT, "Order must be in " + EHotelOrderState.OS_CONFIRMED.name() + " state");

        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        confirmedOrderResponse = IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getInvoiceList().stream()
                        .flatMap(i -> i.getFiscalReceiptList().stream())
                        .filter(fr -> fr.getType() == EFiscalReceiptType.FRT_ACQUIRE)
                        .filter(fr -> !Strings.isNullOrEmpty(fr.getUrl()))
                        .filter(fr -> fr.getUrl().contains("fiscal.receipt"))
                        .count() == 1,
                Duration.ofSeconds(15), "We should have got one fiscal receipt of the ACQUIRE type");
        assertThat(confirmedOrderResponse.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
        assertThat(confirmedOrderResponse.getResult().getPayments(0).getPaidAmount().getAmount()).isEqualTo(1246800);
        assertThat(confirmedOrderResponse.getResult().getPayments(0).getTotalAmount().getAmount()).isEqualTo(1246800);
        assertThat(confirmedOrderResponse.getResult().getZeroFirstPayment()).isFalse();
    }

    @Test
    public void testCreateAndConfirmOrderWithZeroRubleDeferredPayments() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO_FULLY_REFUNDABLE)
                .useDeferred(true)
                .build());
        confirmWithZeroPayment(orderId);

        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);
        assertThat(finEvents.size()).isEqualTo(0);

        TGetOrderInfoRsp order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(order.getResult().hasCurrentInvoice()).isFalse();
        assertThat(order.getResult().getPayments(0).getPaidAmount().getAmount()).isEqualTo(0);
        assertThat(order.getResult().getPayments(0).getTotalAmount().getAmount()).isEqualTo(0);
        assertThat(order.getResult().getZeroFirstPayment()).isTrue();
    }

    @Test
    public void testCreateAndConfirmOrderWithZeroRubleDeferredPaymentsWithPromoCode() {
        String orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO_FULLY_REFUNDABLE)
                .promo(PromoParams.builder()
                        .promo(HUNDRED_RUBLES_PROMO)
                        .build())
                .useDeferred(true)
                .build());
        confirmWithZeroPayment(orderId);

        checkPromoCodeActivated(1);

        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);
        assertThat(finEvents.size()).isEqualTo(0);

        TGetOrderInfoRsp order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(order.getResult().hasCurrentInvoice()).isFalse();
        assertThat(order.getResult().getPayments(0).getPaidAmount().getAmount()).isEqualTo(0);
        assertThat(order.getResult().getPayments(0).getTotalAmount().getAmount()).isEqualTo(0);
    }

    @Test
    public void testCreateAndConfirmOrderWithPercentPromoCode() {
        String orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO_FULLY_REFUNDABLE)
                .promo(PromoParams.builder()
                        .promo(TEN_PERCENT_PROMO)
                        .build())
                .useDeferred(true)
                .build());
        confirmWithZeroPayment(orderId);

        checkPromoCodeActivated(1);
    }

    @Test
    public void testSecondOrderPaymentWithZeroRubleDeferredPayments() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO_FULLY_REFUNDABLE)
                .useDeferred(true)
                .build());
        confirmWithZeroPayment(orderId);

        startAndAuthorizePaymentAndCheck(orderId, null);

        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getPaymentSchedule().getAllInvoices().get(0).getPaidAmount().getNumber().longValueExact()).isEqualTo(0);
            Money secondInvoicePaidAmount = order.getPaymentSchedule().getAllInvoices().get(1).getPaidAmount();
            Long secondPaymentAmount = secondInvoicePaidAmount
                    .multiply(BigDecimal.valueOf(10).pow(secondInvoicePaidAmount.getCurrency().getDefaultFractionDigits()))
                    .getNumber()
                    .longValue();
            assertThat(secondPaymentAmount).isEqualTo(1200000);
            return null;
        });

        TGetOrderInfoRsp order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(order.getResult().hasCurrentInvoice()).isTrue();
        assertThat(order.getResult().getPayments(0).getNextPayments(0).getPaidAmount().getAmount())
                .isEqualTo(order.getResult().getPriceInfo().getOriginalPrice().getAmount());

        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);
        assertThat(finEvents.size()).isEqualTo(1);
        Long amount = finEvents.get(0).getTotalAmount()
                .multiply(BigDecimal.valueOf(10).pow(finEvents.get(0).getTotalAmount().getCurrency().getDefaultFractionDigits()))
                .getNumber()
                .longValue();
        assertThat(amount).isEqualTo(order.getResult().getPriceInfo().getOriginalPrice().getAmount());
        assertThat(order.getResult().getZeroFirstPayment()).isFalse();
    }

    @Test
    public void testCreateAndRefundOrderWithZeroRubleDeferredPayments() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO_FULLY_REFUNDABLE)
                .useDeferred(true)
                .build());
        confirmWithZeroPayment(orderId);
        refundWithZeroPayment(orderId);

        List<FinancialEvent> finEvents = getFinancialEventsOfOrder(orderId);
        assertThat(finEvents.size()).isEqualTo(0);

        TGetOrderInfoRsp order = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(order.getResult().hasCurrentInvoice()).isFalse();
        assertThat(order.getResult().getPayments(0).getPaidAmount().getAmount()).isEqualTo(0);
        assertThat(order.getResult().getPayments(0).getTotalAmount().getAmount()).isEqualTo(0);
        assertThat(order.getResult().getZeroFirstPayment()).isTrue();
        assertThat(getOrderBalance(orderId).isZero()).isTrue();
    }

    private void doTestCommissionPatch(Supplier<String> createOrderFunc,
                                       Consumer<FinancialEvent> checkPaymentEvent,
                                       Consumer<FinancialEvent> checkRefundEvent,
                                       Consumer<FinancialEvent> checkCorrectionPaymentEvent,
                                       Consumer<FinancialEvent> checkCorrectionRefundEvent,
                                       Integer comissionOverride) {
        var orderId = createOrderFunc.get();
        var expectedEventsCnt = checkRefundEvent == null ? 1 : 2;
        waitNFinancialEvents(orderId, expectedEventsCnt);
        List<FinancialEvent> orderEvents = transactionTemplate.execute(txStatus ->
                financialEventRepository.findAll().stream()
                        .filter(e -> e.getOrder().getId().toString().equals(orderId))
                        .collect(Collectors.toList()));
        //noinspection ConstantConditions
        FinancialEvent paymentEvent = orderEvents.stream().filter(e -> e.getType() == PAYMENT).findAny().orElse(null);
        FinancialEvent refundEvent = orderEvents.stream().filter(e -> e.getType() == REFUND).findAny().orElse(null);
        assertThat(paymentEvent).isNotNull();
        assertThat(paymentEvent.getBillingClientId()).isEqualTo(-10000005L);
        checkPaymentEvent.accept(paymentEvent);
        if (checkRefundEvent != null) {
            assertThat(refundEvent).isNotNull();
            assertThat(refundEvent.getBillingClientId()).isEqualTo(-10000005L);
            checkRefundEvent.accept(refundEvent);
        }

        int targetComissionPct = comissionOverride == null ? 6 : comissionOverride;

        var patchRsp = adminClient.patchPartnerCommission(TPatchPartnerCommissionReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setTargetConfirmedRatePct(targetComissionPct)
                .setTargetRefundedRatePct(targetComissionPct)
                .build());

        var newAgreement = transactionTemplate.execute(txStatus -> {
            var order = orderRepository.getOne(UUID.fromString(orderId));
            var orderItem = order.getOrderItems().get(0);
            Preconditions.checkState(orderItem instanceof DirectHotelBillingPartnerAgreementProvider);
            return ((DirectHotelBillingPartnerAgreementProvider) orderItem).getAgreement();
        });
        assertThat(newAgreement.getOrderConfirmedRate()).isEqualTo(BigDecimal.valueOf(targetComissionPct, 2));
        assertThat(newAgreement.getOrderRefundedRate()).isEqualTo(BigDecimal.valueOf(targetComissionPct, 2));

        if (checkCorrectionPaymentEvent == null && checkCorrectionRefundEvent == null) {
            assertThat(patchRsp.getRegisteredNewEvents()).isFalse();
            waitNFinancialEvents(orderId, expectedEventsCnt);
            return;
        }

        assertThat(patchRsp.getRegisteredNewEvents()).isTrue();

        waitNFinancialEvents(orderId, expectedEventsCnt + 2);
        var orderEventsNew = getFinancialEventsOfOrder(orderId).stream()
                .filter(x -> !x.getId().equals(paymentEvent.getId()) && (refundEvent == null || !x.getId().equals(refundEvent.getId())))
                .collect(Collectors.toUnmodifiableList());
        FinancialEvent correctionPaymentEvent =
                orderEventsNew.stream().filter(e -> e.getType() == PAYMENT).findAny().orElse(null);
        FinancialEvent correctionRefundEvent =
                orderEventsNew.stream().filter(e -> e.getType() == REFUND).findAny().orElse(null);

        assertThat(correctionPaymentEvent).isNotNull();
        assertThat(correctionPaymentEvent.getBillingClientId()).isEqualTo(-10000005L);
        checkCorrectionPaymentEvent.accept(correctionPaymentEvent);

        assertThat(correctionRefundEvent).isNotNull();
        assertThat(correctionRefundEvent.getBillingClientId()).isEqualTo(-10000005L);
        assertThat(correctionRefundEvent.getOriginalEvent()).isEqualTo(paymentEvent);
        checkCorrectionRefundEvent.accept(correctionRefundEvent);
    }

    private void doTestCommissionPatch(Supplier<String> createOrderFunc,
                                       Consumer<FinancialEvent> checkPaymentEvent,
                                       Consumer<FinancialEvent> checkRefundEvent,
                                       Integer expectedFeeChange,
                                       Integer comissionOverride) {
        doTestCommissionPatch(createOrderFunc, checkPaymentEvent, checkRefundEvent,
                expectedFeeChange == null ? null : correctionPaymentEvent -> {
                    assertThat(correctionPaymentEvent.getTotalAmount()).isEqualTo(Money.of(expectedFeeChange, "RUB"));
                    assertThat(correctionPaymentEvent.getPartnerAmount()).isEqualTo(Money.of(expectedFeeChange, "RUB"));
                    assertThat(correctionPaymentEvent.getFeeAmount()).isEqualTo(Money.of(0, "RUB"));
                },
                expectedFeeChange == null ? null : correctionRefundEvent -> {
                    assertThat(correctionRefundEvent).isNotNull();
                    assertThat(correctionRefundEvent.getTotalAmount()).isEqualTo(Money.of(expectedFeeChange, "RUB"));
                    assertThat(correctionRefundEvent.getPartnerRefundAmount()).isEqualTo(Money.of(0, "RUB"));
                    assertThat(correctionRefundEvent.getFeeRefundAmount()).isEqualTo(Money.of(expectedFeeChange, "RUB"
                    ));
                },
                comissionOverride
        );
    }

    private void doTestCommissionPatch(Supplier<String> createOrderFunc,
                                       Consumer<FinancialEvent> checkPaymentEvent,
                                       Consumer<FinancialEvent> checkRefundEvent,
                                       Integer expectedFeeChange) {
        doTestCommissionPatch(createOrderFunc, checkPaymentEvent, checkRefundEvent, expectedFeeChange, null);
    }

    @Test
    public void testCommissionPatchSimple() {
        doTestCommissionPatch(
                () -> {
                    var orderId = createOrder(CreateOrderParams.builder()
                            .serviceType(EServiceType.PT_BNOVO_HOTEL)
                            .payloadName(PAYLOAD_BNOVO)
                            .build());
                    confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
                    clear(orderId);
                    return orderId;
                },
                paymentEvent -> {
                    assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
                    assertThat(paymentEvent.getPartnerAmount()).isEqualTo(Money.of(10800, "RUB"));
                    assertThat(paymentEvent.getFeeAmount()).isEqualTo(Money.of(1200, "RUB"));
                },
                null, 480);
    }

    @Test
    public void testCommissionPatchWithPromo() {
        doTestCommissionPatch(
                () -> {
                    var orderId = createOrder(CreateOrderParams.builder()
                            .serviceType(EServiceType.PT_BNOVO_HOTEL)
                            .payloadName(PAYLOAD_BNOVO)
                            .promo(PromoParams.builder()
                                    .promo(Tuple2.tuple(EPromoCodeNominalType.NT_PERCENT, BigDecimal.valueOf(20)))
                                    .build())
                            .build());
                    confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
                    clear(orderId);
                    return orderId;
                },
                paymentEvent -> {
                    assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
                    assertThat(paymentEvent.getPromoCodePartnerAmount()).isEqualTo(Money.of(2400, "RUB"));
                    assertThat(paymentEvent.getPartnerAmount()).isEqualTo(Money.of(8400, "RUB"));
                    assertThat(paymentEvent.getFeeAmount()).isEqualTo(Money.of(1200, "RUB"));
                },
                null, 480);
    }

    @Test
    public void testCommissionPatchWithPromoAndPlus() {
        doTestCommissionPatch(
                () -> {
                    var promocode = createSuccessPromoCode(Tuple2.tuple(EPromoCodeNominalType.NT_PERCENT,
                            BigDecimal.valueOf(20)), null);
                    var orderId = createOrder(CreateOrderParams.builder()
                            .serviceType(EServiceType.PT_BNOVO_HOTEL)
                            .payloadName(PAYLOAD_BNOVO)
                            .promoCode(promocode.getCode())
                            .payloadCustomizer(getBnovoPayloadCustomizerForYandexPlusWithdraw())
                            .build());
                    confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
                    clear(orderId);
                    return orderId;
                },
                paymentEvent -> {
                    assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
                    assertThat(paymentEvent.getPromoCodePartnerAmount()).isEqualTo(Money.of(2400, "RUB"));
                    assertThat(paymentEvent.getPartnerAmount()).isEqualTo(Money.of(6400, "RUB"));
                    assertThat(paymentEvent.getPlusPartnerAmount()).isEqualTo(Money.of(2000, "RUB"));
                    assertThat(paymentEvent.getFeeAmount()).isEqualTo(Money.of(1200, "RUB"));
                },
                null, 480);
    }

    @Test
    public void testCommissionPatchWithPromoPlusAndRefund() {
        doTestCommissionPatch(
                () -> {
                    var promocode = createSuccessPromoCode(Tuple2.tuple(EPromoCodeNominalType.NT_PERCENT,
                            BigDecimal.valueOf(20)), null);
                    var orderId = createOrder(CreateOrderParams.builder()
                            .serviceType(EServiceType.PT_BNOVO_HOTEL)
                            .payloadName(PAYLOAD_BNOVO)
                            .promoCode(promocode.getCode())
                            .payloadCustomizer(getBnovoPayloadCustomizerForYandexPlusWithdraw())
                            .build());
                    confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
                    clear(orderId);

                    performManualRefund(orderId, 3200, 6400);
                    waitNFinancialEvents(orderId, 2);

                    return orderId;
                },
                paymentEvent -> {
                    assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
                    assertThat(paymentEvent.getPromoCodePartnerAmount()).isEqualTo(Money.of(2400, "RUB"));
                    assertThat(paymentEvent.getPartnerAmount()).isEqualTo(Money.of(6400, "RUB"));
                    assertThat(paymentEvent.getPlusPartnerAmount()).isEqualTo(Money.of(2000, "RUB"));
                    assertThat(paymentEvent.getFeeAmount()).isEqualTo(Money.of(1200, "RUB"));
                },
                refundEvent -> {
                    assertThat(refundEvent.getTotalAmount()).isEqualTo(Money.of(8000, "RUB"));
                    assertThat(refundEvent.getPromoCodePartnerRefundAmount()).isEqualTo(Money.of(1600, "RUB"));
                    assertThat(refundEvent.getPartnerRefundAmount()).isEqualTo(Money.of(3600, "RUB"));
                    assertThat(refundEvent.getPlusPartnerRefundAmount()).isEqualTo(Money.of(2000, "RUB"));
                    assertThat(refundEvent.getFeeRefundAmount()).isEqualTo(Money.of(800, "RUB"));
                }, 160);
    }

    @Test
    public void testCommissionPatchWithFullRefund() {
        doTestCommissionPatch(
                () -> {
                    var orderId = createOrder(CreateOrderParams.builder()
                            .serviceType(EServiceType.PT_BNOVO_HOTEL)
                            .payloadName(PAYLOAD_BNOVO)
                            .build());
                    confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
                    clear(orderId);
                    performManualRefund(orderId, 0, 12000);
                    waitNFinancialEvents(orderId, 2);

                    return orderId;
                },
                paymentEvent -> {
                    assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
                    assertThat(paymentEvent.getPartnerAmount()).isEqualTo(Money.of(10800, "RUB"));
                    assertThat(paymentEvent.getFeeAmount()).isEqualTo(Money.of(1200, "RUB"));
                },
                refundEvent -> {
                    assertThat(refundEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
                    assertThat(refundEvent.getPartnerRefundAmount()).isEqualTo(Money.of(10800, "RUB"));
                    assertThat(refundEvent.getFeeRefundAmount()).isEqualTo(Money.of(1200, "RUB"));
                }, null);
    }

    @Test
    public void testCommissionPatchSameCommission() {
        doTestCommissionPatch(
                () -> {
                    var orderId = createOrder(CreateOrderParams.builder()
                            .serviceType(EServiceType.PT_BNOVO_HOTEL)
                            .payloadName(PAYLOAD_BNOVO)
                            .build());
                    confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
                    clear(orderId);

                    return orderId;
                },
                paymentEvent -> {
                    assertThat(paymentEvent.getTotalAmount()).isEqualTo(Money.of(12000, "RUB"));
                    assertThat(paymentEvent.getPartnerAmount()).isEqualTo(Money.of(10800, "RUB"));
                    assertThat(paymentEvent.getFeeAmount()).isEqualTo(Money.of(1200, "RUB"));
                },
                null, null, 10);
    }

    @Test
    public void testCommissionPatchAfterMove() {
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .build());
        confirm(orderId, PromoCodeCheckBehaviour.DO_NOT_CHECK);
        clear(orderId);

        when(balanceContractDictionary.findContractInfoByClientId(anyLong()))
                .thenReturn(ContractInfo.newBuilder().setClientId(-10000004L).setContractId(123).build());

        adminClient.moveHotelOrderToNewContract(TMoveHotelOrderToNewContractReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setClientId(-10000004L)
                .setContractId(123)
                .build());

        waitNFinancialEvents(orderId, 3);
        List<FinancialEvent> orderEvents = transactionTemplate.execute(txStatus ->
                financialEventRepository.findAll().stream()
                        .filter(e -> e.getOrder().getId().toString().equals(orderId))
                        .collect(Collectors.toList()));

        var oldClientEvents =
                orderEvents.stream().filter(e -> e.getBillingClientId() == -10000005L).collect(Collectors.toUnmodifiableList());
        var newClientEvents =
                orderEvents.stream().filter(e -> e.getBillingClientId() == -10000004L).collect(Collectors.toUnmodifiableList());
        FinancialEvent paymentEvent =
                newClientEvents.stream().filter(e -> e.getType() == PAYMENT).findAny().orElse(null);

        assertThat(oldClientEvents.size()).isEqualTo(2);
        assertThat(newClientEvents.size()).isEqualTo(1);

        var patchRsp = adminClient.patchPartnerCommission(TPatchPartnerCommissionReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId).build())
                .setTargetConfirmedRatePct(6)
                .setTargetRefundedRatePct(6)
                .setInferBillingIdsFromEvents(true)
                .build());

        assertThat(patchRsp.getRegisteredNewEvents()).isTrue();

        waitNFinancialEvents(orderId, 5);
        var newClientEventsNew =
                getFinancialEventsOfOrder(orderId).stream().filter(e -> e.getBillingClientId() == -10000004L).collect(Collectors.toUnmodifiableList());

        var orderEventsNew = newClientEventsNew.stream()
                .filter(x -> !x.getId().equals(paymentEvent.getId()))
                .collect(Collectors.toUnmodifiableList());
        FinancialEvent correctionPaymentEvent =
                orderEventsNew.stream().filter(e -> e.getType() == PAYMENT).findAny().orElse(null);
        FinancialEvent correctionRefundEvent =
                orderEventsNew.stream().filter(e -> e.getType() == REFUND).findAny().orElse(null);

        assertThat(correctionPaymentEvent).isNotNull();
        assertThat(correctionPaymentEvent.getBillingClientId()).isEqualTo(-10000004L);
        assertThat(correctionPaymentEvent.getBillingContractId()).isEqualTo(123);
        assertThat(correctionPaymentEvent.getTotalAmount()).isEqualTo(Money.of(480, "RUB"));
        assertThat(correctionPaymentEvent.getPartnerAmount()).isEqualTo(Money.of(480, "RUB"));
        assertThat(correctionPaymentEvent.getFeeAmount()).isEqualTo(Money.of(0, "RUB"));

        assertThat(correctionRefundEvent).isNotNull();
        assertThat(correctionRefundEvent.getBillingClientId()).isEqualTo(-10000004L);
        assertThat(correctionRefundEvent.getBillingContractId()).isEqualTo(123);
        assertThat(correctionRefundEvent.getOriginalEvent()).isEqualTo(paymentEvent);
        assertThat(correctionRefundEvent.getTotalAmount()).isEqualTo(Money.of(480, "RUB"));
        assertThat(correctionRefundEvent.getPartnerRefundAmount()).isEqualTo(Money.of(0, "RUB"));
        assertThat(correctionRefundEvent.getFeeRefundAmount()).isEqualTo(Money.of(480, "RUB"));
    }

    private void waitNFinancialEvents(String orderId, int n) {
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                r -> getFinancialEventsOfOrder(orderId).size() == n,
                Duration.ofSeconds(15), String.format("Order must have %d finevents", n));
    }

    @Test
    public void testWhiteLabelOrder() {
        TGetWhiteLabelPointsPropsRsp pointProps = TGetWhiteLabelPointsPropsRsp.newBuilder()
                .setPointsLinguistics(TWhiteLabelPointsLinguistics.newBuilder()
                        .setNameForNumeralNominative("миль")
                        .build())
                .build();
        when(promoServiceHelper.getWhiteLabelPointsProps(any(), anyInt()))
                .thenReturn(CompletableFuture.completedFuture(pointProps));

        String orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .payloadCustomizer(getBnovoPayloadCustomizerForWhiteLabel())
                .build());

        TGetOrderInfoRsp orderInfo = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(orderInfo.getResult().hasPriceInfo()).isTrue();
        assertThat(orderInfo.getResult().getPriceInfo().hasPromoCampaignsInfo()).isTrue();
        assertThat(orderInfo.getResult().getPriceInfo().getPromoCampaignsInfo().hasWhiteLabel()).isTrue();

        TWhiteLabelPromoCampaignInfo whiteLabelInfo =
                orderInfo.getResult().getPriceInfo().getPromoCampaignsInfo().getWhiteLabel();
        assertThat(whiteLabelInfo.getEligible()).isTrue();
        assertThat(whiteLabelInfo.getPoints().getAmount()).isEqualTo(2000);
        assertThat(whiteLabelInfo.getPoints().getPointsType()).isEqualTo(EWhiteLabelPointsType.WLP_S7);
        assertThat(whiteLabelInfo.getPointsLinguistics().getNameForNumeralNominative()).isEqualTo("миль");

        List<TOrderServiceInfo> services = orderInfo.getResult().getServiceList().stream().collect(toList());
        assertThat(services.size()).isGreaterThan(0);
        TJson payload = services.get(0).getServiceInfo().getPayload();
        HotelItinerary itinerary = ProtoUtils.fromTJson(payload, BNovoHotelItinerary.class);
        assertThat(itinerary.getAppliedPromoCampaigns()).isNotNull();
        assertThat(itinerary.getAppliedPromoCampaigns().getWhiteLabel()).isNotNull();
        assertThat(itinerary.getAppliedPromoCampaigns().getWhiteLabel().getCustomerNumber()).isEqualTo(
                "CustomerCardNumber");
    }

    private Function<String, String> getBnovoPayloadCustomizerForWhiteLabel() {
        return payloadCustomizer(BNovoHotelItinerary.class, itinerary -> {
            itinerary.setAppliedPromoCampaigns(AppliedPromoCampaigns.builder()
                    .whiteLabel(WhiteLabelApplication.builder()
                            .customerNumber("CustomerCardNumber")
                            .build())
                    .build());
            itinerary.setActivePromoCampaigns(PromoCampaignsInfo.builder()
                    .whiteLabel(WhiteLabelPromoCampaign.builder()
                            .eligible(EWhiteLabelEligibility.WLE_ELIGIBLE)
                            .partnerId(EWhiteLabelPartnerId.WL_S7)
                            .points(WhiteLabelPromoCampaign.WhiteLabelPoints.builder()
                                    .amount(2000)
                                    .pointsType(EWhiteLabelPointsType.WLP_S7)
                                    .build())
                            .pointsLinguistics(WhiteLabelPromoCampaign.WhiteLabelPointsLinguistics.builder()
                                    .nameForNumeralNominative("Устаревшие мили")
                                    .build())
                            .build())
                    .build());
        });
    }

    @Test
    public void testWhiteLabelDataForNoWhiteLabelOrder() {
        String orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_BNOVO_HOTEL)
                .payloadName(PAYLOAD_BNOVO)
                .payloadCustomizer(getBnovoPayloadCustomizerForNotEligibleWhiteLabel())
                .build());

        TGetOrderInfoRsp orderInfo = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(orderInfo.getResult().hasPriceInfo()).isTrue();
        assertThat(orderInfo.getResult().getPriceInfo().hasPromoCampaignsInfo()).isTrue();
        assertThat(orderInfo.getResult().getPriceInfo().getPromoCampaignsInfo().hasWhiteLabel()).isTrue();

        TWhiteLabelPromoCampaignInfo whiteLabelInfo = orderInfo.getResult().getPriceInfo().getPromoCampaignsInfo().getWhiteLabel();
        assertThat(whiteLabelInfo.getEligible()).isFalse();
        assertThat(whiteLabelInfo.hasPoints()).isFalse();
    }

    private void tryStartPayment(String orderId) {
        try {
            client.startPayment(TStartPaymentReq.newBuilder()
                    .setInvoiceType(EInvoiceType.IT_TRUST).setSource("desktop")
                    .setUseNewInvoiceModel(true)
                    .setOrderId(orderId).setReturnUrl("some_return_url").build());
            log.info("startPayment call success");
        } catch (Exception e) {
            log.info("startPayment call error");
        }
    }

    private Function<String, String> getBnovoPayloadCustomizerForNotEligibleWhiteLabel() {
        return payloadCustomizer(BNovoHotelItinerary.class, itinerary -> {
            itinerary.setActivePromoCampaigns(PromoCampaignsInfo.builder()
                    .whiteLabel(WhiteLabelPromoCampaign.builder()
                            .eligible(EWhiteLabelEligibility.WLE_UNUSED)
                            .partnerId(EWhiteLabelPartnerId.WL_UNKNOWN)
                            .points(null)
                            .pointsLinguistics(null)
                            .build())
                    .build());
        });
    }
}
