package ru.yandex.travel.orders.integration.hotels;

import java.time.Duration;

import org.junit.Test;
import org.springframework.boot.test.context.SpringBootTest;

import ru.yandex.travel.orders.commons.proto.EPaymentOutcome;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;

import static ru.yandex.travel.hotels.common.partners.expedia.ApiVersion.V2_4;

/**
 * The test uses {@link ru.yandex.travel.orders.services.mock.MockDBTrustClient} instead of
 * {@link ru.yandex.travel.orders.services.mock.MockTrustClient}
 */
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "quartz.enabled=true",
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "trust-hotels.clearing-refresh-timeout=100ms",
                "trust-hotels.payment-refresh-timeout=100ms",
                "trust-hotels.schedule-clearing-rate=100ms",
                "single-node.auto-start=true",
                "hotel-workflow.check-expiration-task.enabled=true",
                "hotel-workflow.check-expiration-task.period=100ms",
                "financial-events.enabled=true",
                "mock-trust.enabled=false"
        }
)
@SuppressWarnings("ResultOfMethodCallIgnored")
public class HotelOrderFlowTestsMockDbTrust extends AbstractHotelOrderFlowTest {

    @Test
    public void testExpediaPaymentFailed() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .paymentTestContext(getFailedPaymentContext())
                .build());

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getHotelOrderState() == EHotelOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(3), "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setUseNewInvoiceModel(true)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        TGetOrderInfoRsp waitingOrder = IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED, Duration.ofSeconds(10)
                , "Order must have failed payment state");

    }

    @Test
    public void testExpediaPaymentSuccess() {
        initializeExpediaMocksForConfirmedAndRefunded(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .paymentTestContext(getSuccessPaymentContext())
                .build());

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getHotelOrderState() == EHotelOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(3), "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setUseNewInvoiceModel(true)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url


        TGetOrderInfoRsp confirmedOrder = IntegrationUtils.waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getHotelOrderState() == EHotelOrderState.OS_CONFIRMED, Duration.ofSeconds(10)
                , "Order must be in OS_CONFIRMED state");

    }

    private TPaymentTestContext getFailedPaymentContext() {
        return TPaymentTestContext.newBuilder().setPaymentOutcome(EPaymentOutcome.PO_FAILURE).build();
    }


    private TPaymentTestContext getSuccessPaymentContext() {
        return TPaymentTestContext.newBuilder().setPaymentOutcome(EPaymentOutcome.PO_SUCCESS).build();
    }

}
