package ru.yandex.travel.orders.integration.hotels;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.temporal.ChronoUnit;

import org.javamoney.moneta.Money;
import org.junit.Test;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.promo.PromoAction;
import ru.yandex.travel.orders.proto.TEstimateDiscountReq;
import ru.yandex.travel.orders.proto.TEstimateDiscountRsp;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.testing.misc.TestResources;

import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.travel.hotels.common.partners.expedia.ApiVersion.V2_4;

public class PromoCodeFlowTests extends AbstractHotelOrderFlowTest {

    @Test
    public void testEstimateDiscount() {

        createSuccessPromoCode(HUNDRED_RUBLES_PROMO, null);

        TEstimateDiscountReq req = TEstimateDiscountReq.newBuilder()
                .addService(TEstimateDiscountReq.TEstimateDiscountService.newBuilder()
                        .setSourcePayload(
                                TJson.newBuilder().setValue(TestResources.readResource(
                                        "integration/hotels/expedia.json"
                                )).build()
                        )
                        .setServiceType(EServiceType.PT_EXPEDIA_HOTEL)
                        .build())
                .addPromoCode(promoCodeString)
                .addPromoCode("NOT_FOUND")
                .build();

        TEstimateDiscountRsp rsp = client.estimateDiscount(req);
        assertThat(rsp.getPromoCodeApplicationResultsList().size()).isEqualTo(2);
        assertThat(rsp.getPromoCodeApplicationResults(0)).satisfies(pc -> {
            assertThat(pc.getCode()).isEqualTo(promoCodeString);
            assertThat(pc.getType()).isEqualTo(EPromoCodeApplicationResultType.ART_SUCCESS);
            assertThat(ProtoUtils.fromTPrice(pc.getDiscountAmount())).isEqualTo(Money.of(HUNDRED_RUBLES_PROMO.get2(),
                    ProtoCurrencyUnit.RUB));
        });
        assertThat(rsp.getPromoCodeApplicationResults(1)).satisfies(pc -> {
            assertThat(pc.getCode()).isEqualTo("NOT_FOUND");
            assertThat(pc.getType()).isEqualTo(EPromoCodeApplicationResultType.ART_NOT_FOUND);
        });
    }

    @Test
    public void testEstimateDiscountBudgeted() {

        BigDecimal budget = BigDecimal.valueOf(100);
        createSuccessPromoCode(HUNDRED_RUBLES_PROMO, null, budget);

        TEstimateDiscountReq req = TEstimateDiscountReq.newBuilder()
                .addService(TEstimateDiscountReq.TEstimateDiscountService.newBuilder()
                        .setSourcePayload(
                                TJson.newBuilder().setValue(TestResources.readResource(
                                        "integration/hotels/expedia.json"
                                )).build()
                        )
                        .setServiceType(EServiceType.PT_EXPEDIA_HOTEL)
                        .build())
                .addPromoCode(promoCodeString)
                .addPromoCode("NOT_FOUND")
                .build();

        TEstimateDiscountRsp rsp = client.estimateDiscount(req);
        assertThat(rsp.getPromoCodeApplicationResultsList().size()).isEqualTo(2);
        assertThat(rsp.getPromoCodeApplicationResults(0)).satisfies(pc -> {
            assertThat(pc.getCode()).isEqualTo(promoCodeString);
            assertThat(pc.getType()).isEqualTo(EPromoCodeApplicationResultType.ART_SUCCESS);
            assertThat(ProtoUtils.fromTPrice(pc.getDiscountAmount())).isEqualTo(Money.of(HUNDRED_RUBLES_PROMO.get2(),
                    ProtoCurrencyUnit.RUB));
        });
        assertThat(rsp.getPromoCodeApplicationResults(1)).satisfies(pc -> {
            assertThat(pc.getCode()).isEqualTo("NOT_FOUND");
            assertThat(pc.getType()).isEqualTo(EPromoCodeApplicationResultType.ART_NOT_FOUND);
        });
    }

    @Test
    public void testEstimateDiscountEmptyBudget() {

        BigDecimal budget = BigDecimal.TEN;
        createSuccessPromoCode(HUNDRED_RUBLES_PROMO, null, budget);

        TEstimateDiscountReq req = TEstimateDiscountReq.newBuilder()
                .addService(TEstimateDiscountReq.TEstimateDiscountService.newBuilder()
                        .setSourcePayload(
                                TJson.newBuilder().setValue(TestResources.readResource(
                                        "integration/hotels/expedia.json"
                                )).build()
                        )
                        .setServiceType(EServiceType.PT_EXPEDIA_HOTEL)
                        .build())
                .addPromoCode(promoCodeString)
                .addPromoCode("NOT_FOUND")
                .build();

        TEstimateDiscountRsp rsp = client.estimateDiscount(req);
        assertThat(rsp.getPromoCodeApplicationResultsList().size()).isEqualTo(2);
        assertThat(rsp.getPromoCodeApplicationResults(0)).satisfies(pc -> {
            assertThat(pc.getCode()).isEqualTo(promoCodeString);
            assertThat(pc.getType()).isEqualTo(EPromoCodeApplicationResultType.ART_EMPTY_BUDGET);
        });
        assertThat(rsp.getPromoCodeApplicationResults(1)).satisfies(pc -> {
            assertThat(pc.getCode()).isEqualTo("NOT_FOUND");
            assertThat(pc.getType()).isEqualTo(EPromoCodeApplicationResultType.ART_NOT_FOUND);
        });
    }

    @Test
    public void whenPromoCodeIsNotApplicableAfterPaymentTheOrderIsRefunded() throws InterruptedException {
        initializeExpediaMocksForOrderReservedAndCancelled(V2_4);
        // on reservation the promo code is checked but not fully applied (counted) yet
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .testContext(getAllSuccessfulContext())
                .promo(PromoParams.builder().build())
                .build());
        reserveOrder(orderId);

        // make the code not applicable between reservation and payment
        transactionTemplate.executeWithoutResult(ignored -> {
            PromoAction promoAction = promoCodeRepository.findByCodeEquals(promoCodeString).getPromoAction();
            promoAction.setValidTill(Instant.now().minus(1, ChronoUnit.MILLIS));
            promoActionRepository.save(promoAction);
        });

        // the promo code is not applicable at the moment of payment
        startPayment(orderId);
        authorizePayment(orderId);
        // it causes order cancellation
        waitForOrderState(orderId, EHotelOrderState.OS_CANCELLED);
        assertThat(getOrderBalance(orderId).getNumberStripped())
                .isEqualTo(BigDecimal.ZERO);
        checkPromoCodeActivated(0);
    }

    @Test
    public void whenReservationFailsPromoCodeGetsReturned() {
        initializeExpediaMocksForOrderConfirmFailed(V2_4);
        var orderId = createOrder(CreateOrderParams.builder()
                .serviceType(EServiceType.PT_EXPEDIA_HOTEL)
                .payloadName(PAYLOAD_EXPEDIA)
                .testContext(getContextFailedOnConfirmation())
                .promo(PromoParams.builder().build())
                .build());
        confirmFailed(orderId);
        checkPromoCodeActivated(0);
    }

}
