package ru.yandex.travel.orders.integration.invoice;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.List;
import java.util.UUID;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import org.javamoney.moneta.Money;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.admin.proto.OrdersAdminInterfaceV1Grpc;
import ru.yandex.travel.orders.admin.proto.TRetryMoneyRefundReq;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.Account;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.context.OrderStateContext;
import ru.yandex.travel.orders.grpc.OrdersAdminService;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.integration.train.factories.TrainOrderItemFactory;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.repository.AccountRepository;
import ru.yandex.travel.orders.repository.InvoiceRepository;
import ru.yandex.travel.orders.repository.MoneyRefundRepository;
import ru.yandex.travel.orders.repository.OrderRefundRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.payments.InvoicePaymentFlags;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponseOrder;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustRefundState;
import ru.yandex.travel.orders.services.payments.model.TrustRefundStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustResponseStatus;
import ru.yandex.travel.orders.services.payments.model.TrustStartRefundResponse;
import ru.yandex.travel.orders.services.train.TrainRefundLogService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentRefund;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.trust.refund.proto.ETrustRefundState;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

@SuppressWarnings("ResultOfMethodCallIgnored")
@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "quartz.enabled=true",
                "single-node.auto-start=true",
                "workflow-processing.pending-workflow-polling-interval=100ms",
        }
)
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@DirtiesContext
@ActiveProfiles("test")
public class InvoiceFlowTest {
    private static final String SESSION_KEY = "qwerty";
    private static final String YANDEX_UID = "1234567890";

    @Autowired
    private TrustClient trustClient;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private OrderRefundRepository orderRefundRepository;

    @Autowired
    private MoneyRefundRepository moneyRefundRepository;

    @Autowired
    private TrainOrderItemRepository trainOrderItemRepository;

    @Autowired
    private InvoiceRepository invoiceRepository;

    @Autowired
    private WorkflowRepository workflowRepository;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private WorkflowMessageSender workflowMessageSender;

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();

    @Autowired
    private OrdersService ordersService;
    @Autowired
    private OrdersAdminService ordersAdminService;

    private Context context;

    private OrdersAdminInterfaceV1Grpc.OrdersAdminInterfaceV1BlockingStub adminClient;

    @MockBean
    private NotificationHelper notificationHelper;

    @MockBean
    private TrainRefundLogService trainRefundLogService;

    @Before
    public void setUpCredentialsContext() {
        UserCredentialsBuilder userCredentialsBuilder = new UserCredentialsBuilder();
        UserCredentials credentials = userCredentialsBuilder.build(SESSION_KEY, YANDEX_UID, "passport-id-1", "user1",
                null, "127.0.0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();
        adminClient = IntegrationUtils.createAdminServerAndBlockingStub(cleanupRule, ordersAdminService);
    }

    @After
    public void tearDownCredentialsContext() {
        Context.current().detach(context);
    }

    @Test
    @Ignore("TODO TRAVELBACK-3391 flaky https://paste.yandex-team.ru/9233979")
    public void testRetryCrashedRefundPayment() throws InterruptedException {
        when(trustClient.createRefund(any(), any())).thenThrow(new RuntimeException("test createRefund error"));
        UUID orderId = transactionTemplate.execute(i -> {
            var order = this.createConfirmedOrder();
            var invoice = this.createClearedInvoice(order);
            var orderRefund = this.createOrderRefund(order);
            order.setState(EOrderState.OS_REFUNDING);
            var refundMsg = TPaymentRefund.newBuilder().setReason("Refund")
                    .setOrderRefundId(orderRefund.getId().toString())
                    .putTargetFiscalItems(222222L, ProtoUtils.toTPrice(Money.of(2000, ProtoCurrencyUnit.RUB)))
                    .build();
            workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(), refundMsg);
            return order.getId();
        });
        TestUtils.waitForCheck(Duration.ofSeconds(2), () -> transactionTemplate.execute(i -> {
            //noinspection ConstantConditions
            var order = orderRepository.getOne(orderId);
            var invoice = order.getCurrentInvoice();
            assertThat(order.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(invoice.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(invoice.getInvoiceState()).isEqualTo(ETrustInvoiceState.IS_REFUNDING);
            var refund = invoice.getActiveTrustRefund();
            assertThat(refund).isNotNull();
            assertThat(refund.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_CRASHED);
            return null;
        }));

        Mockito.reset(trustClient);
        mockTrustClientForRefund(TrustRefundState.SUCCESS);

        //noinspection ConstantConditions
        adminClient.retryMoneyRefund(TRetryMoneyRefundReq.newBuilder().setOrderId(orderId.toString()).build());
        TestUtils.waitForCheck(Duration.ofSeconds(4), () -> transactionTemplate.execute(i -> {
            var order = orderRepository.getOne(orderId);
            var invoice = order.getCurrentInvoice();
            assertThat(order.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(invoice.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(invoice.getInvoiceState()).isEqualTo(ETrustInvoiceState.IS_CLEARED);
            var refund = invoice.getActiveTrustRefund();
            assertThat(refund).isNotNull();
            assertThat(refund.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(refund.getState()).isEqualTo(ETrustRefundState.RS_SUCCESS);
            return null;
        }));
    }

    @Test
    @Ignore("TODO TRAVELBACK-3391 flaky https://paste.yandex-team.ru/8446169")
    public void testRetryErrorRefundPayment() throws InterruptedException {
        mockTrustClientForRefund(TrustRefundState.ERROR);
        UUID orderId = transactionTemplate.execute(i -> {
            var order = this.createConfirmedOrder();
            var invoice = this.createClearedInvoice(order);
            var orderRefund = this.createOrderRefund(order);
            order.setState(EOrderState.OS_REFUNDING);
            var refundMsg = TPaymentRefund.newBuilder().setReason("Refund")
                    .setOrderRefundId(orderRefund.getId().toString())
                    .putTargetFiscalItems(222222L, ProtoUtils.toTPrice(Money.of(2000, ProtoCurrencyUnit.RUB)))
                    .build();
            workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(), refundMsg);
            return order.getId();
        });
        TestUtils.waitForCheck(Duration.ofSeconds(4), () -> transactionTemplate.execute(i -> {
            //noinspection ConstantConditions
            var order = orderRepository.getOne(orderId);
            var invoice = order.getCurrentInvoice();
            assertThat(order.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(invoice.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(invoice.getInvoiceState()).isEqualTo(ETrustInvoiceState.IS_REFUNDING);
            var refund = invoice.getActiveTrustRefund();
            assertThat(refund).isNotNull();
            assertThat(refund.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(refund.getState()).isEqualTo(ETrustRefundState.RS_ERROR);
            return null;
        }));

        Mockito.reset(trustClient);
        mockTrustClientForRefund(TrustRefundState.SUCCESS);

        //noinspection ConstantConditions
        adminClient.retryMoneyRefund(TRetryMoneyRefundReq.newBuilder().setOrderId(orderId.toString()).build());

        TestUtils.waitForCheck(Duration.ofSeconds(5), () -> transactionTemplate.execute(i -> {
            var order = orderRepository.getOne(orderId);
            var invoice = order.getCurrentInvoice();
            assertThat(order.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(invoice.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(invoice.getInvoiceState()).isEqualTo(ETrustInvoiceState.IS_CLEARED);
            var refund = invoice.getActiveTrustRefund();
            assertThat(refund).isNotNull();
            assertThat(refund.getWorkflow().getState()).isEqualTo(EWorkflowState.WS_RUNNING);
            assertThat(refund.getState()).isEqualTo(ETrustRefundState.RS_SUCCESS);
            return null;
        }));
    }

    private void mockTrustClientForRefund(TrustRefundState refundState) {
        var createRefundRsp = new TrustCreateRefundResponse();
        createRefundRsp.setStatus(TrustResponseStatus.WAIT_FOR_NOTIFICATION);
        createRefundRsp.setTrustRefundId("refund1");
        when(trustClient.createRefund(any(), any())).thenReturn(createRefundRsp);
        var startRefundRsp = new TrustStartRefundResponse();
        startRefundRsp.setStatus(TrustResponseStatus.SUCCESS);
        startRefundRsp.setFiscalReceiptUrl("trust-test.ya.ru/any-check");
        when(trustClient.startRefund(anyString(), any())).thenReturn(startRefundRsp);
        var refundStatusRsp = new TrustRefundStatusResponse();
        refundStatusRsp.setStatus(refundState);
        refundStatusRsp.setFiscalReceiptUrl("trust-test.ya.ru/any-check");
        refundStatusRsp.setStatusDesc("ok");
        when(trustClient.getRefundStatus(anyString(), any())).thenReturn(refundStatusRsp);
        var basketStatusResponse = new TrustBasketStatusResponse();
        basketStatusResponse.setPaymentStatus(PaymentStatusEnum.CLEARED);
        var trustOrder = new TrustBasketStatusResponseOrder();
        trustOrder.setOrderId("trust-order-id-1");
        trustOrder.setOrigAmount(BigDecimal.valueOf(5000));
        trustOrder.setPaidAmount(BigDecimal.valueOf(2000));
        basketStatusResponse.setOrders(List.of(trustOrder));
        when(trustClient.getBasketStatus(anyString(), any())).thenReturn(basketStatusResponse);
    }

    private GenericOrderUserRefund createOrderRefund(GenericOrder order) {
        var orderRefund = new GenericOrderUserRefund();
        orderRefund.setOrder(order);
        orderRefund.setId(UUID.randomUUID());
        orderRefund.setState(EOrderRefundState.RS_WAITING_INVOICE_REFUND);
        orderRefund = orderRefundRepository.saveAndFlush(orderRefund);
        var moneyRefund = new MoneyRefund();
        moneyRefund.setOrderRefundId(orderRefund.getId());
        moneyRefund.setState(MoneyRefundState.IN_PROGRESS);
        order.addMoneyRefund(moneyRefund);
        return orderRefund;
    }

    private TrustInvoice createClearedInvoice(Order order) {
        Account account = Account.createAccount(ProtoCurrencyUnit.RUB);
        account = accountRepository.saveAndFlush(account);

        var invoice = TrustInvoice.createInvoice(order, AuthorizedUser.create(order.getId(), new UserCredentials(
                        "session_key", "yandex_uid", null, "test@yandex-team.ru", null, "127.0.0.1", false, false),
                AuthorizedUser.OrderUserRole.OWNER), InvoicePaymentFlags.builder()
                .force3ds(true)
                .useMirPromo(false)
                .processThroughYt(false)
                .build());
        var item = new InvoiceItem();
        item.setFiscalItemId(222222L);
        item.setFiscalItemType(FiscalItemType.TRAIN_TICKET);
        item.setPrice(BigDecimal.valueOf(5000));
        item.setClearedSum(BigDecimal.valueOf(5000));
        item.setRefundedSum(BigDecimal.ZERO);
        item.setFiscalNds(VatType.VAT_20);
        item.setFiscalInn("11111111");
        item.setTrustOrderId("trust-order-id-1");
        invoice.addInvoiceItem(item);
        invoice.setState(ETrustInvoiceState.IS_CLEARED);
        invoice.setAccount(account);
        invoice.setPurchaseToken("purchase-token");
        invoice = invoiceRepository.saveAndFlush(invoice);

        Workflow workflow = Workflow.createWorkflowForEntity(invoice, order.getWorkflow().getId());
        workflowRepository.saveAndFlush(workflow);
        return invoice;
    }

    private GenericOrder createConfirmedOrder() {
        var order = new GenericOrder();
        order.setState(EOrderState.OS_CONFIRMED);
        order.setStateContext(new OrderStateContext());
        order.setDisplayType(EDisplayOrderType.DT_TRAIN);
        order.setId(UUID.randomUUID());
        order.setPrettyId(UUID.randomUUID().toString());
        order.setCurrency(ProtoCurrencyUnit.RUB);
        Workflow orderWorkflow = Workflow.createWorkflowForEntity(order);
        orderWorkflow.setSupervisorId(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(orderWorkflow);
        order = orderRepository.saveAndFlush(order);
        var factory = new TrainOrderItemFactory();
        factory.setOrderItemState(EOrderItemState.IS_CONFIRMED);
        var orderItem = factory.createTrainOrderItem();
        order.addOrderItem(orderItem);
        Workflow orderItemWorkflow = Workflow.createWorkflowForEntity(orderItem, orderWorkflow.getId());
        workflowRepository.saveAndFlush(orderItemWorkflow);
        trainOrderItemRepository.saveAndFlush(orderItem);
        return order;
    }


    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient trainTrustClient() {
            return mock(TrustClient.class);
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trainTrustClient) {
            return paymentProfile -> trainTrustClient;
        }
    }
}
