package ru.yandex.travel.orders.integration.notification;

import java.time.Instant;
import java.util.Base64;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;

import com.fasterxml.jackson.databind.node.POJONode;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.EmailChannelInfo;
import ru.yandex.travel.orders.entities.notifications.ImAttachmentProviderData;
import ru.yandex.travel.orders.entities.notifications.MailSenderAttachment;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.entities.notifications.trains.TrainConfirmedMailSenderArgs;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.MailSenderService;
import ru.yandex.travel.orders.services.cloud.s3.InMemoryS3Object;
import ru.yandex.travel.orders.services.cloud.s3.S3Service;
import ru.yandex.travel.orders.workflow.notification.proto.EAttachmentState;
import ru.yandex.travel.orders.workflow.notification.proto.ENotificationState;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientIOException;
import ru.yandex.travel.train.partners.im.model.OrderReservationBlankRequest;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.exceptions.RetryableException;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.testing.misc.MockitoUtils.voidAnswer;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "notification.preparing-expired-task-initial-start-delay=50ms",
                "notification.preparing-expired-task-period=300ms",
                "single-node.auto-start=true"
        }
)
@DirtiesContext
@ActiveProfiles("test")
public class NotificationFlowTest {
    @Autowired
    private NotificationRepository notificationRepository;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private WorkflowRepository workflowRepository;

    @MockBean
    private MailSenderService mailSenderService;

    @MockBean
    private ImClient imClient;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private WorkflowMessageSender workflowProcessService;

    @MockBean
    private S3Service s3Service;

    @Test
    public void testMailSent() {
        var blankRequest1 = new OrderReservationBlankRequest(1111, 1);
        var blankRequest2 = new OrderReservationBlankRequest(1111, 2);
        AtomicReference<InMemoryS3Object> attachmentS3Data = new AtomicReference<>();
        when(imClient.orderReservationBlank(eq(blankRequest1), any())).thenReturn(Base64.getDecoder().decode("pdfdata12345"));
        when(imClient.orderReservationBlank(eq(blankRequest2), any())).thenThrow(new ImClientIOException("failed"));

        doThrow(new RetryableException("testMailSent retryable write exception"))
                .doAnswer(voidAnswer(inv -> attachmentS3Data.set(inv.getArgument(0))))
                .when(s3Service).uploadObject(any());
        when(s3Service.readObject(any()))
                .thenThrow(new RetryableException("testMailSent retryable read exception"))
                .thenAnswer(inv -> attachmentS3Data.get());

        UUID emailId = transactionTemplate.execute(i -> {
            var order = createOrder();
            var email = createEmailNotification(order);
            Workflow workflow = Workflow.createWorkflowForEntity(email,
                    WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
            workflow = workflowRepository.saveAndFlush(workflow);
            var attachment1 = Attachment.createImAttachment(email, "1.pdf", "application/pdf", true,
                    new ImAttachmentProviderData(blankRequest1));
            var attachment2 = Attachment.createImAttachment(email, "2.pdf", "application/pdf", false,
                    new ImAttachmentProviderData(blankRequest2));
            workflowRepository.saveAndFlush(Workflow.createWorkflowForEntity(attachment1, workflow.getId()));
            workflowRepository.saveAndFlush(Workflow.createWorkflowForEntity(attachment2, workflow.getId()));
            workflowProcessService.scheduleEvent(workflow.getId(), TSend.newBuilder().build());
            notificationRepository.save(email);
            return email.getId();
        });

        //noinspection ConstantConditions
        TestUtils.waitForState("Email is sent", () -> transactionTemplate.execute(
                s -> notificationRepository.getOne(emailId).getState() == ENotificationState.NS_SENT));

        transactionTemplate.execute(i -> {
            @SuppressWarnings("ConstantConditions")
            var email = notificationRepository.getOne(emailId);
            Attachment attachment1 = null;
            Attachment attachment2 = null;
            for (var a : email.getAttachments()) {
                if (a.getFilename().equals("1.pdf")) {
                    attachment1 = a;
                } else if (a.getFilename().equals("2.pdf")) {
                    attachment2 = a;
                }
            }

            assertThat(email.getState()).isEqualTo(ENotificationState.NS_SENT);
            assertThat(attachment1).isNotNull();
            assertThat(attachment1.getState()).isEqualTo(EAttachmentState.AS_FETCHED);
            assertThat(attachment2).isNotNull();
            assertThat(attachment2.getState()).isEqualTo(EAttachmentState.AS_NEW);
            assertThat(email.isExpired()).isTrue();
            return null;
        });
        verify(mailSenderService).sendEmailSync(any(), eq("train@test.com"), any(), any(), any(), any(),
                eq(List.of(new MailSenderAttachment("1.pdf", "application/pdf", "pdfdata12345"))));
        verify(s3Service, times(2)).uploadObject(argThat(o -> o.getId().startsWith("attachments/im_vouchers/1.pdf.") &&
                o.getData().length > 5 && o.getMimeType().equals("application/pdf") && o.getFileName().equals("1.pdf")));
        verify(s3Service, times(2)).readObject(argThat(id -> id.startsWith("attachments/im_vouchers/1.pdf.")));
    }

    private Notification createEmailNotification(TrainOrder order) {
        var channelInfo = new EmailChannelInfo();
        channelInfo.setTarget("train@test.com");
        channelInfo.setArguments(new POJONode(new TrainConfirmedMailSenderArgs()));
        var email = Notification.createEmailNotification(order, channelInfo);
        email.setPreparingAttachmentsTill(Instant.now().plusSeconds(1));
        email = notificationRepository.saveAndFlush(email);
        return email;
    }

    private TrainOrder createOrder() {
        var order = new TrainOrder();
        order.setState(ETrainOrderState.OS_NEW);
        order.setDisplayType(EDisplayOrderType.DT_TRAIN);
        order.setId(UUID.randomUUID());
        order.setPrettyId(order.getId().toString());
        Workflow orderWorkflow = Workflow.createWorkflowForEntity(order);
        orderWorkflow.setSupervisorId(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(orderWorkflow);
        order = orderRepository.saveAndFlush(order);
        return order;
    }
}
