package ru.yandex.travel.orders.integration.startrek;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;

import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Service;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.entities.Ticket;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.TicketRepository;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqData;
import ru.yandex.travel.orders.workflow.ticket.proto.ETicketState;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.WorkflowMessagePair;
import ru.yandex.travel.workflow.WorkflowProcessService;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "quartz.enabled=true"
        }
)
//@TestExecutionListeners(listeners = TruncateDatabaseTestExecutionListener.class, mergeMode =
//        TestExecutionListeners.MergeMode.MERGE_WITH_DEFAULTS)
@ActiveProfiles("dev")
@DirtiesContext
// for local dev tests only
@Ignore
@Slf4j
public class StarTrekIntegrationTest {
    @Autowired
    private StarTrekService starTrekService;
    @Autowired
    private TicketRepository ticketRepository;
    @Autowired
    private TransactionTemplate txTemplate;
    @Autowired
    private TestMessagingContext fakeMessageContext;

    @Test(timeout = 60 * 60 * 1000)
    public void testAeroflotLostMessage() throws Exception {
        AeroflotMqData order = AeroflotMqData.builder()
                .bookingDate(ZonedDateTime.parse("2007-12-03T10:15:30+03:00[Europe/Moscow]"))
                .sentDate(ZonedDateTime.parse("2019-12-03T10:15:30-05:00[CST6CDT]"))
                .pnr("TLGSTTST")
                //.tickets(null)
                .tickets(List.of("t1", "t2"))
                .sourceMessage("<some>\n    xml %%%message%%% :)\n</some>")
                .build();
        UUID ticketId = txTemplate.execute(t -> starTrekService.createIssueForAeroflotLostOrder(order,
                fakeMessageContext));
        while (processTicketInTx(ticketId, Ticket::getState) != ETicketState.TS_CREATED) {
            Thread.sleep(100);
        }
        String issueId = processTicketInTx(ticketId, Ticket::getIssueId);
        String issueUrl = "https://st.yandex-team.ru/" + issueId;
        log.info("The ticket has been created: id={}, url={}", issueId, issueUrl);
    }

    private <T> T processTicketInTx(UUID ticketId, Function<Ticket, T> processor) {
        return txTemplate.execute(t -> processor.apply(ticketRepository.getOne(ticketId)));
    }

    @TestConfiguration
    static class StarTrekIntegrationTestConfiguration {
        @Bean
        public TestMessagingContext testMessagingContext(WorkflowProcessService workflowProcessService) {
            return new TestMessagingContext(workflowProcessService);
        }
    }

    @Service
    @RequiredArgsConstructor
    static class TestMessagingContext implements MessagingContext {
        private final WorkflowProcessService workflowProcessService;

        @Override
        public UUID getWorkflowId() {
            throw new UnsupportedOperationException("unsupported");
        }

        @Override
        public Integer getWorkflowVersion() {
            throw new UnsupportedOperationException("unsupported");
        }

        @Override
        public Object getWorkflowEntity() {
            throw new UnsupportedOperationException("unsupported");
        }

        @Override
        public void scheduleEvent(Message event) {
            throw new UnsupportedOperationException("unsupported");
        }

        @Override
        public void scheduleExternalEvent(UUID workflowId, Message event) {
            workflowProcessService.scheduleEvent(workflowId, event);
        }

        @Override
        public List<WorkflowMessagePair> getScheduledEvents() {
            throw new UnsupportedOperationException("unsupported");
        }

        @Override
        public int getAttempt() {
            throw new UnsupportedOperationException();
        }
    }
}
