package ru.yandex.travel.orders.integration.suburban;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Predicate;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import lombok.Data;
import lombok.experimental.Accessors;
import org.javamoney.moneta.Money;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.suburban.SuburbanNotificationHelper;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.suburban.model.SuburbanReservation;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.verify;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;


@SuppressWarnings("ResultOfMethodCallIgnored")
@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=10ms",
                "single-node.auto-start=true",
                "suburban.providers.movista.common.book-max-attempts=3",
                "suburban.providers.movista.common.confirm-max-attempts=3",
                "suburban.providers.im.common.book-max-attempts=3",
                "suburban.providers.im.common.confirm-max-attempts=3",
                "suburban.providers.im.get-ticket-barcode-max-attempts=3",
                "suburban.providers.im.get-ticket-barcode-timeout=1s",
                "suburban.providers.aeroexpress.common.book-max-attempts=3",
                "suburban.providers.aeroexpress.common.confirm-max-attempts=3"
        }
)
@DirtiesContext
@ActiveProfiles("test")
public abstract class AbstractSuburbanOrderFlowTests {
    protected String SESSION_KEY = "qwerty";
    protected String YANDEX_UID = "1234567890";
    protected UserCredentialsBuilder userCredentialsBuilder = new UserCredentialsBuilder();

    protected int PROVIDER_REQUEST_MAX_ATTEMPTS = 3;

    public Context context;
    public OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub orcApi;

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();

    @Autowired
    protected OrdersService ordersService;
    @Autowired
    protected OrderRepository orderRepository;
    @Autowired
    protected TransactionTemplate transactionTemplate;
    @Autowired
    protected TrustClient trustClient;

    @MockBean
    protected SuburbanNotificationHelper suburbanNotificationHelper;

    protected static final Duration bookTtl = Duration.ofMinutes(30);

    @Before
    public void setUp() {
        UserCredentials credentials = userCredentialsBuilder.build(SESSION_KEY, YANDEX_UID, null, null, null, "127.0" +
                ".0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();
        orcApi = IntegrationUtils.createServerAndBlockingStub(cleanupRule, ordersService);
    }

    @After
    public void tearDown() {
        Context.current().detach(context);
    }

    /**
     * Emulate TravelApi behavior:
     * orcApi.createOrder + orcApi.reserve are called from TravelApi/create_order
     */
    protected String travelApiCreateOrder(TCreateOrderReq orderReq) {
        TCreateOrderRsp createOrderResp = orcApi.createOrder(orderReq);
        String orderId = createOrderResp.getNewOrder().getOrderId();
        orcApi.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        return orderId;
    }

    /**
     * Emulate TravelApi behavior:
     * orcApi.checkout + orcApi.startPayment are called from TravelApi/start_payment
     */
    protected void travelApiStartPayment(String orderId) {
        orcApi.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        orcApi.startPayment(TStartPaymentReq.newBuilder()
                .setInvoiceType(EInvoiceType.IT_TRUST)
                .setOrderId(orderId)
                .setSource("mobile")
                .setReturnUrl("some_return_url").build());
    }

    protected TCreateOrderReq buildCreateOrderRequest(SuburbanReservation reservation) {
        return TCreateOrderReq.newBuilder()
                .setDeduplicationKey(UUID.randomUUID().toString())
                .setOrderType(EOrderType.OT_GENERIC)
                .setOwner(TUserInfo.newBuilder()
                        .setEmail("test@test.com")
                        .setPhone("+79111111111")
                        .setYandexUid(YANDEX_UID)
                        .setGeoId(213)
                        .setIp("192.168.0.1")
                )
                .setLabel("label42")
                .setCurrency(ECurrency.C_RUB)
                .addCreateServices(TCreateServiceReq.newBuilder()
                        .setServiceType(EServiceType.PT_SUBURBAN)
                        .setSourcePayload(ProtoUtils.toTJson(reservation))).build();
    }

    public void trustAuthorizePayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    protected void waitStateAndCheck(Predicate<TGetOrderInfoRsp> predicate, String description,
                                   OrderCheckConfig checkConfig) {
        TGetOrderInfoRsp resp = waitForPredicateOrTimeout(
                orcApi, checkConfig.orderId,
                predicate, Duration.ofMillis(20000), description);

        TOrderInfo order = resp.getResult();
        assertThat(order.getOrderType()).isEqualTo(EOrderType.OT_GENERIC);
        assertThat(order.getGenericOrderState()).isEqualTo(checkConfig.orderState);
        assertThat(order.getLabel()).isEqualTo("label42");
        assertThat(order.getServiceCount()).isEqualTo(1);
        assertThat(order.getInvoiceCount()).isEqualTo(checkConfig.invoicesCount);

        TOrderServiceInfo service = order.getService(0);
        assertThat(service.getServiceType()).isEqualTo(EServiceType.PT_SUBURBAN);
        assertThat(service.getServiceInfo().getGenericOrderItemState()).isEqualTo(checkConfig.orderItemState);

        SuburbanReservation reservation = fromTJson(service.getServiceInfo().getPayload(), SuburbanReservation.class);

        if (checkConfig.checkOrder != null) {
            assertThat(order).satisfies(checkConfig.checkOrder);
        }
        if (checkConfig.checkReservation != null) {
            assertThat(reservation).satisfies(checkConfig.checkReservation);
        }
    }

    protected void waitStateAndCheck(EOrderState orderStateToWait, OrderCheckConfig checkConfig) {
        waitStateAndCheck(
                order -> order.getResult().getGenericOrderState() == orderStateToWait,
                String.format("Expected order state: %s", orderStateToWait),
                checkConfig.orderState(orderStateToWait)
        );
    }

    protected void waitStateAndCheck(ETrustInvoiceState invoiceStateToWait, OrderCheckConfig checkConfig) {
        waitStateAndCheck(
                rsp -> rsp.getResult().getCurrentInvoice().getTrustInvoiceState() == invoiceStateToWait,
                String.format("Expected invoice state: %s", invoiceStateToWait),
                checkConfig
        );
    }

    protected OrderCheckConfig checkOrderBooking(SuburbanReservation reservation,
                                                 String providerOrderId, BigDecimal price) {
        String orderId = travelApiCreateOrder(buildCreateOrderRequest(reservation));
        OrderCheckConfig checkConfig = new OrderCheckConfig().orderId(orderId);
        waitStateAndCheck(EOrderState.OS_RESERVED, checkConfig
                .orderItemState(EOrderItemState.IS_RESERVED)
                .checkOrder(order ->
                        assertThat(ProtoUtils.fromTPrice(order.getPriceInfo().getPrice())).isEqualTo(
                                Money.of(price, ProtoCurrencyUnit.RUB)
                        ))
                .checkReservation(reserv -> {
                    assertThat(reserv.getProviderOrderId()).isEqualTo(providerOrderId);
                    assertThat(reserv.getStationFrom().getTitleDefault()).isEqualTo("Некая станция");
                })
        );
        return checkConfig;
    }

    protected void checkStartPayment(OrderCheckConfig checkConfig) {
        travelApiStartPayment(checkConfig.orderId);
        waitStateAndCheck(ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                checkConfig.orderState(EOrderState.OS_WAITING_PAYMENT).invoicesCount(1));
    }

    protected void checkConfirmedReservation(SuburbanReservation reservation) {
        assertThat(reservation.getProviderTicketBody()).isEqualTo("ticket42");
        assertThat(reservation.getProviderTicketNumber()).isEqualTo("123");
    }

    protected void checkSuccessfulConfirmation(OrderCheckConfig checkConfig) {
        trustAuthorizePayment(checkConfig.orderId);

        waitStateAndCheck(EOrderState.OS_CONFIRMED, checkConfig
                .orderItemState(EOrderItemState.IS_CONFIRMED)
                .checkReservation(this::checkConfirmedReservation));

        // check email method called
        ArgumentCaptor<Order> orderCaptor = ArgumentCaptor.forClass(Order.class);
        verify(suburbanNotificationHelper).createWorkflowForSuburbanOrderConfirmedEmail(orderCaptor.capture());
        assertThat(orderCaptor.getAllValues()).hasSize(1);
        assertThat(orderCaptor.getValue().getId().toString()).isEqualTo(checkConfig.orderId);
    }

    protected void checkBookExpired(SuburbanReservation reservation){
        // этот заказ контрольный - он не должен быть expired, и должен оставаться в IS_RESERVED
        String orderToStayReserved = travelApiCreateOrder(buildCreateOrderRequest(reservation));
        OrderCheckConfig expectedToStayReserved = new OrderCheckConfig().orderId(orderToStayReserved);
        waitStateAndCheck(EOrderState.OS_RESERVED, expectedToStayReserved.orderItemState(EOrderItemState.IS_RESERVED));

        // этот заказ будем проверять на expire
        String orderToExpire = travelApiCreateOrder(buildCreateOrderRequest(reservation));
        OrderCheckConfig expectedForExpiration = new OrderCheckConfig().orderId(orderToExpire);
        waitStateAndCheck(EOrderState.OS_RESERVED, expectedForExpiration.orderItemState(EOrderItemState.IS_RESERVED));

        // еще не прошло время expired
        transactionTemplate.execute(ignored -> {
            OrderItem orderItem = orderRepository.getOne(UUID.fromString(orderToExpire)).getOrderItems().get(0);
            orderItem.setExpiresAt(orderItem.getExpiresAt().minus(bookTtl).plusSeconds(120));
            return null;
        });

        waitStateAndCheck(EOrderState.OS_RESERVED, expectedForExpiration.orderItemState(EOrderItemState.IS_RESERVED));
        waitStateAndCheck(EOrderState.OS_RESERVED, expectedToStayReserved.orderItemState(EOrderItemState.IS_RESERVED));

        // уже прошло время expired
        transactionTemplate.execute(ignored -> {
            OrderItem orderItem = orderRepository.getOne(UUID.fromString(orderToExpire)).getOrderItems().get(0);
            orderItem.setExpiresAt(orderItem.getExpiresAt().minus(bookTtl));
            return null;
        });

        waitStateAndCheck(EOrderState.OS_CANCELLED, expectedForExpiration.orderItemState(EOrderItemState.IS_CANCELLED));
        waitStateAndCheck(EOrderState.OS_RESERVED, expectedToStayReserved.orderItemState(EOrderItemState.IS_RESERVED));
    }

    protected void checkBookCancelling(SuburbanReservation reservation, String commonMessage, String exceptionMessage) {
        String orderId = travelApiCreateOrder(buildCreateOrderRequest(reservation));
        waitStateAndCheck(EOrderState.OS_CANCELLED, new OrderCheckConfig().orderId(orderId)
                .orderItemState(EOrderItemState.IS_CANCELLED)
                .checkReservation(reserv -> {
                    System.out.println(reserv.getError().toString());
                    assertThat(reserv.getProviderOrderId()).isNull();
                    assertThat(reserv.getError().toString()).contains(commonMessage);
                    assertThat(reserv.getError().toString()).contains(exceptionMessage);
                })
        );
    }

    protected void checkBookRetryableCancelling(SuburbanReservation reservation, String exceptionMessage) {
        checkBookCancelling(reservation,
                String.format("Suburban book %d retries failed", PROVIDER_REQUEST_MAX_ATTEMPTS), exceptionMessage);
    }

    protected void checkBookNonRetryableCancelling(SuburbanReservation reservation, String exceptionMessage) {
        checkBookCancelling(reservation, "Suburban book failed", exceptionMessage);
    }

    protected void checkConfirmCancelling(OrderCheckConfig checkConfig, String commonMessage, String exceptionMessage) {
        waitStateAndCheck(EOrderState.OS_CANCELLED, checkConfig
                .orderItemState(EOrderItemState.IS_CANCELLED)
                .checkReservation(reserv -> checkConfirmCancellingReservation(reserv, commonMessage, exceptionMessage))
        );
    }

    private void checkConfirmCancellingReservation(SuburbanReservation reserv, String commonMessage, String exceptionMessage) {
        assertThat(reserv.getProviderTicketBody()).isNull();
        assertThat(reserv.getProviderTicketNumber()).isNull();
        assertThat(reserv.getError().toString()).contains(commonMessage);
        assertThat(reserv.getError().toString()).contains(exceptionMessage);
    }

    protected void checkConfirmRetryableCancelling(OrderCheckConfig checkConfig, String exceptionMessage) {
        checkConfirmCancelling(checkConfig,
                String.format("Suburban confirm %d retries failed", PROVIDER_REQUEST_MAX_ATTEMPTS), exceptionMessage);
    }

    protected void checkConfirmNonRetryableCancelling(OrderCheckConfig checkConfig, String exceptionMessage) {
        checkConfirmCancelling(checkConfig, "Suburban confirm failed", exceptionMessage);
    }

    @Data
    @Accessors(fluent = true)
    protected static class OrderCheckConfig {
        Consumer<TOrderInfo> checkOrder;
        Consumer<SuburbanReservation> checkReservation;
        protected String orderId;
        protected EOrderState orderState;
        protected EOrderItemState orderItemState;
        protected int invoicesCount = 0;
    }
}
