package ru.yandex.travel.orders.integration.suburban;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;

import org.javamoney.moneta.Money;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.suburban.exceptions.SuburbanRetryableException;
import ru.yandex.travel.suburban.model.AeroexpressReservation;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.suburban.partners.SuburbanCarrier;
import ru.yandex.travel.suburban.partners.SuburbanProvider;
import ru.yandex.travel.suburban.partners.aeroexpress.AeroexpressClient;
import ru.yandex.travel.suburban.partners.aeroexpress.exceptions.AeroexpressParseResponseException;
import ru.yandex.travel.suburban.partners.aeroexpress.exceptions.AeroexpressUnknownException;
import ru.yandex.travel.suburban.partners.aeroexpress.model.RequestTickets3;
import ru.yandex.travel.suburban.partners.aeroexpress.model.RequestTicketsResponse;
import ru.yandex.travel.suburban.partners.aeroexpress.model.TicketHistory2;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.when;


public class AeroexpressSuburbanOrderFlowTests extends AbstractSuburbanOrderFlowTests  {
    protected static int orderIdCounter = 4000;

    @MockBean
    public AeroexpressClient aeroexpressClient;

    @Test
    public void testHappyPath() {
        OrderCheckConfig checkConfig = createAndCheckAeroexpressOrderBeforeConfirmation();

        doNothing().when(aeroexpressClient).payOrder(anyLong());
        when(aeroexpressClient.getOrderTickets(anyLong())).thenReturn(getOrderTicketsResponse());

        checkSuccessfulConfirmation(checkConfig);
    }

    @Test
    public void testBookExpired() {
        SuburbanReservation reservation = createReservation();
        RequestTicketsResponse requestTicketsResponse = createRequestTicketResponse();
        when(aeroexpressClient.requestTickets(any(RequestTickets3.class))).thenReturn(requestTicketsResponse);

        checkBookExpired(reservation);
    }

    @Test
    public void testRequestTicketsRetryableException() {
        SuburbanReservation reservation = createReservation();
        when(aeroexpressClient.requestTickets(any(RequestTickets3.class)))
                .thenThrow(new SuburbanRetryableException("timeout happened"));
        checkBookRetryableCancelling(reservation,"timeout happened");
    }

    @Test
    public void testRequestTicketsNonRetryableException() {
        SuburbanReservation reservation = createReservation();
        when(aeroexpressClient.requestTickets(any(RequestTickets3.class)))
                .thenThrow(new AeroexpressUnknownException("bad stuff happened"));
        checkBookNonRetryableCancelling(reservation,"bad stuff happened");
    }

    @Test
    public void testPayOrderRetryableException() {
        OrderCheckConfig checkConfig = createAndCheckAeroexpressOrderBeforeConfirmation();

        doThrow(new SuburbanRetryableException("timeout happened"))
                .when(aeroexpressClient).payOrder(anyLong());

        trustAuthorizePayment(checkConfig.orderId);
        checkConfirmRetryableCancelling(checkConfig, "timeout happened");
    }

    @Test
    public void testPayOrderNonRetryableException() {
        OrderCheckConfig checkConfig = createAndCheckAeroexpressOrderBeforeConfirmation();

        doThrow(new AeroexpressParseResponseException("bad stuff happened"))
                .when(aeroexpressClient).payOrder(anyLong());

        trustAuthorizePayment(checkConfig.orderId);
        checkConfirmNonRetryableCancelling(checkConfig, "bad stuff happened");
    }

    @Test
    public void testGetOrderTicketsRetryableException() {
        OrderCheckConfig checkConfig = createAndCheckAeroexpressOrderBeforeConfirmation();

        doNothing().when(aeroexpressClient).payOrder(anyLong());
        when(aeroexpressClient.getOrderTickets(anyLong()))
                .thenThrow(new SuburbanRetryableException("timeout happened"));

        trustAuthorizePayment(checkConfig.orderId);
        checkConfirmRetryableCancelling(checkConfig, "timeout happened");
    }

    @Test
    public void testGetOrderTicketsNonRetryableException() {
        OrderCheckConfig checkConfig = createAndCheckAeroexpressOrderBeforeConfirmation();

        doNothing().when(aeroexpressClient).payOrder(anyLong());
        when(aeroexpressClient.getOrderTickets(anyLong()))
                .thenThrow(new AeroexpressParseResponseException("bad stuff happened"));

        trustAuthorizePayment(checkConfig.orderId);
        checkConfirmNonRetryableCancelling(checkConfig, "bad stuff happened");
    }

    private static SuburbanReservation createReservation() {
        return SuburbanReservation.builder()
                .provider(SuburbanProvider.AEROEXPRESS)
                .carrier(SuburbanCarrier.AEROEXPRESS)
                .stationFrom(SuburbanReservation.Station.builder()
                        .id(55).titleDefault("Некая станция").build())
                .stationTo(SuburbanReservation.Station.builder()
                        .id(66).build())
                .aeroexpressReservation(AeroexpressReservation.builder()
                        .menuId(22)
                        .orderType(33)
                        .date(LocalDate.now())
                        .build())
                .price(Money.of(499, ProtoCurrencyUnit.RUB))
                .build();
    }

    private static RequestTicketsResponse createRequestTicketResponse() {
        orderIdCounter++;
        var response = new RequestTicketsResponse();
        response.setOrderId((long)orderIdCounter);
        response.setSumm(BigDecimal.valueOf(499));
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss'+03:00'");
        LocalDateTime moscowNow = ZonedDateTime.now(ZoneId.of("Europe/Moscow")).toLocalDateTime();
        response.setReservedUntil(formatter.format(moscowNow.plus(bookTtl)));
        return response;
    }

    private OrderCheckConfig createAndCheckAeroexpressOrderBeforeConfirmation(){
        SuburbanReservation reservation = createReservation();
        RequestTicketsResponse requestTicketsResponse = createRequestTicketResponse();
        when(aeroexpressClient.requestTickets(any(RequestTickets3.class))).thenReturn(requestTicketsResponse);

        BigDecimal bookPrice = reservation.getPrice().getNumberStripped();
        String providerOrderId = String.valueOf(requestTicketsResponse.getOrderId());
        OrderCheckConfig checkConfig = checkOrderBooking(reservation, providerOrderId, bookPrice);

        checkStartPayment(checkConfig);

        return checkConfig;
    }

    private static TicketHistory2 getOrderTicketsResponse() {
        var response = new TicketHistory2();
        response.setTicketId((long)111);
        response.setTicketUrl("ae_ticket_url");
        response.setTariff("ae_tariff");
        response.setStDepart("ae_st_depart");
        response.setTripDate("2022-02-02T00:00:00+03:00");
        response.setValidUntil("2222-02-02T00:00:00+03:00");

        return response;
    }

    @Override
    protected void checkConfirmedReservation(SuburbanReservation reservation) {
        assertThat(reservation.getProviderTicketBody()).isEqualTo("ae_ticket_url&type=pdf");
        assertThat(reservation.getProviderTicketNumber()).isEqualTo("111");

        AeroexpressReservation aeroexpressReservation = reservation.getAeroexpressReservation();
        assertThat(aeroexpressReservation.getOrderId()).isEqualTo(orderIdCounter);
        assertThat(aeroexpressReservation.getTicketId()).isEqualTo(Long.valueOf(111));
        assertThat(aeroexpressReservation.getTicketUrl()).isEqualTo("ae_ticket_url");
        assertThat(aeroexpressReservation.getTariff()).isEqualTo("ae_tariff");
        assertThat(aeroexpressReservation.getStDepart()).isEqualTo("ae_st_depart");
        assertThat(aeroexpressReservation.getTripDate()).isEqualTo("2022-02-02T00:00:00+03:00");
        assertThat(aeroexpressReservation.getValidUntil()).isEqualTo("2222-02-02T00:00:00+03:00");
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {

        @Bean
        @Primary
        public TrustClient trustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trustClient) {
            return paymentProfile -> trustClient;
        }
    }
}
