package ru.yandex.travel.orders.integration.suburban;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;

import org.javamoney.moneta.Money;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.integration.train.factories.ImReservationCreateResponseFactory;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.suburban.model.ImReservation;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.suburban.partners.SuburbanCarrier;
import ru.yandex.travel.suburban.partners.SuburbanProvider;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientIOException;
import ru.yandex.travel.train.partners.im.ImClientParseException;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeResponse;
import ru.yandex.travel.train.partners.im.model.ReservationConfirmResponse;
import ru.yandex.travel.train.partners.im.model.ReservationCreateRequest;
import ru.yandex.travel.train.partners.im.model.ReservationCreateResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.Mockito.when;


public class ImSuburbanOrderFlowTests extends AbstractSuburbanOrderFlowTests {
    protected static int orderIdCounter = 3000;

    @MockBean(name = "imSuburbanClient")
    public ImClient imSuburbanClient;

    @Test
    public void testHappyPath() {
        OrderCheckConfig checkConfig = createAndCheckImOrderBeforeConfirmation();

        when(imSuburbanClient.reservationConfirm(anyInt())).thenReturn(new ReservationConfirmResponse());
        when(imSuburbanClient.orderInfo(anyInt())).thenReturn(orderInfoResponse(ImOperationStatus.OK));
        when(imSuburbanClient.orderReservationTicketBarcode(
                any(OrderReservationTicketBarcodeRequest.class), any(Duration.class)
        )).thenReturn(orderReservationTicketBarcodeResponse());

        checkSuccessfulConfirmation(checkConfig);
    }

    @Test
    public void testBookExpired() {
        SuburbanReservation reservation = createReservation();
        ReservationCreateResponse imReserveResponse = createReserveResponse();
        when(imSuburbanClient.reservationCreate(any(ReservationCreateRequest.class), any()))
                .thenReturn(imReserveResponse);

        checkBookExpired(reservation);
    }

    @Test
    public void testBadPriceOnReserve() {
        SuburbanReservation reservation = createReservation();
        ReservationCreateResponse imReserveResponse = createReserveResponse();
        imReserveResponse.setAmount(BigDecimal.valueOf(100500));
        when(imSuburbanClient.reservationCreate(any(ReservationCreateRequest.class), any()))
                .thenReturn(imReserveResponse);

        String orderId = travelApiCreateOrder(buildCreateOrderRequest(reservation));
        waitStateAndCheck(EOrderState.OS_CANCELLED, new OrderCheckConfig().orderId(orderId)
                .orderItemState(EOrderItemState.IS_CANCELLED)
                .checkReservation(reserv -> {
                    assertThat(reserv.getProviderOrderId()).isEqualTo(String.valueOf(imReserveResponse.getOrderId()));
                    assertThat(reserv.getError().toString()).contains("actual book price is greater than known price");
                })
        );
    }

    @Test
    public void testImReserveRetryableException() {
        SuburbanReservation reservation = createReservation();
        when(imSuburbanClient.reservationCreate(any(ReservationCreateRequest.class), any()))
                .thenThrow(new ImClientIOException("timeout happened"));
        checkBookRetryableCancelling(reservation,"timeout happened");
    }

    @Test
    public void testImReserveNonRetryableException() {
        SuburbanReservation reservation = createReservation();
        when(imSuburbanClient.reservationCreate(any(ReservationCreateRequest.class), any()))
                .thenThrow(new ImClientParseException("bad stuff happened"));
        checkBookRetryableCancelling(reservation,"bad stuff happened");
    }

    @Test
    public void testImConfirmRetryableException() {
        OrderCheckConfig checkConfig = createAndCheckImOrderBeforeConfirmation();

        when(imSuburbanClient.reservationConfirm(anyInt()))
                .thenThrow(new ImClientIOException("timeout happened"));
        when(imSuburbanClient.orderInfo(anyInt())).thenReturn(orderInfoResponse(ImOperationStatus.OK));
        when(imSuburbanClient.orderReservationTicketBarcode(
                any(OrderReservationTicketBarcodeRequest.class), any(Duration.class)
        )).thenReturn(orderReservationTicketBarcodeResponse());

        checkSuccessfulConfirmation(checkConfig);
    }

    @Test
    public void testImOrderInfoRetryableException() {
        OrderCheckConfig checkConfig = createAndCheckImOrderBeforeConfirmation();

        when(imSuburbanClient.reservationConfirm(anyInt())).thenReturn(new ReservationConfirmResponse());
        when(imSuburbanClient.orderInfo(anyInt()))
                .thenThrow(new ImClientIOException("timeout happened"));

        trustAuthorizePayment(checkConfig.orderId);

        checkConfirmRetryableCancelling(checkConfig, "timeout happened");
    }

    @Test
    public void testImOrderInfoParseResponseException() {
        OrderCheckConfig checkConfig = createAndCheckImOrderBeforeConfirmation();

        when(imSuburbanClient.reservationConfirm(anyInt())).thenReturn(new ReservationConfirmResponse());
        when(imSuburbanClient.orderInfo(anyInt()))
                .thenThrow(new ImClientParseException("bad stuff happened"));

        trustAuthorizePayment(checkConfig.orderId);
        checkConfirmRetryableCancelling(checkConfig, "bad stuff happened");
    }

    @Test
    public void testImOrderInfoBadOrderItemStatus() {
        OrderCheckConfig checkConfig = createAndCheckImOrderBeforeConfirmation();

        when(imSuburbanClient.reservationConfirm(anyInt())).thenReturn(new ReservationConfirmResponse());
        when(imSuburbanClient.orderInfo(anyInt())).thenReturn(orderInfoResponse(ImOperationStatus.IN_PROCESS));

        trustAuthorizePayment(checkConfig.orderId);
        checkConfirmRetryableCancelling(checkConfig, "IM reservation order item simpleOperationStatus is InProcess");
    }

    @Test
    public void testImTicketBarcodeRetryableException() {
        OrderCheckConfig checkConfig = createAndCheckImOrderBeforeConfirmation();

        when(imSuburbanClient.reservationConfirm(anyInt())).thenReturn(new ReservationConfirmResponse());
        when(imSuburbanClient.orderInfo(anyInt())).thenReturn(orderInfoResponse(ImOperationStatus.OK));
        when(imSuburbanClient.orderReservationTicketBarcode(
                any(OrderReservationTicketBarcodeRequest.class), any(Duration.class)
        )).thenThrow(new ImClientIOException("timeout happened"));

        trustAuthorizePayment(checkConfig.orderId);
        checkGetTicketBarcodeCancelling(checkConfig,
                String.format("Suburban getTicketBarcode %d retries failed", PROVIDER_REQUEST_MAX_ATTEMPTS),
                "timeout happened");
    }

    @Test
    public void testImTicketBarcodeParseResponseException() {
        OrderCheckConfig checkConfig = createAndCheckImOrderBeforeConfirmation();

        when(imSuburbanClient.reservationConfirm(anyInt())).thenReturn(new ReservationConfirmResponse());
        when(imSuburbanClient.orderInfo(anyInt())).thenReturn(orderInfoResponse(ImOperationStatus.OK));
        when(imSuburbanClient.orderReservationTicketBarcode(
                any(OrderReservationTicketBarcodeRequest.class), any(Duration.class)
        )).thenThrow(new ImClientParseException("bad stuff happened"));

        trustAuthorizePayment(checkConfig.orderId);
        checkGetTicketBarcodeCancelling(checkConfig,
                String.format("Suburban getTicketBarcode %d retries failed", PROVIDER_REQUEST_MAX_ATTEMPTS),
                "bad stuff happened");
    }

    private static SuburbanReservation createReservation() {
        return SuburbanReservation.builder()
            .provider(SuburbanProvider.IM)
            .carrier(SuburbanCarrier.SZPPK)
            .stationFrom(SuburbanReservation.Station.builder()
                    .id(55).titleDefault("Некая станция").build())
            .stationTo(SuburbanReservation.Station.builder()
                    .id(66).build())
            .imReservation(ImReservation.builder()
                    .date(LocalDate.now().plusDays(2))
                    .stationFromExpressId(2004004)
                    .stationToExpressId(2005065)
                    .trainNumber("6666")
                    .imProvider("P6")
                    .build())
            .price(Money.of(72, ProtoCurrencyUnit.RUB)).build();
    }

    private static ReservationCreateResponse createReserveResponse() {
        LocalDateTime moscowNow = ZonedDateTime.now(ZoneId.of("Europe/Moscow")).toLocalDateTime();
        var factory = ImReservationCreateResponseFactory.create();
        var reservationResponse = factory.createReservationCreateResponse();

        reservationResponse.setOrderId(orderIdCounter++);
        reservationResponse.setConfirmTill(moscowNow.plus(bookTtl));
        reservationResponse.setAmount(BigDecimal.valueOf(72));

        return reservationResponse;
    }

    private OrderCheckConfig createAndCheckImOrderBeforeConfirmation(){
        SuburbanReservation reservation = createReservation();
        ReservationCreateResponse imReserveResponse = createReserveResponse();
        when(imSuburbanClient.reservationCreate(any(ReservationCreateRequest.class), any()))
                .thenReturn(imReserveResponse);

        BigDecimal bookPrice = reservation.getPrice().getNumberStripped();
        String providerOrderId = String.valueOf(imReserveResponse.getOrderId());
        OrderCheckConfig checkConfig = checkOrderBooking(reservation, providerOrderId, bookPrice);

        checkStartPayment(checkConfig);

        return checkConfig;
    }

    private static OrderInfoResponse orderInfoResponse(ImOperationStatus orderItemSimpleOperationStatus) {
        var blank = new OrderItemBlank();
        blank.setBlankNumber("123");
        blank.setOrderItemBlankId(123123);

        var orderItem = new OrderItemResponse();
        List<OrderItemBlank> blanks = new ArrayList<>();
        orderItem.setOrderItemId(111);
        orderItem.setOrderItemBlanks(blanks);
        orderItem.setSimpleOperationStatus(orderItemSimpleOperationStatus);
        blanks.add(blank);

        var orderItems = new ArrayList<OrderItemResponse>();
        orderItems.add(orderItem);

        var orderInfoResponse = new OrderInfoResponse();
        orderInfoResponse.setOrderItems(orderItems);
        return orderInfoResponse;
    }

    private static OrderReservationTicketBarcodeResponse orderReservationTicketBarcodeResponse() {
        var barCodeResponse = new OrderReservationTicketBarcodeResponse();
        barCodeResponse.setTicketBarcodeText("ticket42");
        return barCodeResponse;
    }

    protected void checkGetTicketBarcodeCancelling(OrderCheckConfig checkConfig, String commonMessage, String exceptionMessage) {
        waitStateAndCheck(EOrderState.OS_CANCELLED, checkConfig
                .orderItemState(EOrderItemState.IS_CANCELLED)
                .checkReservation(reserv -> {
                    assertThat(reserv.getProviderTicketBody()).isNull();
                    assertThat(reserv.getProviderTicketNumber()).isEqualTo("123");
                    assertThat(reserv.getError().toString()).contains(commonMessage);
                    assertThat(reserv.getError().toString()).contains(exceptionMessage);
                })
        );
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {

        @Bean
        @Primary
        public TrustClient trustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trustClient) {
            return paymentProfile -> trustClient;
        }
    }
}
