package ru.yandex.travel.orders.integration.suburban;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;

import org.javamoney.moneta.Money;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.suburban.exceptions.SuburbanException;
import ru.yandex.travel.suburban.exceptions.SuburbanRetryableException;
import ru.yandex.travel.suburban.model.MovistaReservation;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.suburban.partners.SuburbanCarrier;
import ru.yandex.travel.suburban.partners.SuburbanProvider;
import ru.yandex.travel.suburban.partners.movista.MovistaClient;
import ru.yandex.travel.suburban.partners.movista.MovistaErrorCode;
import ru.yandex.travel.suburban.partners.movista.MovistaModel;
import ru.yandex.travel.suburban.partners.movista.MovistaOrderStatus;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaRequestException;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaUnknownException;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.when;


public class MovistaSuburbanOrderFlowTests extends AbstractSuburbanOrderFlowTests {
    private static int orderIdCounter = 2000;
    @MockBean
    public MovistaClient movistaClient;

    @Test
    public void testHappyPath() {
        SuburbanReservation reservation = createReservation();
        MovistaModel.OrderResponse movistaBook = createBookResponse(reservation.getMovistaReservation().getDate());

        // actual book price could be less than known price
        // https://st.yandex-team.ru/RASPFRONT-9234
        BigDecimal bookPrice = reservation.getPrice().getNumberStripped().subtract(BigDecimal.valueOf(2));
        movistaBook.setPrice(bookPrice);
        when(movistaClient.book(any(MovistaModel.BookRequest.class))).thenReturn(movistaBook);

        OrderCheckConfig checkConfig = checkOrderBooking(reservation, movistaBook.getOrderId().toString(), bookPrice);

        MovistaModel.OrderResponse movistaConfirm = movistaBook.toBuilder()
                .ticketBody("ticket42").ticketNumber(123).build();
        when(movistaClient.confirm(any(MovistaModel.ConfirmRequest.class))).thenReturn(movistaConfirm);

        checkStartPayment(checkConfig);

        checkSuccessfulConfirmation(checkConfig);
    }

    @Test
    public void testBookExpired() {
        SuburbanReservation reservation = createReservation();
        MovistaModel.OrderResponse movistaBook = createBookResponse(reservation.getMovistaReservation().getDate());
        when(movistaClient.book(any(MovistaModel.BookRequest.class))).thenReturn(movistaBook);

        checkBookExpired(reservation);
    }

    @Test
    public void testBadPriceOnReserve() {
        SuburbanReservation reservation = createReservation();
        MovistaModel.OrderResponse movistaBook = createBookResponse(reservation.getMovistaReservation().getDate());
        movistaBook.setPrice(BigDecimal.valueOf(100500));  // price > reservation.price => bad price
        when(movistaClient.book(any(MovistaModel.BookRequest.class))).thenReturn(movistaBook);

        String orderId = travelApiCreateOrder(buildCreateOrderRequest(reservation));
        waitStateAndCheck(EOrderState.OS_CANCELLED, new OrderCheckConfig().orderId(orderId)
                .orderItemState(EOrderItemState.IS_CANCELLED)
                .checkReservation(reserv -> {
                    assertThat(reserv.getProviderOrderId()).isEqualTo(movistaBook.getOrderId().toString());
                    assertThat(reserv.getError().toString()).contains("actual book price is greater than known price");
                })
        );
    }

    @Test
    public void testMovistaBookTimeout() {
        SuburbanReservation reservation = createReservation();
        when(movistaClient.book(any(MovistaModel.BookRequest.class))).thenThrow(
                new SuburbanRetryableException("timeout happened"));
        checkBookRetryableCancelling(reservation, "timeout happened");
    }

    @Test
    public void testMovistaBookUnknownException() {
        SuburbanReservation reservation = createReservation();
        when(movistaClient.book(any(MovistaModel.BookRequest.class))).thenThrow(
                new MovistaUnknownException("bad stuff happened"));

        checkBookNonRetryableCancelling(reservation,"bad stuff happened");
    }

    @Test
    public void testMovistaConfirmTimeout() {
        SuburbanReservation reservation = createReservation();
        OrderCheckConfig checkConfig = setupConfirmFail(new SuburbanRetryableException("timeout happened"), reservation);
        checkConfirmRetryableCancelling(checkConfig, "timeout happened");
    }

    @Test
    public void testMovistaConfirmUnknownException() {
        SuburbanReservation reservation = createReservation();
        OrderCheckConfig checkConfig = setupConfirmFail(
                new MovistaUnknownException("bad stuff happened"), reservation
        );
        checkConfirmNonRetryableCancelling(checkConfig, "bad stuff happened");
    }

    @Test
    public void testMovistaAlreadyConfirmed() {
        SuburbanReservation reservation = createReservation();
        OrderCheckConfig checkConfig = setupConfirmFail(
                new MovistaRequestException(MovistaErrorCode.ORDER_STATUS_INVALID, "already confirmed!!"),
                MovistaOrderStatus.CONFIRMED, reservation);

        waitStateAndCheck(EOrderState.OS_CONFIRMED, checkConfig.orderItemState(EOrderItemState.IS_CONFIRMED));
    }

    @Test
    public void testMovistaNotAlreadyConfirmed() {
        SuburbanReservation reservation = createReservation();
        OrderCheckConfig checkConfig = setupConfirmFail(
                new MovistaRequestException(MovistaErrorCode.DATE_INVALID, "wrong!"),
                MovistaOrderStatus.CONFIRMED, reservation);

        waitStateAndCheck(EOrderState.OS_CANCELLED, checkConfig.orderItemState(EOrderItemState.IS_CANCELLED));
    }

    private static SuburbanReservation createReservation() {
        return SuburbanReservation.builder()
                .provider(SuburbanProvider.MOVISTA)
                .carrier(SuburbanCarrier.CPPK)
                .stationFrom(SuburbanReservation.Station.builder()
                        .id(55).titleDefault("Некая станция").build())
                .stationTo(SuburbanReservation.Station.builder()
                        .id(66).build())
                .movistaReservation(MovistaReservation.builder()
                        .date(LocalDate.now().plusDays(2))
                        .stationFromExpressId(2000055)
                        .stationToExpressId(2000006)
                        .fareId(1200675)
                        .build())
                .price(Money.of(72, ProtoCurrencyUnit.RUB))
                .build();
    }

    private static MovistaModel.OrderResponse createBookResponse(LocalDate travelDate) {
        LocalDateTime moscowNow = ZonedDateTime.now(ZoneId.of("Europe/Moscow")).toLocalDateTime();
        return MovistaModel.OrderResponse.builder()
                .orderId(orderIdCounter++)
                .status(MovistaOrderStatus.CREATED)
                .type(1)
                .travelDate(travelDate)
                .validDate(moscowNow.plus(bookTtl))
                .fareId(1200675)
                .farePlan("Пассажирский")
                .ticketType("Разовый полный")
                .instruction("MID2Tutorial")
                .price(BigDecimal.valueOf(72))
                .fromExpressId(2000055)
                .fromStationName("ОДИНЦОВО")
                .toExpressId(2000006)
                .toStationName("МОСКВА (Белорусский вокзал)")
                .provider("АО «ЦЕНТРАЛЬНАЯ ППК»")
                .inn("7705705370")
                .phone("88007750000").build();
    }

    private OrderCheckConfig setupConfirmFail(SuburbanException error, SuburbanReservation reservation) {
        return setupConfirmFail(error, null, reservation);
    }

    private OrderCheckConfig setupConfirmFail(SuburbanException error, MovistaOrderStatus orderStatus,
                                              SuburbanReservation reservation) {
        MovistaModel.OrderResponse movistaBook = createBookResponse(reservation.getMovistaReservation().getDate());
        movistaBook.setPrice(reservation.getPrice().getNumberStripped());
        when(movistaClient.book(any(MovistaModel.BookRequest.class))).thenReturn(movistaBook);

        String orderId = travelApiCreateOrder(buildCreateOrderRequest(reservation));
        OrderCheckConfig checkConfig = new OrderCheckConfig().orderId(orderId);
        waitStateAndCheck(EOrderState.OS_RESERVED, checkConfig
                .orderItemState(EOrderItemState.IS_RESERVED)
                .checkReservation(reserv ->
                        assertThat(reserv.getProviderOrderId()).isEqualTo(movistaBook.getOrderId().toString())));

        travelApiStartPayment(orderId);
        waitStateAndCheck(ETrustInvoiceState.IS_WAIT_FOR_PAYMENT, checkConfig
                .orderState(EOrderState.OS_WAITING_PAYMENT).invoicesCount(1));

        when(movistaClient.confirm(any(MovistaModel.ConfirmRequest.class))).thenThrow(error);

        if (orderStatus != null) {
            var orderInfoResp = movistaBook.toBuilder().status(orderStatus);
            if (orderStatus == MovistaOrderStatus.CONFIRMED) {
                orderInfoResp.ticketNumber(123).ticketBody("456");
            }
            when(movistaClient.orderInfo(any(Integer.class))).thenReturn(orderInfoResp.build());
        }

        trustAuthorizePayment(orderId);

        return checkConfig;
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {

        @Bean
        @Primary
        public TrustClient trustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trustClient) {
            return paymentProfile -> trustClient;
        }
    }
}
