package ru.yandex.travel.orders.integration.takeout;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.grpc.TakeoutOrdersService;
import ru.yandex.travel.orders.integration.TestGrpcContext;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.takeout.GenericOrderTakeoutService;
import ru.yandex.travel.orders.takeout.proto.ETakeoutJobState;
import ru.yandex.travel.orders.takeout.proto.ETakoutJobType;
import ru.yandex.travel.orders.takeout.proto.TCheckOrdersExistenceReq;
import ru.yandex.travel.orders.takeout.proto.TFindJobReq;
import ru.yandex.travel.orders.takeout.proto.TFindJobRsp;
import ru.yandex.travel.orders.takeout.proto.TStartJobReq;
import ru.yandex.travel.orders.takeout.proto.TStartJobRsp;
import ru.yandex.travel.orders.takeout.proto.TakeoutOrdersInterfaceV1Grpc;
import ru.yandex.travel.takeout.models.TakeoutOrder;
import ru.yandex.travel.takeout.models.TakeoutResponse;
import ru.yandex.travel.takeout.models.TrainTakeoutOrderItem;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "quartz.enabled=true",
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "single-node.auto-start=true",
        }
)
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@DirtiesContext
@ActiveProfiles("test")
public class TakeoutFlowTests {
    private static final String PASSPORT_ID = "passport-id-takeouttestuser";

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();

    @Autowired
    private TakeoutOrdersService takeoutOrdersService;

    private final UserCredentialsBuilder userCredentialsBuilder = new UserCredentialsBuilder();

    @MockBean
    private GenericOrderTakeoutService genericOrderTakeoutService;

    @MockBean
    private OrderRepository ordersRepository;

    private TakeoutOrdersInterfaceV1Grpc.TakeoutOrdersInterfaceV1BlockingStub takeoutClient;

    private Context context;

    @Before
    public void setUpCredentialsContext() {
        UserCredentials credentials = userCredentialsBuilder.build("asdasd", "0099", "passport-id-1", "user1", null,
                "127.0.0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();
        TestGrpcContext context = TestGrpcContext.createTestServer(cleanupRule, takeoutOrdersService);
        takeoutClient = TakeoutOrdersInterfaceV1Grpc.newBlockingStub(context.createChannel());
    }

    @After
    public void tearDownCredentialsContext() {
        Context.current().detach(context);
    }

    @Test
    public void testTakeoutOrders() throws InterruptedException {
        var trainOrder = new TakeoutOrder();
        trainOrder.setTrainItems(new ArrayList<>());
        trainOrder.setHotelItems(new ArrayList<>());
        trainOrder.setAviaItems(new ArrayList<>());
        trainOrder.setSuburbanItems(new ArrayList<>());
        var trainItem = new TrainTakeoutOrderItem();
        trainOrder.getTrainItems().add(trainItem);
        when(genericOrderTakeoutService.getOrdersByUser(any())).thenReturn(List.of(trainOrder));

        TStartJobRsp takeoutJobStartRsp = takeoutClient.startTakeoutJob(TStartJobReq.newBuilder()
                .setPassportId(PASSPORT_ID)
                .setJobType(ETakoutJobType.TT_GET)
                .build());
        TFindJobReq jobResultReq = TFindJobReq.newBuilder()
                .setJobUid(takeoutJobStartRsp.getJobUid())
                .setJobType(ETakoutJobType.TT_GET)
                .build();
        TFindJobRsp jobResult = waitForPredicateOrTimeout(takeoutClient, jobResultReq,
                trsp -> trsp.getState() == ETakeoutJobState.TS_DONE,
                Duration.ofSeconds(3), "Takeout job must be DONE");

        TakeoutResponse jobResultPayload = ProtoUtils.fromTJson(jobResult.getPayload(), TakeoutResponse.class);
        assertThat(jobResultPayload.getGenericOrders().size()).isEqualTo(1);
        assertThat(jobResultPayload.getGenericOrders().get(0)).isEqualTo(trainOrder);
    }

    @Test
    public void testCheckOrderExistenceIsTrueWhenOrdersExist() {
        var order = new GenericOrder();
        order.setRemoved(false);
        when(ordersRepository.findAllOrdersOwnedByUser(PASSPORT_ID)).thenReturn(List.of(order));

        var result = takeoutClient.checkOrdersExistence(TCheckOrdersExistenceReq.newBuilder()
                .setPassportId(PASSPORT_ID).build());

        assertThat(result.getOrdersExist()).isTrue();
    }

    @Test
    public void testCheckOrderExistenceIsFalseWhenNoOrdersExist() {
        when(ordersRepository.findAllOrdersOwnedByUser(PASSPORT_ID)).thenReturn(List.of());

        var result = takeoutClient.checkOrdersExistence(TCheckOrdersExistenceReq.newBuilder()
                .setPassportId(PASSPORT_ID).build());

        assertThat(result.getOrdersExist()).isFalse();
    }

    @Test
    public void testCheckOrderExistenceIsFalseWhenAllOrdersAreRemoved() {
        var order = new GenericOrder();
        order.setRemoved(true);
        when(ordersRepository.findAllOrdersOwnedByUser(PASSPORT_ID)).thenReturn(List.of(order));

        var result = takeoutClient.checkOrdersExistence(TCheckOrdersExistenceReq.newBuilder()
                .setPassportId(PASSPORT_ID).build());

        assertThat(result.getOrdersExist()).isFalse();
    }
}
