package ru.yandex.travel.orders.integration.topup;

import java.util.List;

import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustBoundPaymentMethod;
import ru.yandex.travel.orders.services.payments.model.TrustBoundPaymentMethodType;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateTopupResponse;
import ru.yandex.travel.testing.mockito.ThreadSafeMock;
import ru.yandex.travel.testing.mockito.ThreadSafeMockBuilder;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.when;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "single-node.auto-start=true",
                "yandex-plus-promo.await-payment-check-period=100ms",
                "yandex-plus-promo.await-payment-processor.schedule-rate=100ms",
                "yandex-plus-promo.await-payment-processor.initial-start-delay=10ms",
                "yandex-plus-promo.min-topup-at=2021-07-01T00:00:00Z",
                "yandex-plus-promo.tours-enabled=true",
                "yandex-plus-grpc-service.enabled=true",
        }
)
@DirtiesContext
@ActiveProfiles("test")
public abstract class AbstractYandexPlusTopupFlowTest {
    @Autowired
    protected TransactionTemplate transactionTemplate;

    @MockBean
    protected TrustClientProvider trustClientProvider;
    protected final ThreadSafeMock<TrustClient> trustClientSafe =
            ThreadSafeMockBuilder.newThreadSafeMockBuilder(TrustClient.class).build();

    protected void initTrustMocks(String passportId, String accountId, String purchaseToken) {
        trustClientSafe.initNewMocks(client -> {
            // should move the logic to MockDBTrustClient eventually,
            // but the class should treat baskets and topups equally
            when(client.getPaymentMethods(argThat(ui -> ui.getUid().equals(passportId))))
                    .thenReturn(TrustPaymentMethodsResponse.builder()
                            .boundPaymentMethods(List.of(TrustBoundPaymentMethod.builder()
                                    .id(accountId)
                                    .paymentMethod(TrustBoundPaymentMethodType.YANDEX_ACCOUNT)
                                    .currency("RUB")
                                    .build()))
                            .build());
            when(client.createTopup(argThat(a -> a.getPaymethodId().equals(accountId)), any()))
                    .thenReturn(TrustCreateTopupResponse.builder().purchaseToken(purchaseToken).build());
            when(client.getBasketStatus(eq(purchaseToken), any())).thenReturn(TrustBasketStatusResponse.builder()
                    .paymentStatus(PaymentStatusEnum.CLEARED)
                    .build());
        });
    }
}
