package ru.yandex.travel.orders.integration.topup;

import java.util.UUID;
import java.util.stream.Collectors;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.testing.GrpcCleanupRule;
import org.javamoney.moneta.Money;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.jupiter.api.Assertions;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.grpc.YandexPlusGrpcService;
import ru.yandex.travel.orders.integration.TestGrpcContext;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.repository.YandexPlusTopupRepository;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.workflow.plus.proto.EYandexPlusTopupState;
import ru.yandex.travel.orders.yandex_plus.EPaymentProfile;
import ru.yandex.travel.orders.yandex_plus.TGetPlusTopupStatusReq;
import ru.yandex.travel.orders.yandex_plus.TGetPlusTopupStatusRsp;
import ru.yandex.travel.orders.yandex_plus.TSchedulePlusTopupReq;
import ru.yandex.travel.orders.yandex_plus.TSchedulePlusTopupRsp;
import ru.yandex.travel.orders.yandex_plus.YandexPlusServiceGrpc;
import ru.yandex.travel.testing.TestUtils;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;


public class ExternalYandexPlusTopupFlowTest extends AbstractYandexPlusTopupFlowTest {
    @Autowired
    private YandexPlusTopupRepository topupRepository;

    @Autowired
    private FinancialEventRepository repository;

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();

    @Autowired
    protected YandexPlusGrpcService yandexPlusGrpcService;

    protected YandexPlusServiceGrpc.YandexPlusServiceBlockingStub yandexPlusGrpcClient;

    @Before
    public void init() {
        // should actually initialize this answer only once, before the whole test set
        when(trustClientProvider.getTrustClientForPaymentProfile(any()))
                .thenReturn((TrustClient) trustClientSafe);
        trustClientSafe.resetToDefaultMocks();

        TestGrpcContext grpcContext = TestGrpcContext.createTestServer(cleanupRule, yandexPlusGrpcService);

        yandexPlusGrpcClient = YandexPlusServiceGrpc.newBlockingStub(grpcContext.createChannel());
    }

    @Test
    public void testExternalTopupWithFinEvents() {
        doTestExternalTopup(false);
    }

    @Test
    public void testExternalTopupWithoutFinEvents() {
        doTestExternalTopup(true);
    }

    public void doTestExternalTopup(boolean skipFinancialEvents) {
        String passportId = "passport-" + UUID.randomUUID().toString();
        String accountId = "ya-acc-" + UUID.randomUUID().toString();
        String purchaseToken = "some-token-" + UUID.randomUUID().toString();
        initTrustMocks(passportId, accountId, purchaseToken);
        String externalOrderId = "external-order-" + UUID.randomUUID().toString();
        int topupAmount = 100;

        var operationId = UUID.randomUUID().toString();

        var topupResult = schedulePlusTopup(operationId, topupAmount, passportId, externalOrderId, skipFinancialEvents);

        assertThat(topupResult.getScheduled()).isTrue();

        var topupInfo = waitTopupToFinish(operationId);

        assertThat(topupInfo).isNotNull();
        assertThat(topupInfo.getState()).isEqualTo(skipFinancialEvents ? EYandexPlusTopupState.PS_CLEARED : EYandexPlusTopupState.PS_FINANCIAL_EVENT_SENT);
        assertThat(topupInfo.getAmount()).isEqualTo(topupAmount);
        assertThat(topupInfo.getCurrency().getCurrencyCode()).isEqualTo("RUB");
        assertThat(topupInfo.getPassportId()).isEqualTo(passportId);
        assertThat(topupInfo.getExternalOrderId()).isEqualTo(externalOrderId);

        var financialEvents = repository.findAll().stream()
                .filter(x -> x.getOrderPrettyId().equals(externalOrderId))
                .collect(Collectors.toUnmodifiableList());

        if (skipFinancialEvents) {
            assertThat(financialEvents.size()).isEqualTo(0);
        } else {
            assertThat(financialEvents.size()).isEqualTo(1);

            assertThat(financialEvents.get(0).getType()).isEqualTo(FinancialEventType.YANDEX_ACCOUNT_TOPUP_PAYMENT);
            assertThat(financialEvents.get(0).getPlusTopupAmount()).isEqualTo(Money.of(topupAmount, "RUB"));
            assertThat(financialEvents.get(0).getTotalAmount()).isEqualTo(Money.of(topupAmount, "RUB"));
        }

        verify(trustClientSafe.getCurrentMocksHolder()).createTopup(argThat(a -> a.getPaymethodId().equals(accountId)
                && a.getPassParams().getPayload().getBaseAmount().longValue() == 1000
                && a.getPassParams().getPayload().getCommissionAmount().longValue() == 200
                && a.getPassParams().getPayload().getVatCommissionAmount().longValue() == 40), any());
        verify(trustClientSafe.getCurrentMocksHolder()).startPayment(eq(purchaseToken), any());
    }

    @Test
    public void testExternalTopupDuplicateWithFinEvents() {
        doTestExternalTopupDuplicate(false);
    }

    @Test
    public void testExternalTopupDuplicateWithoutFinEvents() {
        doTestExternalTopupDuplicate(true);
    }

    public void doTestExternalTopupDuplicate(boolean skipFinancialEvents) {
        String passportId = "passport-" + UUID.randomUUID().toString();
        String accountId = "ya-acc-" + UUID.randomUUID().toString();
        String purchaseToken = "some-token-" + UUID.randomUUID().toString();
        initTrustMocks(passportId, accountId, purchaseToken);
        String externalOrderId = "external-order-" + UUID.randomUUID().toString();
        int topupAmount = 100;

        var operationId = UUID.randomUUID().toString();

        var firstTopupResult = schedulePlusTopup(operationId, topupAmount, passportId, externalOrderId, skipFinancialEvents);
        assertThat(firstTopupResult.getScheduled()).isTrue();

        var exception = Assertions.assertThrows(StatusRuntimeException.class,
                () -> schedulePlusTopup(operationId, topupAmount, passportId, externalOrderId, skipFinancialEvents),
                String.format("Topup operation with ID (%s) already exists", operationId));
        assertThat(exception.getStatus().getCode()).isEqualTo(Status.Code.ALREADY_EXISTS);

        waitTopupToFinish(operationId);

        var financialEvents = repository.findAll().stream()
                .filter(x -> x.getOrderPrettyId().equals(externalOrderId))
                .collect(Collectors.toUnmodifiableList());
        assertThat(financialEvents.size()).isEqualTo(skipFinancialEvents ? 0 : 1);

        verify(trustClientSafe.getCurrentMocksHolder()).createTopup(argThat(a -> a.getPaymethodId().equals(accountId)
                && a.getPassParams().getPayload().getBaseAmount().longValue() == 1000
                && a.getPassParams().getPayload().getCommissionAmount().longValue() == 200
                && a.getPassParams().getPayload().getVatCommissionAmount().longValue() == 40), any());
        verify(trustClientSafe.getCurrentMocksHolder()).startPayment(eq(purchaseToken), any());
    }

    private TSchedulePlusTopupRsp schedulePlusTopup(String operationId, int topupAmount, String passportId, String externalOrderId, boolean skipFinancialEvents) {
        return yandexPlusGrpcClient.schedulePlusTopup(TSchedulePlusTopupReq.newBuilder()
                .setOperationId(operationId)
                .setPoints(topupAmount)
                .setCurrency(ECurrency.C_RUB)
                .setPassportId(passportId)
                .setExternalOrderId(externalOrderId)
                .setTotalAmountForPayload(ProtoUtils.toTPrice(Money.of(1000, "RUB")))
                .setCommissionAmountForPayload(ProtoUtils.toTPrice(Money.of(200, "RUB")))
                .setVatCommissionAmountForPayload(ProtoUtils.toTPrice(Money.of(40, "RUB")))
                .setPaymentProfile(EPaymentProfile.PP_HOTEL)
                .setUserIp("user-ip")
                .setSkipFinancialEvents(skipFinancialEvents)
                .build());
    }

    private YandexPlusTopup waitTopupToFinish(String operationId) {
        TestUtils.waitForState("Topup scheduled", () -> {
            var topupStatus = yandexPlusGrpcClient.getPlusTopupStatus(TGetPlusTopupStatusReq.newBuilder()
                    .setOperationId(operationId)
                    .build());

            return topupStatus.getSchedulingStatus() == TGetPlusTopupStatusRsp.EScheduledTopupStatus.STC_SUCCEED &&
                    topupStatus.hasTopupInfo() && topupStatus.getTopupInfo().getSucceed();
        });

        return transactionTemplate.execute(status -> {
            var topupStatus = yandexPlusGrpcClient.getPlusTopupStatus(TGetPlusTopupStatusReq.newBuilder()
                    .setOperationId(operationId)
                    .build());
            return topupRepository.getOne(UUID.fromString(topupStatus.getTopupInfo().getTopupEntityId()));
        });
    }
}
