package ru.yandex.travel.orders.integration.topup;

import java.time.LocalDate;
import java.util.List;
import java.util.UUID;

import javax.money.CurrencyUnit;
import javax.persistence.EntityManager;

import org.javamoney.moneta.Money;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.mock.mockito.MockBean;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.orders.ConfirmationInfo;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.orders.promo.YandexPlusApplication;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EPaymentOutcome;
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.repository.YandexPlusTopupRepository;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.finances.OverallServiceBalance;
import ru.yandex.travel.orders.services.payments.InvoicePaymentFlags;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.plus.YandexPlusPromoService;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.plus.proto.EYandexPlusTopupState;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.workflow.entities.Workflow;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

public class YandexPlusTopupFlowTest extends AbstractYandexPlusTopupFlowTest {
    @Autowired
    private EntityManager em;

    @Autowired
    private YandexPlusTopupRepository topupRepository;

    @Autowired
    private YandexPlusPromoService yandexPlusPromoService;

    @MockBean
    private FinancialEventService financialEventService;

    @Before
    public void init() {
        // should actually initialize this answer only once, before the whole test set
        when(trustClientProvider.getTrustClientForPaymentProfile(any()))
                .thenReturn((TrustClient) trustClientSafe);
        trustClientSafe.resetToDefaultMocks();
    }

    @SuppressWarnings("ConstantConditions")
    @Test
    public void testHappyPath() {
        when(financialEventService.getOverallServiceBalance(any())).thenReturn(testBalance());
        when(financialEventService.getYandexFeeVat(any())).thenReturn(VatType.VAT_20);
        String passportId = "passport-o9794197214142";
        String accountId = "ya-acc-w/01924y017417441-42o";
        String purchaseToken = "some_token_o18279461941";
        initTrustMocks(passportId, accountId, purchaseToken);

        UUID orderItemId = transactionTemplate.execute(status -> {
            HotelOrder order = createOrder();
            ExpediaOrderItem item = createOrderItem(order);
            var user = AuthorizedUser.createLogged(order.getId(), "yandexuid", passportId, "login", null);
            var invoice = TrustInvoice.createInvoice(order, user, InvoicePaymentFlags.builder().build());
            em.persist(invoice);

            // starting the topup flow
            yandexPlusPromoService.registerConfirmedService(item);
            return item.getId();
        });

        TestUtils.waitForState("Topup cleared and fin event sent", () -> transactionTemplate.execute(status -> {
            YandexPlusTopup topup = topupRepository.findByOrderItemId(orderItemId);
            return topup != null && topup.getState() == EYandexPlusTopupState.PS_FINANCIAL_EVENT_SENT;
        }));
        UUID topupId = transactionTemplate.execute(status -> {
            YandexPlusTopup topup = topupRepository.findByOrderItemId(orderItemId);
            assertThat(topup.getAmount()).isEqualTo(1234);
            assertThat(topup.getCurrency().getCurrencyCode()).isEqualTo("RUB");
            return topup.getId();
        });

        verify(trustClientSafe.getCurrentMocksHolder()).startPayment(eq(purchaseToken), any());
        verify(financialEventService).registerPlusPointsTopup(argThat(t -> t.getId().equals(topupId)));
    }

    private HotelOrder createOrder() {
        var order = new HotelOrder();
        order.setState(EHotelOrderState.OS_CONFIRMED);
        order.setDisplayType(EDisplayOrderType.DT_HOTEL);
        order.setId(UUID.randomUUID());
        order.setPrettyId("YA-COOL-ID-1");
        order.setPaymentTestContext(TPaymentTestContext.newBuilder()
                .setPaymentOutcome(EPaymentOutcome.PO_SUCCESS)
                .build());
        order.setCurrency(ProtoCurrencyUnit.RUB);
        Workflow orderWorkflow = Workflow.createWorkflowForEntity(order);
        orderWorkflow.setSupervisorId(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid());
        em.persist(orderWorkflow);
        em.persist(order);
        return order;
    }

    private ExpediaOrderItem createOrderItem(HotelOrder order) {
        var item = new ExpediaOrderItem();
        Guest guest = new Guest();
        guest.setFirstName("Test");
        guest.setLastName("Testov");
        ExpediaHotelItinerary itinerary = new ExpediaHotelItinerary();
        itinerary.setGuests(List.of(guest));
        itinerary.setCustomerEmail("user1@unit.test");
        itinerary.setCustomerPhone("71234567890");
        itinerary.setOrderDetails(OrderDetails.builder()
                .hotelName("hotel")
                // scheduling it into the past to trigger the single operation task processor without any delays
                .checkinDate(LocalDate.parse("2021-07-03"))
                .checkoutDate(LocalDate.parse("2021-07-10"))
                .hotelPhone("12345678")
                .ratePlanDetails("специальная инструкция")
                .build());
        ConfirmationInfo confirmation = new ConfirmationInfo();
        confirmation.setHotelConfirmationId("confirmationId");
        itinerary.setConfirmation(confirmation);
        itinerary.setFiscalPrice(Money.of(1000, "RUB"));
        itinerary.setAppliedPromoCampaigns(AppliedPromoCampaigns.builder()
                .yandexPlus(YandexPlusApplication.builder()
                        .mode(YandexPlusApplication.Mode.TOPUP)
                        .points(1234)
                        .build())
                .build());
        item.setItinerary(itinerary);
        item.setState(EExpediaItemState.IS_CONFIRMED);
        item.setId(UUID.randomUUID());
        var fiscalItem = new FiscalItem();
        fiscalItem.setMoneyAmount(Money.of(1000, ProtoCurrencyUnit.RUB));
        fiscalItem.setType(FiscalItemType.EXPEDIA_HOTEL);
        fiscalItem.setVatType(VatType.VAT_20);
        item.addFiscalItem(fiscalItem);
        em.persist(item);
        Workflow workflow = Workflow.createWorkflowForEntity(item, order);
        em.persist(workflow);
        order.addOrderItem(item);
        return item;
    }

    private OverallServiceBalance testBalance() {
        CurrencyUnit currency = ProtoCurrencyUnit.RUB;
        return OverallServiceBalance.builder()
                .userPartner(Money.of(900, currency))
                .userFee(Money.of(100, currency))
                .plusPartner(Money.zero(currency))
                .plusFee(Money.zero(currency))
                .promoPartner(Money.zero(currency))
                .promoFee(Money.zero(currency))
                .techFee(Money.zero(currency))
                .build();
    }
}
