package ru.yandex.travel.orders.integration.train;

import java.util.Arrays;
import java.util.UUID;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.ETrainChangeElectronicRegistrationOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsurancePricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundPricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationCreateOutcome;
import ru.yandex.travel.orders.commons.proto.TTrainTestContext;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.integration.train.factories.TrainOrderItemFactory;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.services.MailSenderService;
import ru.yandex.travel.orders.services.UrlShortenerService;
import ru.yandex.travel.orders.services.YaSmsService;
import ru.yandex.travel.orders.services.attachments.AttachmentProviderService;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;

import static org.assertj.core.api.Assertions.assertThat;


@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "quartz.enabled=true",
                "workflow-processing.pending-workflow-polling-interval=100ms",

                "trust-hotels.clearing-refresh-timeout=1s",
                "trust-hotels.payment-refresh-timeout=1s",

                "single-node.auto-start=true",

                "train-workflow.check-ticket-refund-delay=1s",
                "train-workflow.check-ticket-refund-task-period=200ms",
                "train-workflow.check-ticket-refund-max-tries=50",

                "train-workflow.check-passenger-discounts-enabled=true",
                "train-workflow.check-expiration-task-period=100ms",

                "generic-workflow.train-rebooking-enabled=true",
        }
)
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@DirtiesContext
@ActiveProfiles("test")
public abstract class AbstractTrainOrderFlowTest {
    protected static final String SESSION_KEY = "qwerty";
    protected static final String YANDEX_UID = "1234567890";

    protected String passportId;
    protected Long passportIdNum;

    protected final UserCredentialsBuilder userCredentialsBuilder = new UserCredentialsBuilder();


    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();

    @Autowired
    protected OrdersService ordersService;

    protected Context context;

    protected OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client;

    @MockBean
    protected YaSmsService yaSmsService;

    @MockBean
    protected UrlShortenerService urlShortenerService;

    @Autowired
    protected UserOrderCounterService counterService;

    @MockBean
    private MailSenderService mailSenderService;

    @MockBean
    private AttachmentProviderService attachmentProviderService;

    @MockBean
    protected StarTrekService starTrekService;


    @Before
    public void setUp() throws Exception {
        // passportId is Long
        passportIdNum = System.currentTimeMillis() % 100_000;
        passportId = String.valueOf(passportIdNum);

        UserCredentials credentials = userCredentialsBuilder.build(SESSION_KEY, YANDEX_UID, passportId, "user1",
                null, "127.0.0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();
        client = IntegrationUtils.createServerAndBlockingStub(cleanupRule, ordersService);
    }

    @After
    public void tearDownCredentialsContext() {
        Context.current().detach(context);
    }

    protected TCreateOrderReq.Builder createOrderRequest(int passengers) {
        var factory = new TrainOrderItemFactory();
        TrainReservation payload = factory.createTrainReservation();
        for (int i = 1; i < passengers; i++) {
            TrainPassenger extraPassenger = factory.createTrainPassenger();
            extraPassenger.setFirstName(extraPassenger.getFirstName() + 'A' + i);
            extraPassenger.setDocumentNumber(extraPassenger.getDocumentNumber() + i);
            payload.getPassengers().add(extraPassenger);
        }
        return createOrderRequest(payload, factory);
    }

    protected TCreateOrderReq.Builder createOrderRequest(TrainReservation payload,
                                                         TrainOrderItemFactory factory) {
        return createOrderRequest(ProtoUtils.toTJson(payload).getValue());
    }

    protected TCreateOrderReq.Builder createOrderRequest(TrainReservation payload) {
        return createOrderRequest(ProtoUtils.toTJson(payload).getValue());
    }

    protected TCreateOrderReq.Builder createOrderRequest(String... payloadJsons) {
        var builder = TCreateOrderReq.newBuilder()
                .setOrderType(orderTypeForOrderReq())
                .setDeduplicationKey(UUID.randomUUID().toString())
                .setCurrency(ECurrency.C_RUB)
                .setOwner(TUserInfo.newBuilder()
                        .setEmail("train@test.com")
                        .setPhone("+79111111111")
                        .setPassportId(passportId)
                        .setLogin("user1")
                        .setYandexUid(YANDEX_UID)
                );
        Arrays.stream(payloadJsons).forEach(
                payloadJson -> builder.addCreateServices(
                        getServiceForPayload(payloadJson)
                ));
        return builder;
    }

    protected EOrderType orderTypeForOrderReq() {
        return EOrderType.OT_GENERIC;
    }

    protected TCreateServiceReq.Builder getServiceForPayload(String payloadJson) {
        return TCreateServiceReq.newBuilder()
                .setServiceType(EServiceType.PT_TRAIN)
                .setTrainTestContext(TTrainTestContext.newBuilder()
                        .setInsurancePricingOutcome(ETrainInsurancePricingOutcome.IPO_SUCCESS)
                        .setInsuranceCheckoutOutcome(ETrainInsuranceCheckoutOutcome.ICO_SUCCESS)
                        .setInsuranceCheckoutConfirmOutcome(ETrainInsuranceCheckoutConfirmOutcome.ICCO_SUCCESS)
                        .setRefundPricingOutcome(ETrainRefundPricingOutcome.RPO_SUCCESS)
                        .setRefundCheckoutOutcome(ETrainRefundCheckoutOutcome.RCO_SUCCESS)
                        .setReservationCreateOutcome(ETrainReservationCreateOutcome.RCRO_SUCCESS)
                        .setReservationConfirmOutcome(ETrainReservationConfirmOutcome.RCOO_SUCCESS)
                        .setTrainChangeElectronicRegistrationOutcome(ETrainChangeElectronicRegistrationOutcome.CERO_SUCCESS)
                        .build())
                .setSourcePayload(TJson.newBuilder().setValue(payloadJson).build());
    }

    protected void userHasConfirmedOrders(int num) {
        assertThat(counterService.getNumberOfOrdersConfirmedByUser(passportIdNum, EDisplayOrderType.DT_TRAIN))
                .isEqualTo(num);
    }

}
